<?php

	namespace APIPublic;

	use Accounts\Account;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Redirect;
	use Nox\Http\Request;
	use Nox\Http\RequestPayload;
	use Nox\Http\TextPayload;
	use PHPUnit\Framework\TestCase;
	use Roles\Role;

	final class APIPublicControllerTests extends TestCase
	{

		public function testGetAdminAccountLoginTokenUsingAPIKey(): void{

			// Create a new super admin role
			$role = new Role();
			$role->name = "API Test Role " . time();
			$role->hasAllPermissions = 1;
			$role->save();

			// Create an account that will be fetched
			$account = new Account();
			$account->username = "test" . time();
			$account->roleID = $role->id;
			$account->save();

			$request = new Request();

			$controller = new APIPublicController();
			$response = $controller->getAdminAccountLoginTokenUsingAPIKey($request);

			$this->assertInstanceOf(
				expected: JSONSuccess::class,
				actual: $response,
			);

			// Clean up
			$account->delete();
			$role->delete();
		}

		/**
		 * The API will create a new account if one does not exist, to allow remote logins to always succeed. This tests the truth of that.
		 */
		public function testGetAdminAccountLoginTokenUsingAPIKeyCreatesNewAccount(): void{

			$request = new Request();

			$controller = new APIPublicController();
			$response = $controller->getAdminAccountLoginTokenUsingAPIKey($request);

			$this->assertInstanceOf(
				expected: JSONSuccess::class,
				actual: $response,
			);

			// Clean up
			// Fetch the session from the token
			// Then get the user ID
			$cookieSession = APIPublicService::getValidAccountSessionFromToken($response->toArray()['token']);
			$accountID = $cookieSession->userID;
			/** @var ?Account $account */
			$account = Account::fetch($accountID);
			$role = Role::fetch($account->roleID);

			$account->delete();
			$role->delete();
			$cookieSession->delete();
		}

		/**
		 * Uses runInSeparateProcess to avoid fatal error about headers already being sent - as the controller uses setcookie
		 * @depends testGetAdminAccountLoginTokenUsingAPIKey
		 * @depends testGetAdminAccountLoginTokenUsingAPIKeyCreatesNewAccount
		 * @runInSeparateProcess
		 */
		public function testLoginWithToken(): void{
			$request = new Request();

			$controller = new APIPublicController();
			$response = $controller->getAdminAccountLoginTokenUsingAPIKey($request);

			$this->assertInstanceOf(
				expected: JSONSuccess::class,
				actual: $response,
			);

			$token = $response->toArray()['token'];
			$request->setQueryValue("token", $token);
			$response = $controller->loginWithToken($request);
			$this->assertInstanceOf(
				expected: Redirect::class,
				actual: $response,
			);

			// Clean up
			// Fetch the session from the token
			// Then get the user ID
			$cookieSession = APIPublicService::getValidAccountSessionFromToken($token);
			$accountID = $cookieSession->userID;

			/** @var ?Account $account */
			$account = Account::fetch($accountID);
			$role = Role::fetch($account->roleID);

			$account->delete();
			$role->delete();
			$cookieSession->delete();
		}

		public function testSetSiteSuspendedState(): void{
			$request = new Request();
			$payload = new RequestPayload();

			$siteSuspendedPayload = new TextPayload();
			$siteSuspendedPayload->name = "suspended";
			$siteSuspendedPayload->contents = "0";

			$payload->pushPayload($siteSuspendedPayload);
			$request->setPayload($payload);

			$controller = new APIPublicController();
			$response = $controller->setSiteSuspendedState($request);

			$this->assertInstanceOf(
				expected: JSONSuccess::class,
				actual: $response,
			);
		}
	}