<?php
	require_once __DIR__ . "/../../vendor/autoload.php";
	require_once __DIR__ . "/../nox-env.php";

	use MonologWrapper\MonologWrapper;
	use Nox\Http\Request;
	use Nox\Nox;
	use Nox\ORM\Abyss;
	use Nox\ORM\DatabaseCredentials;
	use Nox\Router\BaseController;
	use Page\Page;

	// Load the source directory
	$nox = new Nox();
	$nox->setSourceCodeDirectory(__DIR__ . "/../src");

	$request = new Request();
	BaseController::$noxInstance = new Nox();

	// Load the credentials for any and all databases used by the models
	Abyss::addCredentials(new DatabaseCredentials(
		host: NoxEnv::MYSQL_HOST,
		username: NoxEnv::MYSQL_USERNAME,
		password: NoxEnv::MYSQL_PASSWORD,
		database: NoxEnv::MYSQL_DB_NAME,
		port: NoxEnv::MYSQL_PORT,
	));

	$logger = MonologWrapper::getLoggerWithSTDOutStream();
	$logger->info("Build to zip export script started.");

	$root = __DIR__ . "/../..";
	$ignoredFiles = [
		realpath($root . "/.git"),
		realpath($root . "/.gitignore"),
		realpath($root . "/.idea"),
		realpath($root . "/extraction-test"),
		realpath($root . "/.phpunit.result.cache"),
		realpath($root . "/composer.json"),
		realpath($root . "/composer.lock"),
		realpath($root . "/install-entries.json"),
		realpath($root . "/README.md"),
		realpath($root . "/.htaccess"),
		realpath($root . "/uplift-core"),
		realpath($root . "/vendor"),
		realpath($root . "/tests"),
	];

	$outputLocation = __DIR__ . "/../../export.zip";
	$redirects = [];

	// Remove any existing export
	if (file_exists($outputLocation)){
		$logger->info("Removing existing export.zip file.");
		unlink($outputLocation);
	}

	$zip = new ZipArchive();
	$zip->open($outputLocation, ZipArchive::CREATE);

	// Export all page objects
	/** @var Page[] $allPages */
	$allPages = Page::query();
	foreach($allPages as $page){
		$uri = $page->pageRoute;

		// Set the REQUEST_URI, as some layouts use it
		$_SERVER['REQUEST_URI'] = $uri;

		// Set the current page for shortcodes to read from
		Page::setCurrentPage($page);

		// Get the rendered HTML
		$htmlFileContents = $page->getRenderedHTML();

		// Parse directories from the URI
		$directories = explode("/", $uri);

		// Check if this file is an index file - the URI ends with a "/"
		$isDirectoryIndexFile = str_ends_with($uri, "/");

		$resultingFilePath = "";
		$apacheConfigRedirect = "";

		// Get rid of empty values in the directories array
		$directories = array_filter($directories, function ($str){
			return !empty($str);
		});

		// If this is not the index of a directory, then remove the last entry from the $directories
		// as it's just the page's file name
		if (!$isDirectoryIndexFile) {
			$fileName = array_pop($directories);
		}


		if ($isDirectoryIndexFile){
			if ($uri === "/"){
				// Home page
				$resultingFilePath = "index.html";
			}else {
				$resultingFilePath = sprintf("%s/index.html", implode("/", $directories));
			}
		}else{
			if (!empty($directories)) {
				$resultingFilePath = sprintf("%s/%s.html", implode("/", $directories), $fileName);
			}else{
				$resultingFilePath = sprintf("%s.html", $fileName);
			}
		}

		if (!$isDirectoryIndexFile){
			// It needs a direct
			$redirects[] = [$uri, $resultingFilePath];
		}

		// Add the file and its contents
		$zip->addFromString($resultingFilePath, $htmlFileContents);
		$logger->debug("Adding $resultingFilePath to archive.");
	}

	// Add redirections for the pages to their now raw HTML files
	if (!empty($redirects)){
		$logger->info("Adding .htaccess for redirects.");
		$content = "RewriteEngine On\n";
		/** @var array $redirect */
		foreach($redirects as $redirect){
			$fromURI = $redirect[0];
			$toURI = $redirect[1];
			$content .= "RewriteCond %{REQUEST_URI} ^$fromURI$\nRewriteRule .* /$toURI\n\n";
		}

		$zip->addFromString(".htaccess", $content);
		$logger->info("Wrote .htaccess file.");
	}

	// Handle any non-ignored files and directories by adding them to the zip contents as well
	// This includes the uplift-data directory and any misc files
	$logger->info("Adding root contents that are not ignored.");
	$files = array_diff(scandir($root), ['.','..']);
	foreach($files as $fileName){
		$filePath = realpath($root . "/" . $fileName);
		if (!in_array($filePath, $ignoredFiles)){
			if (is_dir($filePath)){
				$directoryPath = $filePath;
				$directoryName = basename($directoryPath);
				$filesIterator = new RecursiveIteratorIterator(
					new RecursiveDirectoryIterator($directoryPath),
					RecursiveIteratorIterator::LEAVES_ONLY
				);

				/** @var DirectoryIterator $dirIterator */
				foreach($filesIterator as $dirIterator){
					if (!$dirIterator->isDir()){
						$fullPath = $dirIterator->getRealPath();
						$relativePath = "$directoryName/" . substr($fullPath, strlen($directoryPath) + 1);
						$zip->addFile($fullPath, $relativePath);
						$logger->debug("Adding $relativePath to archive.");
					}
				}
			}else{
				$zip->addFile($filePath, $fileName);
			}
		}
	}

	$zip->close();