<?php

	use MonologWrapper\MonologWrapper;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\Request;
	use Nox\Nox;
	use Nox\ORM\Abyss;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\DatabaseCredentials;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\BaseController;
	use Nox\Router\DynamicRoute;
	use Nox\Router\Exceptions\NoMatchingRoute;
	use Page\Page;
	use Page\PublicationStatus;
	use Redirects\Redirect;
	use Sitemaps\Sitemap;
	use Sitemaps\SitemapContentTypes;
	use Settings\Setting;
	use Settings\Settings;
	use Sitemaps\SitemapsService;
	use System\HttpHelper;
	use System\System;
	use System\SystemInstallationState;
	use System\Updater\UpdaterService;

	require_once __DIR__ . "/../vendor/autoload.php";

	$nox = new Nox();
	BaseController::$noxInstance = $nox;

	// Set a static file serving directory
	$nox->addStaticDirectory(
		uriStub: "/uplift-assets",
		directoryPath: __DIR__ . "/resources/static",
	);

	$nox->addStaticDirectory(
		uriStub: "/fbm-data",
		directoryPath: __DIR__ . "/../uplift-data",
	);

	$nox->addStaticDirectory(
		uriStub: "/uplift-data",
		directoryPath: __DIR__ . "/../uplift-data",
	);
	
	$nox->addStaticDirectory(
		uriStub: "/vendor/twbs",
		directoryPath: __DIR__ . "/../vendor/twbs",
	);

	// Handle legacy v3 to v4 builds that may have put resource directories from the layout into the
	// document root
	$legacyJSDirectory = __DIR__ . "/../js";
	$legacyImagesDirectory = __DIR__ . "/../images";
	$legacyFontsDirectory = __DIR__ . "/../fonts";
	$legacyStylesDirectory = __DIR__ . "/../styles";

	if (file_exists($legacyJSDirectory)){
		$nox->addStaticDirectory(
			uriStub: "/js",
			directoryPath: $legacyJSDirectory,
		);
	}

	if (file_exists($legacyImagesDirectory)){
		$nox->addStaticDirectory(
			uriStub: "/images",
			directoryPath: $legacyImagesDirectory,
		);
	}

	if (file_exists($legacyFontsDirectory)){
		$nox->addStaticDirectory(
			uriStub: "/fonts",
			directoryPath: $legacyFontsDirectory,
		);
	}

	if (file_exists($legacyStylesDirectory)){
		$nox->addStaticDirectory(
			uriStub: "/styles",
			directoryPath: $legacyStylesDirectory,
		);
	}

	// For compatability reasons, allow the root directory to be served. :( This sucks
	$nox->addStaticDirectory(
		uriStub: "/",
		directoryPath: __DIR__ . "/..",
	);

	// Support static file mime types so the browser can recognize the static files
	$nox->mapExtensionToMimeType("css", "text/css");
	$nox->mapExtensionToMimeType("scss", "text/css");
	$nox->mapExtensionToMimeType("map", "text/plain");
	$nox->mapExtensionToMimeType("png", "image/png");
	$nox->mapExtensionToMimeType("jpg", "image/jpeg");
	$nox->mapExtensionToMimeType("jpeg", "image/jpeg");
	$nox->mapExtensionToMimeType("js", "text/javascript");
	$nox->mapExtensionToMimeType("mjs", "text/javascript");
	$nox->mapExtensionToMimeType("gif", "image/gif");
	$nox->mapExtensionToMimeType("weba", "audio/webm");
	$nox->mapExtensionToMimeType("webm", "video/webm");
	$nox->mapExtensionToMimeType("webp", "image/webp");
	$nox->mapExtensionToMimeType("pdf", "application/pdf");
	$nox->mapExtensionToMimeType("svg", "image/svg+xml");
	$nox->mapExtensionToMimeType("ico", "image/vnd.microsoft.icon");
	$nox->mapExtensionToMimeType("html", "text/html");
	$nox->mapExtensionToMimeType("vcf", "text/x-vcard");

	// Font mimes
	$nox->mapExtensionToMimeType("otf", "font/otf");
	$nox->mapExtensionToMimeType("ttf", "font/ttf");
	$nox->mapExtensionToMimeType("woff", "font/woff");
	$nox->mapExtensionToMimeType("woff2", "font/woff2");
	$nox->mapExtensionToMimeType("eot", "application/vnd.ms-fontobject");

	// General mime cache control ages
	$nox->addCacheTimeForMime("text/html", 1800);
	$nox->addCacheTimeForMime("image/png", 86400 * 60);
	$nox->addCacheTimeForMime("image/jpeg", 86400 * 60);
	$nox->addCacheTimeForMime("text/plain", 1800);
	$nox->addCacheTimeForMime("text/gif", 86400 * 60);
	$nox->addCacheTimeForMime("text/svg", 86400 * 60);
	$nox->addCacheTimeForMime("image/webp", 86400 * 60);
	$nox->addCacheTimeForMime("image/vnd.microsoft.icon", 86400 * 60);

	// Define short cache-control ages for development or often-changed files.
	// It doesn't mean they will resend their full contents after expiratory,
	// as Nox implements ETag and checks to respond with 304 if the file wasn't modified
	$nox->addCacheTimeForMime("text/javascript", 60);
	$nox->addCacheTimeForMime("text/css", 60);

	// Font caches
	$nox->addCacheTimeForMime("font/otf", 86400 * 60);
	$nox->addCacheTimeForMime("font/ttf", 86400 * 60);
	$nox->addCacheTimeForMime("font/woff", 86400 * 60);
	$nox->addCacheTimeForMime("font/woff2", 86400 * 60);
	$nox->addCacheTimeForMime("application/vnd.ms-fontobject", 86400 * 60);

	// Process static files before anything else, to keep static file serving fast
	$nox->router->processRequestAsStaticFile();

	// If the code gets here, then it's not a static file. Load the rest of the setting directories
	$nox->setViewsDirectory(__DIR__ . "/resources/views");
	$nox->setLayoutsDirectory(__DIR__ . "/resources/layouts");
	$nox->setSourceCodeDirectory(__DIR__ . "/src");

	// Register Abyss credentials
	if (file_exists(__DIR__ . "/nox-env.php")){
		require_once __DIR__ . "/nox-env.php";
		try {
			Abyss::addCredentials(new DatabaseCredentials(
				host: NoxEnv::MYSQL_HOST,
				username: NoxEnv::MYSQL_USERNAME,
				password: NoxEnv::MYSQL_PASSWORD,
				database: NoxEnv::MYSQL_DB_NAME,
				port: NoxEnv::MYSQL_PORT,
			));
		} catch (mysqli_sql_exception $e) {
			$logger = MonologWrapper::getLogger();
			$logger->critical($e->getMessage());
			$nox->router->requestPath = "/invalid-database-credentials";
		}
	}else{
		$nox->router->requestPath = "/missing-nox-env-file";
	}

	// Only check installation state if we're not re-routing to a route
	// for having invalid DB credentials or a missing env file
	if ($nox->router->requestPath !== "/invalid-database-credentials") {
		$installationState = System::getInstallationState();
		if (
			$nox->router->requestPath !== "/missing-nox-env-file"
		) {
			/*
			 * Checks the installation state - this function also registered Abyss credentials
			 * if there are MySQL credentials in NoxEnv
			 */
			if ($installationState === SystemInstallationState::NO_DATABASE_TABLES) {
				if (
					$nox->router->requestPath !== "/invalid-database-credentials" // Not the invalid DB route
					&&
					$_SERVER['REQUEST_URI'] !== "/uplift/install" // Not the installation URL
					&&
					$_SERVER["REQUEST_METHOD"] === "GET" // It is a GET request
				) {
					// Redirect the request
					header("location: /uplift/install");
					exit();
				}
			} elseif ($installationState === SystemInstallationState::COMPLETE) {
				if ($_SERVER['REQUEST_URI'] === "/uplift/install") {
					// Installed but tried to access to install view
					// Redirect them to the admin panel
					header("location: /admin-panel");
					exit();
				}
			}
		}

		// Check if an update is currently happening
		if ($installationState === SystemInstallationState::COMPLETE) {
			if (UpdaterService::isSystemUpdating()) {
				http_response_code(503);
				exit();
			}
		}
	}

	// Check if the current scheme is https
	// If not, is force HTTPs on? If so, then redirect the user
	$isURLHttps = false;
	$isLocalHost = false;
	if (str_starts_with($_SERVER['HTTP_HOST'], "localhost") || str_starts_with($_SERVER['HTTP_HOST'], "wsl.localhost")){
		$isLocalHost = true;
	}

	if (isset($_SERVER['HTTPS']) && strtolower($_SERVER['HTTPS']) !== "off"){
		$isURLHttps = true;
	}

	if (!$isURLHttps){
		// Check if this is the localhost
		if (!$isLocalHost){
			// Redirect them if the setting for forcing HTTPs is set to "1"
			if (Setting::getSettingValue(Settings::FORCE_HTTPS->value) === "1") {
				http_response_code(301);
				header(sprintf("Location: https://%s%s", $_SERVER['HTTP_HOST'], $_SERVER['REQUEST_URI']));
				exit();
			}
		}
	}

	// Force "www" subdomain for "domain.com" domain forms
	$domainComponents = explode(".", $_SERVER['HTTP_HOST']);
	if (count($domainComponents) === 2 && !$isLocalHost){
		// Missing a subdomain. Redirect to www.domain.com
		http_response_code(301);
		header(sprintf("Location: %s://www.%s%s", $_SERVER['REQUEST_SCHEME'], $_SERVER['HTTP_HOST'], $_SERVER['REQUEST_URI']));
		exit();
	}

	// Process the request as a routable request
	try {
		$nox->router->processRoutableRequest();
	} catch (NoMatchingRoute) {

		// Run crons
		\Crons\CronsService::runCrons();

		// Handle the homepage if the site is suspended
		if(Setting::getSettingValue(Settings::SITE_SUSPENDED->value) === "1" && $nox->router->requestPath === "/") {
			$nox->router->addDynamicRoute(new DynamicRoute(
				requestMethod: "get",
				requestPath: "/",
				isRegex: false,
				onRender: (function () {
					return Renderer::renderView(
						viewFileName: "suspended/main.php"
					);
				})(...),
			));
		}

		// No matching internal route, try pages built with the CMS
		$userPages = Page::query();
		$sitemaps = Sitemap::query();

		// Honor redirects. Find any redirect that matches the current request path
		// or any redirects that are regular expression redirects
		/** @var Redirect[] $redirects */
		$redirects = Redirect::query(
			columnQuery: (new ColumnQuery())
				->where("from_uri","=",$nox->router->requestPath)
				->or()
				->where("is_regex","=",1),
		);

		foreach($redirects as $redirect){
			if ($redirect->isRegEx === 1){
				$didMatch = preg_match(
					pattern:"@{$redirect->fromURI}@",
					subject: $nox->router->requestPath,
				);
				if ($didMatch === 1){
					// Perform redirection
					http_response_code($redirect->statusCode);
					if ($redirect->preserveQueryString === 1){
						$queryString = HttpHelper::getQueryString();
						if (!empty($queryString)){
							header(sprintf("location: %s?$queryString", $redirect->to));
						}else{
							header(sprintf("location: %s", $redirect->to));
						}
					}else{
						header(sprintf("location: %s", $redirect->to));
					}
					exit();
				}
			}else{
				if (strtolower($redirect->fromURI) === strtolower($nox->router->requestPath)){
					// Perform redirection
					http_response_code($redirect->statusCode);
					if ($redirect->preserveQueryString === 1) {
						$queryString = HttpHelper::getQueryString();
						if (!empty($queryString)) {
							header(sprintf("location: %s?$queryString", $redirect->to));
						} else {
							header(sprintf("location: %s", $redirect->to));
						}
					}else{
						header(sprintf("location: %s", $redirect->to));
					}
					exit();
				}
			}
		}

		/** @var Page $page */
		foreach($userPages as $page){
			// Is this page published?
			if ($page->publicationStatus === PublicationStatus::Published->value) {
				if ($page->publicationTimestamp === 0 || $page->publicationTimestamp <= time()) {
					$nox->router->addDynamicRoute(new DynamicRoute(
						requestMethod: "get",
						requestPath: $page->pageRoute,
						isRegex: false,
						onRender: (function () use ($page) {
							// Add a Last-Modified header based on when the page was last edited
							$lastEditTime = $page->getLastEditTime();
							$lastModifiedDateTime = new DateTime("now", new DateTimeZone('UTC'));
							$lastModifiedDateTime->setTimestamp($lastEditTime);
							$lastModifiedDateTimestamp = $lastModifiedDateTime->format('D, d M Y H:i:s \G\M\T');

							// Tries to add a Link header for canonicals, if a base URL can be found
							try{
								$baseUrl = HttpHelper::getWebsiteBaseURL();
								header(sprintf("link: <%s%s>; rel=\"canonical\"", $baseUrl, $page->pageRoute));
							}catch(Exception){}

							header(sprintf("last-modified: %s", $lastModifiedDateTimestamp));
							header(sprintf("etag: %d", $lastEditTime));

							// Check if we should just return a 304 status because this page wasn't modified
							$ifNoneMatch = Request::getFirstHeaderValue("If-None-Match");
							if ((string) $lastEditTime === $ifNoneMatch){
								http_response_code(304);
								exit();
							}

							// Set the currently rendered page so other processes (like the ShortcodeParser)
							// can read from the currently rendered page.
							Page::setCurrentPage($page);
							return $page->getRenderedHTML();
						})(...),
					));
				}
			}
		}

		/** @var Sitemap $sitemap */
		foreach($sitemaps as $sitemap){
			$nox->router->addDynamicRoute(new DynamicRoute(
				requestMethod: "get",
				requestPath: $sitemap->route,
				isRegex: false,
				onRender: (function () use ($sitemap) {
					foreach(SitemapContentTypes::cases() as $case) {
						if($case->value === $sitemap->contentType){
							if($case->name === "XML") {
								header('Content-Type: application/xml; charset=utf-8');
							} elseif($case->name === "RSS") {
								header('Content-Type: application/rss+xml; charset=utf-8');
							} elseif($case->name === "Text") {
								header('Content-Type: text/plain; charset=utf-8');
							}
						}
					}

					$sitemapContent = $sitemap->content;
					header("content-length: " . strlen($sitemapContent));

					// Tries to add a Link header for canonicals, if a base URL can be found
					try{
						$baseUrl = HttpHelper::getWebsiteBaseURL();
						header(sprintf("link: <%s%s>; rel=\"canonical\"", $baseUrl, $sitemap->route));
					}catch(Exception){}

					return $sitemapContent;
				})(...),
			));
		}

		// Add the default sitemap.xml here, so it can be overridden by the user in the above dynamic routes
		$nox->router->addDynamicRoute(new DynamicRoute(
			requestMethod:"get",
			requestPath:"/sitemap.xml",
			isRegex:false,
			onRender: (function(){
				header('Content-type: text/xml');
				header('Pragma: public');
				header('Cache-control: private');
				header('Expires: -1');

				// Tries to add a Link header for canonicals, if a base URL can be found
				try{
					$baseUrl = HttpHelper::getWebsiteBaseURL();
					header(sprintf("link: <%s%s>; rel=\"canonical\"", $baseUrl, "/sitemap.xml"));
				}catch(Exception){}

				return SitemapsService::getDefaultSitemapXMLContent();
			})(...),
		));

		try {
			$nox->router->processRoutableRequest();
		}catch(NoMatchingRoute){
			// This is a 404
			// 404
			http_response_code(404);
			// Process a new routable request, but change the path to our known 404 controller method route
			$nox->router->requestPath = "/404";

			try {
				$nox->router->processRoutableRequest();
			}catch(NoMatchingRoute){
				// There is no 404 route
				print("Missing /404 page.");
				exit();
			}
		}
	}