import Scheduler from "../../utils/Scheduler.js";
import SubmittingError from "../components/SubmittingError.js";
import ProjectMediaForm from "./ProjectMediaForm.js";
import Loader from "../components/Loader.js";
import SuccessBox from "../components/SuccessBox.js";
// import { ClientProjectForm } from "../ClientProjectForm.js";

class ProjectTextualForm{
	form = document.querySelector("#client-project-form-textual-section");
	submitButton = document.querySelector("#form-submit-button");
	formCardContainer = document.querySelector("#form-card");
	isProcessing = false;
	successBox = new SuccessBox();
	loader = new Loader();
	submittingError = new SubmittingError(this);
	mediaForm = new ProjectMediaForm(this);
	/**
	 * @type {ClientProjectForm}
	 */
	ClientProjectFormInstance;

	/**
	 * @param {ClientProjectForm} clientProjectFormInstance
	 */
	constructor(clientProjectFormInstance){
		this.ClientProjectFormInstance = clientProjectFormInstance;
		this.form.addEventListener("submit", e => {
			e.preventDefault();
			this.onSubmit();
		});
	}

	async onSubmit(){
		if (this.isProcessing){
			return;
		}

		this.isProcessing = true;
		this.submitButton.classList.add("disabled");
		this.formCardContainer.style.display = "none";
		this.submittingError.hide();
		this.loader.show();
		this.loader.setTextualLoaderAsInProgress();
		this.loader.setMediaLoaderAsWaiting();

		// Check that the minimum amount of images has been added to the upload queue
		if (!this.mediaForm.doesHaveMinimumImagesSetToUpload()){
			this.submittingError.showError("You must upload a cover photo and one (1) additional image for a project post to be valid.");
			this.loader.hide();
			this.isProcessing = false;
			this.submitButton.classList.remove("disabled");
			return;
		}

		// Check all file names are valid
		const resultFromImageValidations = await this.mediaForm.validateAllImageFileNames();
		if (!resultFromImageValidations){
			this.loader.hide();
			this.isProcessing = false;
			this.submitButton.classList.remove("disabled");
			return;
		}

		const fData = new FormData(this.form);

		// Add the selected project tags, if any
		const selectedTagIds = this.ClientProjectFormInstance.ProjectTags.GetSelectedTags();
		fData.set("json-project-tag-ids", JSON.stringify(selectedTagIds));

		const response = await fetch(`/uplift/ipp-form/create-post`, {
		    body:fData,
		    method:"POST",
		    cache:"no-cache",
		    credentials:"same-origin"
		});

		// Minimum wait time
		await Scheduler.wait(850);

		if (response.status === 403){
			// They were logged out while trying to submit a post
			this.loader.hide();
			this.isProcessing = false;
			this.submitButton.classList.remove("disabled");
			this.submittingError.showError("You have been logged out while submitting this project post. You can open a new browser tab and go to the IPP form again to login.");
			return;
		}

		let data;
		try{
		    /** @type {{status: int, error: ?string, newPageID: ?int, projectImagesDirectory: ?string}} **/
		    data = await response.json();
		}catch(jsonSyntaxError){
			this.loader.hide();
			this.isProcessing = false;
			this.submitButton.classList.remove("disabled");
			this.submittingError.showError("The system received an error back from the local server. Please report this to your account representative to have a developer look into this problem. You can also screenshot this error for reference to show the account rep.");
		    return;
		}

		if (data.status === 1){
			this.mediaForm.currentProjectPageID = parseInt(data.newPageID);
			this.mediaForm.currentProjectImagesUploadDirectory = data.projectImagesDirectory;

			this.loader.setTextualLoaderAsFinished();

			/**
			 * The list of successfully uploaded image file names (does not include the cover photo).
			 * @type {string[]|null}
			 */
			let listOfUploadedImageFileNames = [];
			try {
				/**
				 * The list of successfully uploaded image file names (does not include the cover photo).
				 * @type {string[]|null}
				 */
				listOfUploadedImageFileNames = await this.mediaForm.onSubmit();
			}catch(errorMessage){
				// If there was an error, request the newly created IPP be deleted.
				const response = await fetch(`/uplift/ipp-form/undo-ipp-upload/${data.newPageID}`, {
					method:"DELETE",
					cache:"no-cache",
					credentials:"same-origin"
				});
				console.warn(errorMessage);
				this.loader.hide();
				throw errorMessage;
			}

			const finalizeForm = new FormData();
			finalizeForm.set("project-post-id", String(data.newPageID));
			finalizeForm.set("project-images-directory", data.projectImagesDirectory);
			finalizeForm.set("json-list-of-uploaded-image-file-names", JSON.stringify(listOfUploadedImageFileNames));

			// Submit the finalization
			const response = await fetch("/uplift/ipp-form/finalize", {
				body:finalizeForm,
			    method:"POST",
			    cache:"no-cache",
			    credentials:"same-origin"
			});

			let finalizeData;
			try{
			    /** @type {{status: int, error: ?string, projectURL: string}} **/
				finalizeData = await response.json();
			}catch(jsonSyntaxError){
				this.loader.hide();;
				this.isProcessing = false;
				this.submitButton.classList.remove("disabled");
				this.submittingError.showError("The system received an error back from the local server. Please report this to your account representative to have a developer look into this problem. You can also screenshot this error for reference to show the account rep.");
				return;
			}

			if (finalizeData.status === 1){
				this.successBox.show();
				document.querySelector("#view-project-button").setAttribute("href", finalizeData.projectURL);
			}else if (finalizeData.status === -1){
				this.submittingError.showError(finalizeData.error);
			}
		}else if (data.status === -1){
			this.submittingError.showError(data.error);
		}

		this.isProcessing = false;
		this.submitButton.classList.remove("disabled");
		this.loader.hide();
		// this.formCardContainer.style.display = null;
	}
}

export default ProjectTextualForm;