import LinkContextMenu from "./LinkContextMenu.js";

/**
* Loads Ace Editor extensions and default
* properties.
*/
class AceEditorExtension{

	static pageEditors = {
		/**
		 * The body editor here will be null for layouts that have
		 * no PageLayoutSectionsDefinition object
		 */
		body: null,
		head: null
	};

	/**
	 * Set common defaults for the Ace Session
	 * @param aceSession
	 * @param {string|null} defaultLanguage
	 */
	static setSessionDefaults(aceSession, defaultLanguage){
		aceSession.setUseSoftTabs(false);
		aceSession.setTabSize(4);
		aceSession.setUseWrapMode(true);

		if (defaultLanguage !== null){
			aceSession.setMode("ace/mode/" + defaultLanguage);
		}

		// Disable the DOCTYPE notification
		aceSession.on("changeAnnotation", () => {
			const annotations = aceSession.getAnnotations() || [];
			const len = annotations.length;
			let i = len;
			while (i--) {
				if(/doctype first\. Expected/.test(annotations[i].text)) {
					annotations.splice(i, 1);
				}
			}
			if (len > annotations.length) {
				aceSession.setAnnotations(annotations);
			}
		});
	}

	/**
	 * @param {HTMLElement} element
	 * @param {?string} defaultLanguage
	 */
	constructor(element, defaultLanguage){
		this.ace = ace.edit(element);
		this.session = this.ace.getSession();
		this.aceContainer = element;
		this.currentLinkContextMenu = null;

		/*
			You can set the default language of an ace editor
			by adding ace-lang="html" to the div/container
			that the ace editor will be on
		*/
		if (this.aceContainer.getAttribute("ace-lang") !== null) {
			this.aceLanguage = this.aceContainer.getAttribute("ace-lang");
		}

		// Set defaults

		this.ace.setHighlightActiveLine(true);
		//editor.setTheme("ace/theme/textmate");
		this.ace.setOption("showPrintMargin", false);
		this.ace.setFontSize(14);
		this.ace.setOptions({
			scrollPastEnd:0.4
		});
		this.ace.setWrapBehavioursEnabled(true);
		this.ace.setBehavioursEnabled(false); // Stop the annoying stupid auto-closing tags

		AceEditorExtension.setSessionDefaults(this.session, defaultLanguage);

		// editor.commands.addCommand({
		// 	name: 'saveFile',
		// 	bindKey: {
		// 		win: 'Ctrl-S',
		// 		mac: 'Command-S',
		// 		sender: 'editor|cli'
		// 	},
		// 	exec: function(env, args, request) {
		// 		$(v).trigger("save-command", [editor]);
		// 	}
		// });

		// Fix ace editor not being resized when content is pasted
		this.ace.on("paste", () => {
			setTimeout(() => {
				this.ace.resize(true);
			});
		});
	}

	/**
	 * Loads all custom defined methods of AceEditorExtension for this instance
	 */
	loadExtensions(){
		this.loadHighlightInnerTag();
		this.loadLinkize();
		this.loadBold();
		this.loadParagraphize();
		this.loadLinkContextMenu();
		this.loadPasteFormatter();
	}

	/**
	* Load a command that will allow Ctrl + p to
	* wrap text into paragraph tags
	*/
	loadParagraphize(){
		const aceEditor = this.ace;
		aceEditor.commands.addCommand({
			name:"Paragraphize",
			exec: () => {
				const selectionRange = aceEditor.getSelectionRange();
				const currentlySelectedText = aceEditor.getSelectedText();
				const tabStartCheck = currentlySelectedText.match(/^\t+/);
				const startsWithTabs = tabStartCheck !== null;
				let tabPadding = "";
				let newString = ""; // A buffer of the paragraphized string

				if (startsWithTabs){
					tabPadding = tabStartCheck[0];
				}

				if (currentlySelectedText.length > 0){
					const endText = "\n" + tabPadding + "</p>";
					let startString = tabPadding + "<p>\n" + tabPadding;
					let injectedIndent = "";

					if (tabPadding.length === 0){
						// Editor will not auto-insert a tab if there was not one originally
						// So inject it manually
						injectedIndent = "\t";
						startString += injectedIndent;
					}

					// Separate the string by sections
					// Delimited by multiple new lines
					const stringParts = currentlySelectedText.split("\n\n");

					let counter = 0;
					for (let part of stringParts){
						newString += startString;
						newString += part.trim();
						newString += endText;

						if (counter !== stringParts.length - 1){
							// Not the last index
							newString += "\n";
						}

						++counter;
					}

					aceEditor.session.replace(selectionRange, newString);
				}
			},
			bindKey:{mac:"cmd-p", win:"ctrl-p"}
		});
	}

	/**
	* Adds bold command to ace editor with Ctrl + B
	*/
	loadBold(){
		const aceEditor = this.ace;
		const AceRange = ace.require("ace/range").Range;

		aceEditor.commands.addCommand({
			name:"Bold Text",
			exec: () => {
				const selectionRange = aceEditor.getSelectionRange();
				if (aceEditor.getSelectedText().length > 0){
					aceEditor.session.insert(
						selectionRange.start,
						"<strong>"
					);
					aceEditor.session.insert(
						aceEditor.getSelectionRange().end,
						"</strong>"
					);

					// Reset the range to remove the now-selected </strong>
					const currentSelectionRange = aceEditor.getSelectionRange();
					let newSelectionEndPositionRow = currentSelectionRange.end.row;
					let newSelectionEndPositionColumn = currentSelectionRange.end.column - ("</strong>").length;

					if (newSelectionEndPositionColumn < 0){
						// Go up a row
						newSelectionEndPositionRow -= 1;
						newSelectionEndPositionColumn = aceEditor.getSession().doc.getAllLines()[newSelectionEndPositionRow].length + newSelectionEndPositionColumn;
					}

					aceEditor.session.selection.setSelectionRange(new AceRange(currentSelectionRange.start.row, currentSelectionRange.start.column, newSelectionEndPositionRow, newSelectionEndPositionColumn));
				}
			},
			bindKey:{mac:"cmd-b", win:"ctrl-b"}
		});
	}

	/**
	* Command to wrap a text in a link/anchor using Ctrl + L
	*/
	loadLinkize(){
		const aceEditor = this.ace;
		const AceRange = ace.require("ace/range").Range;

		aceEditor.commands.addCommand({
			name:"Link text",
			exec: () => {
				const selectionRange = aceEditor.getSelectionRange();
				if (aceEditor.getSelectedText().length > 0){
					aceEditor.session.insert(
						selectionRange.start,
						"<a href=\"#link\">"
					);
					aceEditor.session.insert(
						aceEditor.getSelectionRange().end,
						"</a>"
					);

					// Reset the range to remove the now-selected </strong>
					const currentSelectionRange = aceEditor.getSelectionRange();
					let newSelectionEndPositionRow = currentSelectionRange.end.row;
					let newSelectionEndPositionColumn = currentSelectionRange.end.column - ("</a>").length;

					if (newSelectionEndPositionColumn < 0){
						// Go up a row
						newSelectionEndPositionRow -= 1;
						newSelectionEndPositionColumn = aceEditor.getSession().doc.getAllLines()[newSelectionEndPositionRow].length + newSelectionEndPositionColumn;
					}

					aceEditor.session.selection.setSelectionRange(new AceRange(currentSelectionRange.start.row, currentSelectionRange.start.column, newSelectionEndPositionRow, newSelectionEndPositionColumn));
				}
			},
			bindKey:{mac:"cmd-l", win:"ctrl-l"}
		});
	}

	/**
	* Press Ctrl + Right Clicking inside of an HTML tag
	* will cause everything in between the < and > to be
	* highlighted/selected
	*/
	loadHighlightInnerTag(){
		const aceEditor = this.ace;
		const AceRange = ace.require("ace/range").Range;

		aceEditor.container.addEventListener("contextmenu", e => {
			if (e.ctrlKey === true){
				e.preventDefault();

				const thisSession = aceEditor.getSession();
				const thisDocument = thisSession.doc;
				const allLines = thisDocument.getAllLines();
				const lastLineNumber = allLines.length - 1; // Last row possible
				const lastColumnOnLastLine = allLines[lastLineNumber].length; // Last column on last line

				// Get cursor position in ace editor
				const cursorPosition = aceEditor.getCursorPosition(); // Gives an object with `row`, `column` keys

				// Move forward until an ending bracket is met
				let currentForwardRow = cursorPosition.row;
				let currentForwardColumn = cursorPosition.column;
				let maxForwardColumn = allLines[currentForwardRow].length;
				let forwardEndRow;
				let forwardEndColumn;
				while (1 == 1 && 2 !== 3){

					// Increment check
					if (maxForwardColumn === currentForwardColumn){
						if (currentForwardRow < lastLineNumber){
							// Move to the next line
							++currentForwardRow;
							currentForwardColumn = 0;
							maxForwardColumn = allLines[currentForwardRow].length;
						}else{
							// Hit the end
							break;
						}
					}

					const currentCharacterRange = new AceRange(currentForwardRow, currentForwardColumn, currentForwardRow, currentForwardColumn + 1);
					const currentCharacter = thisSession.getTextRange(currentCharacterRange);

					if (currentCharacter === ">"){
						forwardEndRow = currentForwardRow;
						forwardEndColumn = currentForwardColumn;
						break;
					}
					++currentForwardColumn;
				}

				// Move backward until an opening bracket is met
				let currentBackwardRow = cursorPosition.row;
				let currentBackwardColumn = cursorPosition.column;
				let backwardEndRow;
				let backwardEndColumn;
				while (1 == 1 && 2 !== 3){

					// Increment check
					if (0 === currentBackwardColumn){
						if (currentBackwardRow > 0){
							// Move to the prior line
							--currentBackwardRow;
							currentBackwardColumn = allLines[currentBackwardRow].length;
						}else{
							// Hit the end
							break;
						}
					}

					const currentCharacterRange = new AceRange(currentBackwardRow, currentBackwardColumn - 1, currentBackwardRow, currentBackwardColumn);
					const currentCharacter = thisSession.getTextRange(currentCharacterRange);

					if (currentCharacter === "<"){
						backwardEndRow = currentBackwardRow;
						backwardEndColumn = currentBackwardColumn;
						break;
					}else if (currentCharacter === ">"){
						break;
					}

					--currentBackwardColumn;
				}

				if (backwardEndRow !== undefined && forwardEndRow !== undefined){
					// Found locations
					// Select it
					// -1 and +1 to include the brackets
					const tagRange = new AceRange(backwardEndRow, backwardEndColumn - 1, forwardEndRow, forwardEndColumn + 1);
					thisSession.selection.setSelectionRange(tagRange);
				}
			}
		});
	}

	/**
	* Loads the HTML parser for pastes that are not raw-paste
	*/
	loadPasteFormatter(){
		const aceEditor = this.ace;
		aceEditor.commands.on("exec", e => {
			const command = e.command;

			// Capture the paste event before it adds text
			// onPaste and .on("paste") fire too late
			// So using "exec" and capturing the event here allows
			// preventDefault() to actually work
			if (command.name === "paste"){
				const originalEvent = e.args.event;
				const html = originalEvent.clipboardData.getData("text/html");
				let fragmentMatch = html.match(new RegExp("<!--StartFragment-->(.+)<!--EndFragment-->"));

				if (fragmentMatch !== null){
					e.preventDefault();
					let rawHtml = fragmentMatch[1];
					let unstyledHtml = rawHtml.replace(/ style="[^"<>]+"/g, "");
					let unclassedHtml = unstyledHtml.replace(/ class="[^"<>]+"/g, "");

					// Remove empty span elements
					let spaceFixedHtml = unclassedHtml.replace(/<span>\s*<\/span>/g," ");

					// Add newlines and a tab to paragraphs
					spaceFixedHtml = spaceFixedHtml.replace(/<p>/g, "<p>\n\t");
					spaceFixedHtml = spaceFixedHtml.replace(/<\/p>/g, "\n</p>\n");

					// Add proper newline to headings
					spaceFixedHtml = spaceFixedHtml.replace(/<\/h([0-9])>/g, "</h$1>\n");

					// Format unordered and ordered lists
					spaceFixedHtml = spaceFixedHtml.replace(/<([uo])l>/g, "<$1l>\n");
					spaceFixedHtml = spaceFixedHtml.replace(/<\/([uo])l>/g, "</$1l>\n");

					// Remove div tags
					spaceFixedHtml = spaceFixedHtml.replace(/<div>/g, "");
					spaceFixedHtml = spaceFixedHtml.replace(/<\/div>/g, "");

					// Remove <br> breaks
					spaceFixedHtml = spaceFixedHtml.replace(/<br.*>\n*/g, "");
					spaceFixedHtml = spaceFixedHtml.replace(/<\/br.*>\n*/g, "");

					spaceFixedHtml = spaceFixedHtml.replace(/<\/li>/g, "</li>\n");
					spaceFixedHtml = spaceFixedHtml.replace(/<li>/g, "\t<li>");

					// Remove links and replace them with #link
					spaceFixedHtml = spaceFixedHtml.replace(/href=".*?"/g, "href=\"#link\"");

					aceEditor.insert(spaceFixedHtml.trim());
				}

			}
		});
	}

	/**
	* Loads the right-click link menu feature.
	* Only works if the cursor is in between href attribute quotes.
	*/
	loadLinkContextMenu(){
		const aceEditor = this.ace;
		const AceRange = ace.require("ace/range").Range;

		aceEditor.container.addEventListener("contextmenu", e => {
			const loopLimit = 20000;
			let numLoops = 0;

			const thisSession = aceEditor.getSession();
			const thisDocument = thisSession.doc;
			const allLines = thisDocument.getAllLines();
			const lastLineNumber = allLines.length - 1; // Last row possible
			const lastColumnOnLastLine = allLines[lastLineNumber].length; // Last column on last line

			// Only show the modal if the selection is empty
			if (!thisSession.getSelection().isEmpty()){
				return;
			}

			// Get cursor position in ace editor
			const cursorPosition = aceEditor.getCursorPosition(); // Gives an object with `row`, `column` keys

			// Move forward until an ending bracket is met
			let currentForwardRow = cursorPosition.row;
			let currentForwardColumn = cursorPosition.column;
			let maxForwardColumn = allLines[currentForwardRow].length;
			let forwardEndRow;
			let forwardEndColumn;

			let forwardBuffer = "";
			let backwardBuffer = "";


			// Keep moving forward until a quotation is hit
			while (1 == 1 && 2 == 2 && 3 == 3 && "john" != "henrik"){

				if (numLoops >= loopLimit){
					break;
				}

				++numLoops;

				// Increment check
				if (maxForwardColumn === currentForwardColumn){
					if (currentForwardRow < lastLineNumber){
						// Move to the next line
						++currentForwardRow;
						currentForwardColumn = 0;
						maxForwardColumn = allLines[currentForwardRow].length;
					}else{
						// Hit the end
						break;
					}
				}

				const currentCharacterRange = new AceRange(currentForwardRow, currentForwardColumn, currentForwardRow, currentForwardColumn + 1);
				const currentCharacter = thisSession.getTextRange(currentCharacterRange);

				if (currentCharacter !== "\""){
					forwardBuffer += currentCharacter;
				}else{
					// Found a quote. Done moving forward
					break;
				}

				++currentForwardColumn;
			}

			// Move backwards until hitting a quote
			let currentBackwardRow = cursorPosition.row;
			let currentBackwardColumn = cursorPosition.column;
			let backwardEndRow;
			let backwardEndColumn;
			while (1 == 1 && 2 !== 3){

				if (numLoops >= loopLimit){
					break;
				}

				++numLoops;

				// Increment check
				if (0 === currentBackwardColumn){
					if (currentBackwardRow > 0){
						// Move to the prior line
						--currentBackwardRow;
						currentBackwardColumn = allLines[currentBackwardRow].length;
					}else{
						// Hit the end
						break;
					}
				}

				const currentCharacterRange = new AceRange(currentBackwardRow, currentBackwardColumn - 1, currentBackwardRow, currentBackwardColumn);
				const currentCharacter = thisSession.getTextRange(currentCharacterRange);

				if (currentCharacter !== "\""){
					backwardBuffer += currentCharacter;
				}else{
					// Found a quote. Done moving backward
					break;
				}

				--currentBackwardColumn;
			}

			let beforeInnerTextForwardsRow = parseInt(currentForwardRow, 10);
			let beforeInnerTextForwardsColumn = parseInt(currentForwardColumn, 10);

			// Check the character before the last backwards character
			// Is it an = (equal sign)?
			// That means an attribute was probably being set. Yay
			let currentCharacterRange = new AceRange(currentBackwardRow, currentBackwardColumn - 2, currentBackwardRow, currentBackwardColumn - 1);
			let currentCharacter = thisSession.getTextRange(currentCharacterRange);
			if (currentCharacter === "="){
				// Go back 4 more charatcters
				// Is it an href?
				let currentCharacterRange = new AceRange(currentBackwardRow, currentBackwardColumn - 6, currentBackwardRow, currentBackwardColumn - 2);
				let currentMatchSet = thisSession.getTextRange(currentCharacterRange);
				if (currentMatchSet === "href"){
					e.preventDefault();

					let innerAnchorText = "";
					let currentParseState = "in-attributes";

					// Get the inner content of the anchor
					while (true){
						if (numLoops >= loopLimit){
							break;
						}

						++numLoops;

						// Increment check
						if (maxForwardColumn === currentForwardColumn){
							if (currentForwardRow < lastLineNumber){
								// Move to the next line
								++currentForwardRow;
								currentForwardColumn = 0;
								maxForwardColumn = allLines[currentForwardRow].length;
							}else{
								// Hit the end
								break;
							}
						}

						const currentCharacterRange = new AceRange(currentForwardRow, currentForwardColumn, currentForwardRow, currentForwardColumn + 1);
						const currentCharacter = thisSession.getTextRange(currentCharacterRange);

						if (currentParseState == "in-attributes"){
							if (currentCharacter === ">"){
								currentParseState = "text-content";
							}
						}else if (currentParseState === "text-content"){
							if (currentCharacter === "<"){
								break; // Done parsing
							}else{
								innerAnchorText += currentCharacter;
							}
						}

						++currentForwardColumn;
					}

					// Show the link modal
					if (this.currentLinkContextMenu !== null){
						this.currentLinkContextMenu.cleanup();
					}
					const cm = new LinkContextMenu(document.body, e.pageX, e.pageY, innerAnchorText);
					const replacementRange = new AceRange(
						currentBackwardRow, currentBackwardColumn, beforeInnerTextForwardsRow, beforeInnerTextForwardsColumn
					);
					// cm.performSearch();
					cm.onPageSelected((pageName, pageRoute) => {
						thisSession.replace(replacementRange, pageRoute);
						thisSession.selection.setSelectionRange(new AceRange(
							currentBackwardRow, currentBackwardColumn, currentBackwardRow, currentBackwardColumn + pageRoute.length
						));
						cm.cleanup();
					});

					this.currentLinkContextMenu = cm;
				}
			}
		});
	}

	/**
	 * @param {ImageComponent} imageComponent
	 * @return {HTMLPictureElement}
	 */
	getImageComponent(imageComponent){
		const template = document.createElement("picture");
		const img = document.createElement("img");


		if (imageComponent.fileExtension !== "webp") {
			img.setAttribute("src", imageComponent.uri);
			img.setAttribute("alt", imageComponent.tryToCreateAlt())
			img.setAttribute("class","");
			template.append(img);
		}else{
			// This is a WebP, so also insert the fallback URI. If none exists, then warn the user.
			if (imageComponent.fallbackImage !== null){
				/** @type {{fileExtension: string, fileName: string, fileNameWithoutExtension: string, uri: string, thumbURI: string}} */
				const fallbackImage = imageComponent.fallbackImage;
				const sourceElement = document.createElement("source");
				let fileSourceType = "image/webp";

				sourceElement.setAttribute("type", fileSourceType);
				sourceElement.setAttribute("srcset", imageComponent.uri);

				img.setAttribute("src", fallbackImage.uri);
				img.setAttribute("alt", imageComponent.tryToCreateAlt())
				img.setAttribute("class","");

				template.append(sourceElement);
				template.append(img);
			}else{
				alert("You are inserting a WebP image and have not made a fallback PNG or JPEG. It is highly recommended you clone the WebP and convert it to a JPEG or PNG image for older iOS devices that do not support WebP.");
				img.setAttribute("src", imageComponent.uri);
				img.setAttribute("alt", imageComponent.tryToCreateAlt())
				img.setAttribute("class","");
				template.append(img);
			}
		}

		return template;
	}

	/**
	 * Returns all whitespace (tabs or spaces) on the current line before
	 * the first non-whitespace is hit.
	 * @return {string}
	 */
	getAllWhitespaceCharactersStartingLine(){
		/**
		 * Get the cursor's current row/column position
		 * @type {{row: int, column: int}}
		 */
		const cursorPosition = this.ace.getSession().getSelection().getCursor();
		const textOnLine = this.ace.getSession().getLine(cursorPosition.row);

		let buffer = "";
		for (const char of textOnLine){
			if (char !== "\t" && char !== " "){
				break;
			}else{
				buffer += char;
			}
		}

		return buffer;
	}

	/**
	 * When ImageComponents are selected to be inserted from the editor.
	 * @param {ImageComponent[]} imageComponents
	 */
	insertImages(imageComponents){
		for (const component of imageComponents){
			const element = this.getImageComponent(component);

			// Get the whitespace (space or tab) starting the line
			const whitespaceStartingLineCursorIsOne = this.getAllWhitespaceCharactersStartingLine();

			// Add newlines and tabs
			const outerHTML = element.outerHTML;
			const formattedHTML = outerHTML
				.replace("<picture>", `<picture>\n`)
				.replace(/<source(.+?)>/, `${whitespaceStartingLineCursorIsOne}\t<source$1>\n`)
				.replace("<img", `${whitespaceStartingLineCursorIsOne}\t<img`)
				.replace("</picture>", `\n${whitespaceStartingLineCursorIsOne}</picture>\n`);
			this.ace.insert(formattedHTML);

			if (imageComponents.length === 1) {
				// Find the last occurrence of class="" - which is inserted with the <img> element HTML
				/** @type {{start: {row: int, column: int}, end: {row: int, column: int}} | undefined}} */
				const findResult = this.ace.find("class=\"\"", {backwards: true}, false);
				if (findResult !== undefined) {
					this.ace.moveCursorTo(findResult.end.row, findResult.end.column - 1);
					this.ace.clearSelection();
					this.ace.focus();
				}
			}
		}
	}
}

export default AceEditorExtension;
