class LinkContextMenu{

	currentSearchJobID = 0;

	constructor(parent, posx, posy, defaultText){

		this.pageSelectedEventFunctions = [];

		const menu = LinkContextMenu.getDOM();

		/** @type {HTMLInputElement} */
		const inputField = menu.querySelector(".link-cm-search");
		const spinner = menu.querySelector(".link-search-loader");
		const form = menu.querySelector("#link-cm-search-form");

		inputField.value = defaultText.trim();

		this.menu = menu;
		this.inputField = inputField;
		this.spinner = spinner;
		this.resultsContainer = menu.querySelector(".link-search-cm-container");

		// Check if the menu would go off the screen
		// 280 is the average size of the context menu
		if ((posx + 280) >= window.innerWidth){
			menu.style.left = String(posx-280) + "px";
		}else{
			menu.style.left = String(posx) + "px";
		}

		if ((posy + 350) >= window.innerHeight){
			menu.style.top = String(posy-350) + "px";
		}else{
			menu.style.top = String(posy) + "px";
		}

		parent.append(menu);

		let func;

		// Function to remove the menu
		func = function(e){
			if (e.target.closest(".link-context-menu") !== menu){
				menu.remove();
				document.body.removeEventListener("click", func);
			}
		}

		// Trigger this event when clicking anywhere in the window
		window.addEventListener("click", func);

		form.addEventListener("submit", e => {
			e.preventDefault();
			this.performSearch();
		});

		this.inputField.addEventListener("input", () => {
			this.performSearch();
		});

		// Focus the input field
		inputField.focus();

		// Perform a default search on creation
		this.performSearch();
	}

	cleanup(){
		this.menu.remove();
	}

	async performSearch(){
		const query = this.inputField.value.trim();

		this.resultsContainer.innerHTML = "";

		// Don't perform a search on empty queries
		if (query.length === 0){
			return;
		}

		this.spinner.style.display = null;

		const urlSearchParams = new URLSearchParams();
		urlSearchParams.set("query", query);

		// Log the current job ID in case another search happens before this one returns
		// This way we don't render overlapping results and just throw outdated responses away
		const currentJobID = ++this.currentSearchJobID;

		const response = await fetch(`/uplift/page-editor/search-pages?${urlSearchParams.toString()}`, {
			method:"GET",
			cache:"no-cache",
			credentials:"same-origin"
		});

		this.spinner.style.display = "none";

		if (response.status === 200) {

			if (currentJobID === this.currentSearchJobID) {
				/** @type {{status: int, pages: Object[]}} */
				const data = await response.json();
				const pages = data.pages;

				for (let pageData of pages) {
					this.renderResultItem(pageData);
				}
			}else{
				// Throw away result
			}
		}else{
			/** @type {{status: int, error: string}} */
			try {
				const data = await response.json();
				alert(`${response.status} Error: ` + data.error);
			}catch(jsonSyntaxError){
				alert(`${response.status} Error: Server sent back a non-JSON response.`);
			}

		}

	}

	/**
	 * @param {{pageName: string, pageRoute: string}} data
	 */
	renderResultItem(data){
		const container = document.createElement("a");
		container.setAttribute("title",data.pageName);
		container.classList.add("link-cm-result-item");
		container.innerHTML = `
			<div class="link-cm-result-page-name-container">
				<span>${data.pageName}</span>
			</div>
			<div class="link-cm-result-page-uri-container">
				<span>${data.pageRoute}</span>
			</div>
		`;

		this.menu.querySelector(".link-search-cm-container").append(container);

		container.addEventListener("click", e => {
			// Fire events
			for (let func of this.pageSelectedEventFunctions){
				func(data.pageName, data.pageRoute);
			}
		});
	}

	onPageSelected(func){
		this.pageSelectedEventFunctions.push(func);
	}

	static getDOM(){
		const container = document.createElement("div");
		container.classList.add("link-context-menu");
		container.innerHTML = `
			<div>
				<form id="link-cm-search-form">
					<input type="text" class="link-cm-search form-control form-control-sm" placeholder="Search pages">
				</form>
			</div>
			<hr>
			<div class="link-search-loader" style="display:none;">
				<div class="spinner-border text-primary" role="status"></div>
			</div>
			<div class="link-search-cm-container"></div>
		`;

		return container
	}
}

export default LinkContextMenu;
