import EditorSidebar from "./EditorSidebar.js";
import Scheduler from "../../utils/Scheduler.js";
import Endpoints from "../../Endpoints.js";
import BlogReelModal from "../page-editor/modals/BlogReelModal.js";
import IPPReelModal from "../page-editor/modals/IPPReelModal.js";
import RecentArticlesModal from "../page-editor/modals/RecentArticlesModal.js";
import RecentProjectsModal from "../page-editor/modals/RecentProjectsModal.js";
import BlogReelForm from "./forms/BlogReelForm.js";
import IPPReelForm from "./forms/IPPReelForm.js";
import RecentArticlesForm from "./forms/RecentArticlesForm.js";
import RecentProjectsForm from "./forms/RecentProjectsForm.js"
import {PageContentSectionEditor} from "./components/PageContentSectionEditor.js";

class EasyEditorManager{
	CurrentPage;
	/**
	 * The currently loaded page's PageContentSections - can be a blank array
	 * @type {{id: number, pageId: number, sectionName: string, content: string}[]}
	 */
	CurrentPageContentSections = null;
	/**
	 * The current PageLayoutSectionsDefinition object for the currently-loaded page's selected layout.
	 * This can be null if the layout the page has chosen does not have a section definition - and thus
	 * no page sections.
	 * @type {{creationTimestamp: number, id: number, layoutFileName: string, pageLayoutSections: Object[]}|null}
	 */
	CurrentPageLayoutSectionsDefinition = null;
	ContentEditorContainer = document.querySelector("#content-editor-container");
	OptionBarViewPageButton = document.querySelector("#view-page-external-link-button");
	OptionBarSavePageButton = document.querySelector('button[form="page-edit-form"]');
	Loader = document.querySelector("#easy-editor-loading-container");
	PageBodyInstanceEditorContainer = document.querySelector("#easy-editor-single-instance-container");
	PageBodyTinyMCE;
	PageContentSectionsInstanceEditorContainer = document.querySelector("#easy-editor-sections-instance-container");
	PhoneNumbers;
	FaxNumbers;
	IsProcessing = false;
	SaveButton = document.querySelector('.btn-success.btn-sm[type="submit"][form="page-edit-form"]');

	/** @type {EasyEditor} */
	EasyEditorInstance;

	constructor() {
		this.Initialize();
	}

	async Initialize(){
		const [phoneNumbers, faxNumbers] = await this.GetPhoneAndFaxNumbers();
		this.PhoneNumbers = phoneNumbers;
		this.FaxNumbers = faxNumbers;

		// We will always have the main page body editor initialized
		this.PageBodyTinyMCE = await this.InitEditorForSelector("#content-editor-container");
		const pageId = this.GetPageIDFromURL();

		if (pageId === null){
			this.Loader.style.display = "none";
		}else{
			await this.LoadPage(pageId);
		}

		new EditorSidebar(pageId);

		// Handle saving
		this.SaveButton.addEventListener("click", () => {
			this.SavePage();
		});

		window.addEventListener("keydown", e => {
			if (e.ctrlKey && e.key === "s"){
				e.preventDefault();
				e.stopPropagation();
				this.SavePage();
			}
		});

		new BlogReelForm();
		new IPPReelForm();
		new RecentArticlesForm();
		new RecentProjectsForm();
	}

	async GetPhoneAndFaxNumbers(){
		const response = await fetch(`${Endpoints.easyEditor.getPhoneAndFaxNumbers}`, {
			cache:"no-cache",
			credentials:"same-origin",
			method:"get",
		});

		/** @type {{status: int, phoneNumbers: string[], faxNumbers}} */
		let data = await response.json();

		return [data.phoneNumbers, data.faxNumbers];
	}

	/**
	 * Loads the page given by the pageId into the editor. Will find out if sections should be shown
	 * or if only a main body editor should be shown.
	 * @param pageId
	 * @returns {Promise<void>}
	 * @constructor
	 */
	async LoadPage(pageId){
		this.PageContentSectionsInstanceEditorContainer.style.display = "none";
		this.PageBodyInstanceEditorContainer.style.display = "none";
		this.Loader.style.display = null;
		// Get new page data
		await this.FetchPageFromRemote(pageId);

		// Clear all section editors
		PageContentSectionEditor.clearAll();

		window.history.pushState({}, "", `/uplift/easy-editor/${pageId}`)

		this.OptionBarViewPageButton.setAttribute("href", this.CurrentPage.pageRoute);
		this.OptionBarViewPageButton.style.display = null;
		this.OptionBarSavePageButton.style.display = null;

		// Determine if this page needs multiple section editors, or to use the single-editor
		if (this.CurrentPageLayoutSectionsDefinition !== null){
			// Load the multi-section editor
			this.LoadSectionsEditors();
		}else{
			// Load the single-content editor
			this.LoadPageBodyEditor();
		}

		await Scheduler.wait(300);
		this.Loader.style.display = "none";
	}

	LoadSectionsEditors(){
		this.PageBodyInstanceEditorContainer.style.display = "none";
		this.PageContentSectionsInstanceEditorContainer.style.display = null;
		this.PageBodyTinyMCE.setContent("");

		// Load the editor components
		for (const layoutSection of this.CurrentPageLayoutSectionsDefinition.pageLayoutSections){
			const pageContentSectionThatMatchesLayoutSectionName = this.CurrentPageContentSections
				.find(contentSection => contentSection.sectionName.toLowerCase() === layoutSection.sectionName.toLowerCase());
			/** @var {{id: number, pageLayoutSectionDefinitionId: number, sectionName: string, creationTimestamp: number}} layoutSection */
			new PageContentSectionEditor(layoutSection, pageContentSectionThatMatchesLayoutSectionName ?? null)
				.build()
				.renderInto(this.PageContentSectionsInstanceEditorContainer);
		}
	}

	/**
	 * Shows and loads the content of the main page body editor
	 * @constructor
	 */
	LoadPageBodyEditor(){
		this.PageBodyInstanceEditorContainer.style.display = null;
		this.PageContentSectionsInstanceEditorContainer.style.display = "none";
		this.PageBodyTinyMCE.setContent(this.CurrentPage.pageBody);
	}

	/**
	 * Fetches the PageLayoutSectionsDefinition for the provided layoutName, if there is one.
	 * @param {string} layoutName E.g.: "General"
	 * @returns {Promise<{creationTimestamp: number, id: number, layoutFileName: string, pageLayoutSections: Object[]}>}
	 * @throws
	 */
	async GetSectionDefinitionForLayoutName(layoutName){
		const params = new URLSearchParams({layoutFileName: layoutName});
		const sectionsDefinitionResponse = await fetch(`/uplift/page-editor/page-layout-sections?${params}`);

		if (sectionsDefinitionResponse.status !== 200){
			throw "Internal server error when loading page layout section definition.";
		}

		/** @type {{status: number}} */
		const sectionsDefinitionApiResponse = await sectionsDefinitionResponse.json();
		if (sectionsDefinitionApiResponse.status === -1){
			/** @var {{status: number, error: string}} sectionsDefinitionApiResponse */
			throw sectionsDefinitionApiResponse.error;
		}

		/** @var {{status: number, definition: {creationTimestamp: number, id: number, layoutFileName: string, pageLayoutSections: Object[]}}} sectionsDefinitionApiResponse */

		return sectionsDefinitionApiResponse.definition;
	}

	async FetchPageFromRemote(pageId){
		const response = await fetch(`/uplift/page-editor/page/${pageId}`, {
			cache:"no-cache",
			credentials:"same-origin",
			method:"get",
		});

		/** @type {{status: int, error: ?string, page: Object, data: Object, breadcrumbs: Object[], pageContentSections: Object[]}} **/
		let data = await response.json();

		let pageLayoutSectionDefinition = null;
		try{
			pageLayoutSectionDefinition = await this.GetSectionDefinitionForLayoutName(data.page.pageLayout);
		}catch(ex){}

		this.CurrentPage = data.page;
		this.CurrentPageContentSections = data.pageContentSections;
		this.CurrentPageLayoutSectionsDefinition = pageLayoutSectionDefinition;
	}

	GetPageIDFromURL(){
		const currentPath = window.location.pathname;
		const matches = currentPath.match(/\/uplift\/easy-editor\/(\d+)$/);
		if (matches !== null && 1 in matches) {
			return parseInt(matches[1]);
		}

		return null;
	}

	GetPhoneNumbersShortcodeValue(index) {
		// We add 1 to the index since starting at 0 seemed like it might confuse users
		return `{{ phone-number which='${index + 1}' }}`;
	}

	GetFaxNumbersShortcodeValue(index) {
		// We add 1 to the index since starting at 0 seemed like it might confuse users
		return `{{ fax-number which='${index + 1}' }}`;
	}

	GetCompanyNameShortcodeValue() {
		return '{{ company-name }}';
	}

	GetCompanyStreetShortcodeValue() {
		return '{{ company-street }}';
	}

	GetCompanyPostalShortcodeValue() {
		return '{{ company-postal }}';
	}

	GetCompanyCityShortcodeValue() {
		return '{{ company-city }}';
	}

	GetCompanyStateShortcodeValue() {
		return '{{ company-state }}';
	}

	GetReviewsShortcodeValue(){
		return '{{ get-reviews columns="1" truncate-body="false" limit="100" offset="0" city="" state="" schema="true" show-empty-message="true" show-header="true" }}';
	}

	/**
	 * Initializes an instance of TinyMC for the provided selector
	 * @param selector
	 * @returns {Object}
	 * @constructor
	 */
	async InitEditorForSelector(selector){
		// Fetches fonts in the active theme
		const response = await fetch(`/uplift/theme-manager/get-fonts`, {
			cache:"no-cache",
			credentials:"same-origin",
			method:"get",
		});

		let data = await response.json();

		// These are the default fonts TinyMCE utilizes
		let defaultFonts = "Andale Mono=andale mono,times; Arial=arial,helvetica,sans-serif; Arial Black=arial black,avant garde; Book Antiqua=book antiqua,palatino; Comic Sans MS=comic sans ms,sans-serif; Courier New=courier new,courier; Georgia=georgia,palatino; Helvetica=helvetica; Impact=impact,chicago; Symbol=symbol; Tahoma=tahoma,arial,helvetica,sans-serif; Terminal=terminal,monaco; Times New Roman=times new roman,times; Trebuchet MS=trebuchet ms,geneva; Verdana=verdana,geneva; Webdings=webdings; Wingdings=wingdings,zapf dingbats;";

		// We'll populate this empty string with our custom fonts for TinyMCE
		let customFonts = "";

		// Loop through each font file
		data.fonts.forEach((font) => {
			let fontName = "";
			let lastDotIndex = font.lastIndexOf('.');

			if (lastDotIndex === -1) {
				fontName = font;
			} else {
				fontName = font.substring(0, lastDotIndex);
			}

			customFonts += fontName + "=" + fontName + ";";
		});

		const editorsCreated = await tinymce.init({
			selector: selector,
			width: '100%',
			promotion: false,
			statusbar: false,
			browser_spellcheck: true,
			plugins: 'table link lists',
			toolbar: 'undo redo | blocks | bold italic | alignleft aligncentre alignright alignjustify | indent outdent | bullist numlist',
			skin: 'tinymce-5',
			menu: {
				shortcodes: { title: 'Shortcodes', items: 'phoneNumbers faxNumbers companyName companyStreet companyPostal companyCity companyState blogReel ippReel recentArticles recentProjects reviews' }
			},
			menubar: 'edit view insert format shortcodes',
			content_css: '/uplift-assets/css/tinymce/default/content.min.css, /uplift/theme-manager/get-fonts/fonts.css',
			font_family_formats: defaultFonts + customFonts,
			setup: (editor) => {
				editor.ui.registry.addNestedMenuItem('phoneNumbers', {
					text: 'Phone Numbers',
					getSubmenuItems: () => {
						return this.PhoneNumbers.map((phoneNumber, index) => ({
							type: 'menuitem',
							text: phoneNumber,
							onAction: () => editor.insertContent(this.GetPhoneNumbersShortcodeValue(index))
						}));
					}
				});
				editor.ui.registry.addNestedMenuItem('faxNumbers', {
					text: 'Fax Numbers',
					getSubmenuItems: () => {
						return this.FaxNumbers.map((faxNumber, index) => ({
							type: 'menuitem',
							text: faxNumber,
							onAction: () => editor.insertContent(this.GetFaxNumbersShortcodeValue(index))
						}));
					}
				});

				const addMenuItem = (name, text, action) => {
					editor.ui.registry.addMenuItem(name, {
						text: text,
						onAction: action
					});
				};

				addMenuItem('companyName', 'Company Name', () => editor.insertContent(this.GetCompanyNameShortcodeValue()));
				addMenuItem('companyStreet', 'Company Street', () => editor.insertContent(this.GetCompanyStreetShortcodeValue()));
				addMenuItem('companyPostal', 'Company Postal', () => editor.insertContent(this.GetCompanyPostalShortcodeValue()));
				addMenuItem('companyCity', 'Company City', () => editor.insertContent(this.GetCompanyCityShortcodeValue()));
				addMenuItem('companyState', 'Company State', () => editor.insertContent(this.GetCompanyStateShortcodeValue()));
				addMenuItem('blogReel', 'Blog Reel', () => BlogReelModal.showModal());
				addMenuItem('ippReel', 'IPP Reel', () => IPPReelModal.showModal());
				addMenuItem('recentArticles', 'Recent Articles', () => RecentArticlesModal.showModal());
				addMenuItem('recentProjects', 'Recent Projects', () => RecentProjectsModal.showModal());
				addMenuItem('reviews', 'Reviews', () => editor.insertContent(this.GetReviewsShortcodeValue()));

				editor.addShortcut('ctrl+s', "Save page", () => {
					editor.save();
					this.SavePage();
				});
			}
		});

		return editorsCreated[0];
	}

	async SavePage() {
		if (this.IsProcessing) {
			return;
		}

		this.IsProcessing = true;
		this.SaveButton.classList.add("disabled");

		const formData = new FormData();

		// Send the main content for the pageBody, always, even if blank for page sections
		formData.set("content", this.PageBodyTinyMCE.getContent());

		// If there are sections, package each section's content into the page-content-sections JSON stringified content
		const pageContentSectionsPayload = [];
		if (this.CurrentPageLayoutSectionsDefinition !== null){
			// Ids will be null if they need to be created
			for (const sectionEditor of PageContentSectionEditor.cache){
				pageContentSectionsPayload.push(sectionEditor.pageContentSectionDto);
			}
		}
		formData.append("page-content-sections", JSON.stringify(pageContentSectionsPayload));

		const response = await fetch(`${Endpoints.easyEditor.savePage}/${this.CurrentPage.id}`, {
			body:formData,
			cache:"no-cache",
			credentials:"same-origin",
			method:"POST"
		});

		if(response.status !== 200){
			console.error("An error occurred while saving this page. Please contact a developer.");
		} else {
			/** @var {{status: number, updatedPageContentSections: {id: number, content: string, pageId: number, sectionName: string}[]|null}} data */
			const apiData = await response.json();

			// We need to match the sectionNames in the return payload to sectionNames we have in our editor and assign
			// ids to them if they were null.
			if (apiData.updatedPageContentSections !== null){
				if (apiData.updatedPageContentSections.length > 0) {
					for (const savedPageContentSection of apiData.updatedPageContentSections) {
						/** @type {PageContentSectionEditor|null} */
						let pageContentEditorWithMatchingSectionName = null;
						for (const component of PageContentSectionEditor.cache) {
							if (component.pageContentSectionDto.sectionName === savedPageContentSection.sectionName) {
								pageContentEditorWithMatchingSectionName = component;
								break;
							}
						}

						if (pageContentEditorWithMatchingSectionName !== null) {
							// Update the pageContentSectionDto
							pageContentEditorWithMatchingSectionName.pageContentSectionDto.id = savedPageContentSection.id;
						}
					}
				}
			}
		}

		this.IsProcessing = false;
		this.SaveButton.classList.remove("disabled");
	}
}

export default new EasyEditorManager();