import EasyEditorManager from "../EasyEditorManager.js";

export class PageContentSectionEditor{
	/**
	 * A cache of currently rendered PageContentSectionEditor
	 * @type {PageContentSectionEditor[]}
	 */
	static cache = [];
	static nextId = 1;

	/**
	 * Clears all the currently rendered PageContentSectionEditor
	 */
	static clearAll(){
		for (const sectionEditor of PageContentSectionEditor.cache){
			sectionEditor.dom.remove();
		}

		PageContentSectionEditor.cache = [];
	}

	currentId = PageContentSectionEditor.nextId++;
	/**
	 * The HTML element this component is represented by
	 * @type {HTMLDivElement}
	 */
	dom;
	/**
	 * The page layout section objects that this page content section should reflect. Can't be null
	 * @type {{id: number, pageLayoutSectionDefinitionId: number, sectionName: string, creationTimestamp: number}}
	 */
	pageLayoutSectionDto;
	/**
	 * The object that represents this individual section for this particular page. Can be null
	 * if it has never been saved yet.
	 * @type {{id: number, pageId: number, sectionName: string, content: string} | null}
	 */
	pageContentSectionDto;
	/**
	 * The instance of TinyMCE that the editor for this component is using
	 * @type {Object}
	 */
	tinyMCE;

	/**
	 * @param {{id: number, creationTimestamp: number, pageLayoutSectionDefinitionId: number, sectionName: string}} pageLayoutSectionDto
	 * @param {{id: number|null, pageId: number, sectionName: string, content: string | null}|null} pageContentSectionDto This can be null
	 */
	constructor(pageLayoutSectionDto, pageContentSectionDto) {
		this.pageLayoutSectionDto = pageLayoutSectionDto;
		this.pageContentSectionDto = pageContentSectionDto;
		this.constructor.cache.push(this);

		// Create a PageContentSection dto if the one we got was null. Just leave the id null for now,
		// as the page saving logic will populate it once the PageContentSection is saved and created
		// on the backend
		if (pageContentSectionDto === null){
			this.pageContentSectionDto = {
				id: null,
				pageId: EasyEditorManager.CurrentPage.id,
				sectionName: this.pageLayoutSectionDto.sectionName,
				content: ""
			};
		}
	}

	build(){
		const template = document.createElement("div");
		template.id = `easy-editor-content-section-${this.currentId}`;
		template.classList.add("easy-editor-content-section-editor-component");
		template.innerHTML = `
			<div class="title-container">
				<strong class="me-2">Section: </strong><span>${this.pageLayoutSectionDto.sectionName}</span>
			</div>
			<div class="editor-container">
				<textarea class="tiny-mce-editor-container" style="height:300px"></textarea>
			</div>
		`;

		this.dom = template;
		return this;
	}

	async renderInto(container){
		container.append(this.dom);
		this.tinyMCE = await EasyEditorManager.InitEditorForSelector(`#easy-editor-content-section-${this.currentId} .tiny-mce-editor-container`);
		this.tinyMCE.setContent(this.pageContentSectionDto.content);
		this.tinyMCE.on("input", () => {
			this.pageContentSectionDto.content = this.tinyMCE.getContent();
		});
	}
}