import DOMHelper from "../DOMHelper.js";
import FileEditorStates from "./FileEditorStates.js";
import EditorFileTab from "./EditorFileTab.js";
import AceEditorExtension from "../AceEditorExtension.js";
import ConfirmModal from "../../utils/ConfirmModal.js";
import SaveFileButton from "./components/SaveFileButton.js";
import FileEditor from "./FileEditor.js";
import {FilePathUtility} from "./FilePathUtility.js";

const AceEditSession = ace.require("ace/edit_session").EditSession;
const AceUndoManager = ace.require("ace/undomanager").UndoManager;

class FileEditorWrapper{
	/**
	 * @type {HTMLButtonElement}
	 */
	layoutSectionsButton = document.querySelector("#layout-sections-button");

	constructor(){
		this.openFileAceSessions = {}; // The object key/property is the full file path
		this.editorContainer = document.querySelector("#content-editor-container");
		this.fileEditor = DOMHelper.getThemeManagerAceEditors().file;
	}

	/**
	* Fetches the contents of the currently active editor
	* @returns {string}
	*/
	getCurrentEditorContents(){
		return this.fileEditor.getSession().getValue();
	}

	/**
	* Inserts text to wherever the cursor currently is
	* in the current editor.
	* @param {string} text
	*/
	insertTextAtCursor(text){
		const editor = DOMHelper.getThemeManagerAceEditors().file;
		if (editor !== null){
			editor.session.replace(editor.session.getSelection().getRange(), text);
		}
	}

	/**
	* Changes the focus to be in the current active editor
	*/
	focusEditor(){
		const editor = DOMHelper.getThemeManagerAceEditors().file;
		if (editor !== null){
			editor.focus();
		}
	}

	/**
	 * Opens a new file into the editor. If there are unsaved changes, prompts the
	 * user first.
	 * @param {string} filePath
	 * @param {string} fileName
	 * @return {Promise} If the open succeeded or not
	 */
	async openFile(filePath, fileName){
		// Hide the layout sections button
		this.layoutSectionsButton.style.display = "none";
		return new Promise(async resolve => {
			// Find the ace session for this file tab, if it exists
			let aceSession = null;
			let editorFileTab = EditorFileTab.getByFilePath(filePath);
			if (editorFileTab === null) {
				// Create a session
				aceSession = new AceEditSession("");
				aceSession.setUndoManager(new AceUndoManager());
				AceEditorExtension.setSessionDefaults(aceSession, null);
			} else {
				aceSession = editorFileTab.aceSession;
			}

			// Set the current file path
			FileEditorStates.currentFilePath = filePath;

			const params = new URLSearchParams();
			params.set("file-path", filePath);

			const response = await fetch(`/uplift/theme-manager/file-contents?${params.toString()}`, {
				credentials: "same-origin",
				cache: "no-cache",
			});

			/** @type {{status: int, fsFile: Object, contents: string, error: ?string}} */
			const data = await response.json();
			if (data.status === 1) {

				/** @type {{fileName: string, fileSize: number, fullFilePath: string, fileExtension: string}} */
				const fsFile = data.fsFile;
				const fileExtension = fsFile.fileExtension;
				switch (fileExtension) {
					case "json":
						aceSession.setMode("ace/mode/json");
						break;
					case "md":
						aceSession.setMode("ace/mode/markdown");
						break;
					case "cs":
						aceSession.setMode("ace/mode/coffeescript");
						break;
					case "svg":
						aceSession.setMode("ace/mode/svg");
						break;
					case "html":
						aceSession.setMode("ace/mode/html");
						break;
					case "php":
						aceSession.setMode("ace/mode/php");
						break;
					case "js":
						aceSession.setMode("ace/mode/javascript");
						break;
					case "css":
						aceSession.setMode("ace/mode/css");
						break;
					case "scss":
						aceSession.setMode("ace/mode/scss");
						break;
					default:
						aceSession.setMode("ace/mode/plain_text");
				}

				aceSession.setValue(data.contents, 1);

				if (editorFileTab === null) {
					editorFileTab = new EditorFileTab(aceSession, filePath, fileName);
				}

				this.editorContainer.style.display = "flex";
				editorFileTab.select();

				if (!SaveFileButton.isShown()){
					SaveFileButton.show();
				}

				if (FilePathUtility.isFileLayoutFile(filePath)){
					// Show the layout sections button
					this.layoutSectionsButton.style.display = null;

					// Load the layout sections for the selected file path
					FileEditor.layoutSectionsModal.modal.querySelector(".current-layout-file-name").textContent = FilePathUtility.getFileName(filePath);
					FileEditor.layoutSectionsModal.loadAndRenderFileSections(filePath)
				}

				resolve(true);
			} else if (data.status === -1) {
				const warningModal = new ConfirmModal("warning-failed-file-open-modal");
				warningModal.hideSecondaryButton();
				warningModal.setTitle("File failed to open");
				warningModal.setContent(data.error);
				warningModal.setConfirmButtonText("Okay");
				warningModal.showModal();
				await warningModal.actionTaken();
				warningModal.cleanup();
				resolve(false);
			}
		});
	}
}

export default (new FileEditorWrapper);
