import FileEditorStates from "./FileEditorStates.js";
import FileEditorWrapper from "./FileEditorWrapper.js";
import EditorEndpoints from "../EditorEndpoints.js";
import Endpoints from "../../Endpoints.js";
import DeleteDirectoryMechanic from "./mechanics/DeleteDirectoryMechanic.js";
import CompileSCSSMechanics from "./mechanics/CompileSCSSMechanics.js";
import DeleteFileMechanic from "./mechanics/DeleteFileMechanic.js";
import NewDirectoryModal from "./modals/NewDirectoryModal.js";
import RenameDirectoryModal from "./modals/RenameDirectoryModal.js";
import NewFileModal from "./modals/NewFileModal.js";
import RenameFileModal from "./modals/RenameFileModal.js";
import ContextMenu from "../../utils/ContextMenu.js";
import Collapse from "./../../_external/bootstrap/collapse.js";
import SidebarLoadingOverlay from "./components/SidebarLoadingOverlay.js";
import Scheduler from "../../utils/Scheduler.js";
import Utils from "./Utils.js";

/**
* Handles the page editor page loading
* and rendering them into the sidebar
*/
class SidebarFileLoader{
	constructor(){
		this.directoryListingContainerDOM = document.querySelector("#content-editor-page-navigation-column");
		this.directoryListingDOM = document.querySelector("#directory-listing");
		this.backButton = document.querySelector("#back-button-directory-listing");
		this.isLoadingDirectory = false;

		let directoryToLoad = FileEditorStates.currentDirectory;
		if (directoryToLoad === "/"){
			directoryToLoad = FileEditorStates.rootThemeDirectory;
		}

		this.loadDirectory(directoryToLoad);
		this.backButton.addEventListener("click", () => {
			this.onBackButtonClicked();
		});

		this.directoryListingContainerDOM.addEventListener("contextmenu", e => {
			e.preventDefault();
			e.stopPropagation();
			this.onSidebarContextMenu(e);
		});
	}

	/**
	* When a context menu is requested on the sidebar
	* @param {Event} e
	*/
	onSidebarContextMenu(e){

		// Check for an existing file editor context menu
		if (FileEditorStates.currentContextMenu){
			FileEditorStates.currentContextMenu.cleanup();
		}

		const menu = new ContextMenu(e.pageX, e.pageY);
		FileEditorStates.currentContextMenu = menu;
		const newFolderButton = menu.addButton(`
			<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-folder-plus" viewBox="0 0 16 16">
				<path d="M.5 3l.04.87a1.99 1.99 0 0 0-.342 1.311l.637 7A2 2 0 0 0 2.826 14H9v-1H2.826a1 1 0 0 1-.995-.91l-.637-7A1 1 0 0 1 2.19 4h11.62a1 1 0 0 1 .996 1.09L14.54 8h1.005l.256-2.819A2 2 0 0 0 13.81 3H9.828a2 2 0 0 1-1.414-.586l-.828-.828A2 2 0 0 0 6.172 1H2.5a2 2 0 0 0-2 2zm5.672-1a1 1 0 0 1 .707.293L7.586 3H2.19c-.24 0-.47.042-.684.12L1.5 2.98a1 1 0 0 1 1-.98h3.672z"/>
				<path d="M13.5 10a.5.5 0 0 1 .5.5V12h1.5a.5.5 0 0 1 0 1H14v1.5a.5.5 0 0 1-1 0V13h-1.5a.5.5 0 0 1 0-1H13v-1.5a.5.5 0 0 1 .5-.5z"/>
			</svg>
			<span>New folder</span>
		`);
		const newFileButton = menu.addButton(`
			<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-file-earmark-plus" viewBox="0 0 16 16">
				<path d="M8 6.5a.5.5 0 0 1 .5.5v1.5H10a.5.5 0 0 1 0 1H8.5V11a.5.5 0 0 1-1 0V9.5H6a.5.5 0 0 1 0-1h1.5V7a.5.5 0 0 1 .5-.5z"/>
				<path d="M14 4.5V14a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V2a2 2 0 0 1 2-2h5.5L14 4.5zm-3 0A1.5 1.5 0 0 1 9.5 3V1H4a1 1 0 0 0-1 1v12a1 1 0 0 0 1 1h8a1 1 0 0 0 1-1V4.5h-2z"/>
			</svg>
			<span>New file</span>
		`);

		newFileButton.addEventListener("click", () => {
			NewFileModal.showModal();
			menu.cleanup();
		});

		newFolderButton.addEventListener("click", () => {
			NewDirectoryModal.showModal();
			menu.cleanup();
		});

		// Render the menu
		menu.render();
	}

	/**
	* When a context menu is requested on a directory button
	* @param {MouseEvent} e
	* @param {HTMLDivElement} directoryContainerNode The <div> containing the button
	*/
	onDirectoryContextMenu(e, directoryContainerNode){

		// Check for an existing file editor context menu
		if (FileEditorStates.currentContextMenu){
			FileEditorStates.currentContextMenu.cleanup();
		}

		const menu = new ContextMenu(e.pageX, e.pageY);
		FileEditorStates.currentContextMenu = menu;

		const renameDirectoryButton = menu.addButton(`
			<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-input-cursor-text" viewBox="0 0 16 16">
				<path fill-rule="evenodd" d="M5 2a.5.5 0 0 1 .5-.5c.862 0 1.573.287 2.06.566.174.099.321.198.44.286.119-.088.266-.187.44-.286A4.165 4.165 0 0 1 10.5 1.5a.5.5 0 0 1 0 1c-.638 0-1.177.213-1.564.434a3.49 3.49 0 0 0-.436.294V7.5H9a.5.5 0 0 1 0 1h-.5v4.272c.1.08.248.187.436.294.387.221.926.434 1.564.434a.5.5 0 0 1 0 1 4.165 4.165 0 0 1-2.06-.566A4.561 4.561 0 0 1 8 13.65a4.561 4.561 0 0 1-.44.285 4.165 4.165 0 0 1-2.06.566.5.5 0 0 1 0-1c.638 0 1.177-.213 1.564-.434.188-.107.335-.214.436-.294V8.5H7a.5.5 0 0 1 0-1h.5V3.228a3.49 3.49 0 0 0-.436-.294A3.166 3.166 0 0 0 5.5 2.5.5.5 0 0 1 5 2z"/>
				<path d="M10 5h4a1 1 0 0 1 1 1v4a1 1 0 0 1-1 1h-4v1h4a2 2 0 0 0 2-2V6a2 2 0 0 0-2-2h-4v1zM6 5V4H2a2 2 0 0 0-2 2v4a2 2 0 0 0 2 2h4v-1H2a1 1 0 0 1-1-1V6a1 1 0 0 1 1-1h4z"/>
			</svg>
			<span>Rename</span>
		`);

		const copyURIButton = menu.addButton(`
			<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-link-45deg" viewBox="0 0 16 16">
				<path d="M4.715 6.542L3.343 7.914a3 3 0 1 0 4.243 4.243l1.828-1.829A3 3 0 0 0 8.586 5.5L8 6.086a1.001 1.001 0 0 0-.154.199 2 2 0 0 1 .861 3.337L6.88 11.45a2 2 0 1 1-2.83-2.83l.793-.792a4.018 4.018 0 0 1-.128-1.287z"/>
				<path d="M6.586 4.672A3 3 0 0 0 7.414 9.5l.775-.776a2 2 0 0 1-.896-3.346L9.12 3.55a2 2 0 0 1 2.83 2.83l-.793.792c.112.42.155.855.128 1.287l1.372-1.372a3 3 0 0 0-4.243-4.243L6.586 4.672z"/>
			</svg>
			<span>Copy URI</span>
		`);


		menu.addSeparator();

		const deleteDirectoryButton = menu.addButton(`
			<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="red" class="bi bi-x-octagon-fill" viewBox="0 0 16 16">
				<path d="M11.46.146A.5.5 0 0 0 11.107 0H4.893a.5.5 0 0 0-.353.146L.146 4.54A.5.5 0 0 0 0 4.893v6.214a.5.5 0 0 0 .146.353l4.394 4.394a.5.5 0 0 0 .353.146h6.214a.5.5 0 0 0 .353-.146l4.394-4.394a.5.5 0 0 0 .146-.353V4.893a.5.5 0 0 0-.146-.353L11.46.146zm-6.106 4.5L8 7.293l2.646-2.647a.5.5 0 0 1 .708.708L8.707 8l2.647 2.646a.5.5 0 0 1-.708.708L8 8.707l-2.646 2.647a.5.5 0 0 1-.708-.708L7.293 8 4.646 5.354a.5.5 0 1 1 .708-.708z"/>
			</svg>
			<span>Delete folder</span>
		`);

		renameDirectoryButton.addEventListener("click", async () => {
			const directoryLocation = directoryContainerNode.getAttribute("location");
			const folderName = directoryContainerNode.querySelector(".directory-name").textContent;
			document.querySelector("#rename-directory-old-path-input").value = directoryLocation;
			document.querySelector("#rename-directory-new-name-input").value = folderName;
			RenameDirectoryModal.showModal();
			menu.cleanup();
		});

		deleteDirectoryButton.addEventListener("click", async () => {
			DeleteDirectoryMechanic.deleteDirectory(directoryContainerNode.getAttribute("location"), directoryContainerNode);
			menu.cleanup();
		});

		copyURIButton.addEventListener("click", async () => {
			const result = await navigator.permissions.query({name: "clipboard-write"});
			if (result.state === "granted" || result.state === "prompt"){
				const fullSystemFilePath = directoryContainerNode.getAttribute("location");
				const relativePathFromAppRoot = fullSystemFilePath.replace(FileEditorStates.applicationRoot, "");
				const browserNavigatableURI = relativePathFromAppRoot.replace(new RegExp(/\\/g), "/");
				await navigator.clipboard.writeText(`${browserNavigatableURI}`);
			}
			menu.cleanup();
		});

		// Render the menu
		menu.render();
	}

	/**
	* When a context menu is requested on a file button
	* @param {MouseEvent} e
	* @param {HTMLDivElement} fileContainerNode The <div> containing the button
	* @param {string} fullFilePath The file's full path location
	*/
	onFileContextMenu(e, fileContainerNode, fullFilePath){

		// Store the file name because some buttons
		// could be available for specific file
		// extensions
		const fileName = fileContainerNode.querySelector(".file-name").textContent;
		const filePath = fileContainerNode.getAttribute("location");
		const fileExtensionRegEx = /\.([^\.]+)$/;
		const fileExtensionMatchGroup = fileName.match(fileExtensionRegEx);
		let fileExtension = null;

		if (fileExtensionMatchGroup !== null){
			fileExtension = fileExtensionMatchGroup[1];
		}

		// Check for an existing file editor context menu
		if (FileEditorStates.currentContextMenu){
			FileEditorStates.currentContextMenu.cleanup();
		}

		const menu = new ContextMenu(e.pageX, e.pageY);
		FileEditorStates.currentContextMenu = menu;

		const renameFileButton = menu.addButton(`
			<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-input-cursor-text" viewBox="0 0 16 16">
				<path fill-rule="evenodd" d="M5 2a.5.5 0 0 1 .5-.5c.862 0 1.573.287 2.06.566.174.099.321.198.44.286.119-.088.266-.187.44-.286A4.165 4.165 0 0 1 10.5 1.5a.5.5 0 0 1 0 1c-.638 0-1.177.213-1.564.434a3.49 3.49 0 0 0-.436.294V7.5H9a.5.5 0 0 1 0 1h-.5v4.272c.1.08.248.187.436.294.387.221.926.434 1.564.434a.5.5 0 0 1 0 1 4.165 4.165 0 0 1-2.06-.566A4.561 4.561 0 0 1 8 13.65a4.561 4.561 0 0 1-.44.285 4.165 4.165 0 0 1-2.06.566.5.5 0 0 1 0-1c.638 0 1.177-.213 1.564-.434.188-.107.335-.214.436-.294V8.5H7a.5.5 0 0 1 0-1h.5V3.228a3.49 3.49 0 0 0-.436-.294A3.166 3.166 0 0 0 5.5 2.5.5.5 0 0 1 5 2z"/>
				<path d="M10 5h4a1 1 0 0 1 1 1v4a1 1 0 0 1-1 1h-4v1h4a2 2 0 0 0 2-2V6a2 2 0 0 0-2-2h-4v1zM6 5V4H2a2 2 0 0 0-2 2v4a2 2 0 0 0 2 2h4v-1H2a1 1 0 0 1-1-1V6a1 1 0 0 1 1-1h4z"/>
			</svg>
			<span>Rename</span>
		`);

		const copyURIButton = menu.addButton(`
			<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-link-45deg" viewBox="0 0 16 16">
				<path d="M4.715 6.542L3.343 7.914a3 3 0 1 0 4.243 4.243l1.828-1.829A3 3 0 0 0 8.586 5.5L8 6.086a1.001 1.001 0 0 0-.154.199 2 2 0 0 1 .861 3.337L6.88 11.45a2 2 0 1 1-2.83-2.83l.793-.792a4.018 4.018 0 0 1-.128-1.287z"/>
				<path d="M6.586 4.672A3 3 0 0 0 7.414 9.5l.775-.776a2 2 0 0 1-.896-3.346L9.12 3.55a2 2 0 0 1 2.83 2.83l-.793.792c.112.42.155.855.128 1.287l1.372-1.372a3 3 0 0 0-4.243-4.243L6.586 4.672z"/>
			</svg>
			<span>Copy URI</span>
		`);

		/**
		* Show compile SCSS button
		*/
		if (fileExtension === "scss"){
			const compileSCSSButton = menu.addButton(`
				<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-braces" viewBox="0 0 16 16">
					<path d="M2.114 8.063V7.9c1.005-.102 1.497-.615 1.497-1.6V4.503c0-1.094.39-1.538 1.354-1.538h.273V2h-.376C3.25 2 2.49 2.759 2.49 4.352v1.524c0 1.094-.376 1.456-1.49 1.456v1.299c1.114 0 1.49.362 1.49 1.456v1.524c0 1.593.759 2.352 2.372 2.352h.376v-.964h-.273c-.964 0-1.354-.444-1.354-1.538V9.663c0-.984-.492-1.497-1.497-1.6zM13.886 7.9v.163c-1.005.103-1.497.616-1.497 1.6v1.798c0 1.094-.39 1.538-1.354 1.538h-.273v.964h.376c1.613 0 2.372-.759 2.372-2.352v-1.524c0-1.094.376-1.456 1.49-1.456V7.332c-1.114 0-1.49-.362-1.49-1.456V4.352C13.51 2.759 12.75 2 11.138 2h-.376v.964h.273c.964 0 1.354.444 1.354 1.538V6.3c0 .984.492 1.497 1.497 1.6z"/>
				</svg>
				<span>Compile SCSS</span>
			`);

			compileSCSSButton.addEventListener("click", () => {
				CompileSCSSMechanics.compileFile(filePath);
				menu.cleanup();
			});
		}

		const newFolderButton = menu.addButton(`
			<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-folder-plus" viewBox="0 0 16 16">
				<path d="M.5 3l.04.87a1.99 1.99 0 0 0-.342 1.311l.637 7A2 2 0 0 0 2.826 14H9v-1H2.826a1 1 0 0 1-.995-.91l-.637-7A1 1 0 0 1 2.19 4h11.62a1 1 0 0 1 .996 1.09L14.54 8h1.005l.256-2.819A2 2 0 0 0 13.81 3H9.828a2 2 0 0 1-1.414-.586l-.828-.828A2 2 0 0 0 6.172 1H2.5a2 2 0 0 0-2 2zm5.672-1a1 1 0 0 1 .707.293L7.586 3H2.19c-.24 0-.47.042-.684.12L1.5 2.98a1 1 0 0 1 1-.98h3.672z"/>
				<path d="M13.5 10a.5.5 0 0 1 .5.5V12h1.5a.5.5 0 0 1 0 1H14v1.5a.5.5 0 0 1-1 0V13h-1.5a.5.5 0 0 1 0-1H13v-1.5a.5.5 0 0 1 .5-.5z"/>
			</svg>
			<span>New folder</span>
		`);

		const newFileButton = menu.addButton(`
			<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-file-earmark-plus" viewBox="0 0 16 16">
				<path d="M8 6.5a.5.5 0 0 1 .5.5v1.5H10a.5.5 0 0 1 0 1H8.5V11a.5.5 0 0 1-1 0V9.5H6a.5.5 0 0 1 0-1h1.5V7a.5.5 0 0 1 .5-.5z"/>
				<path d="M14 4.5V14a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V2a2 2 0 0 1 2-2h5.5L14 4.5zm-3 0A1.5 1.5 0 0 1 9.5 3V1H4a1 1 0 0 0-1 1v12a1 1 0 0 0 1 1h8a1 1 0 0 0 1-1V4.5h-2z"/>
			</svg>
			<span>New file</span>
		`);

		menu.addSeparator();

		const deleteFileButton = menu.addButton(`
			<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="red" class="bi bi-x-octagon-fill" viewBox="0 0 16 16">
				<path d="M11.46.146A.5.5 0 0 0 11.107 0H4.893a.5.5 0 0 0-.353.146L.146 4.54A.5.5 0 0 0 0 4.893v6.214a.5.5 0 0 0 .146.353l4.394 4.394a.5.5 0 0 0 .353.146h6.214a.5.5 0 0 0 .353-.146l4.394-4.394a.5.5 0 0 0 .146-.353V4.893a.5.5 0 0 0-.146-.353L11.46.146zm-6.106 4.5L8 7.293l2.646-2.647a.5.5 0 0 1 .708.708L8.707 8l2.647 2.646a.5.5 0 0 1-.708.708L8 8.707l-2.646 2.647a.5.5 0 0 1-.708-.708L7.293 8 4.646 5.354a.5.5 0 1 1 .708-.708z"/>
			</svg>
			<span>Delete file</span>
		`);

		renameFileButton.addEventListener("click", async () => {
			const fileName = fileContainerNode.querySelector(".file-name").textContent;
			document.querySelector("#rename-file-path").value = fullFilePath;
			document.querySelector("#rename-file-new-name-input").value = fileName;
			RenameFileModal.showModal();
			menu.cleanup();
		});

		copyURIButton.addEventListener("click", async () => {
			const result = await navigator.permissions.query({name: "clipboard-write"});
			if (result.state === "granted" || result.state === "prompt"){
				const fullSystemFilePath = fileContainerNode.getAttribute("location");
				const relativePathFromAppRoot = fullSystemFilePath.replace(FileEditorStates.applicationRoot, "");
				const browserNavigatableURI = relativePathFromAppRoot.replace(new RegExp(/\\/g), "/");
				await navigator.clipboard.writeText(`${browserNavigatableURI}`);
			}
			menu.cleanup();
		});

		deleteFileButton.addEventListener("click", () => {
			DeleteFileMechanic.deleteFile(filePath, fileContainerNode);
			menu.cleanup();
		});

		newFileButton.addEventListener("click", () => {
			NewFileModal.showModal();
			menu.cleanup();
		});

		newFolderButton.addEventListener("click", () => {
			NewDirectoryModal.showModal();
			menu.cleanup();
		});

		// Render the menu
		menu.render();
	}

	/**
	* When the directory navigation back button is clicked
	*/
	onBackButtonClicked(){
		const currentHistory = FileEditorStates.directoryLocationHistory;
		if (this.isLoadingDirectory === false){
			if (currentHistory.length > 0){
				// Pop the latest (which is the current) location
				FileEditorStates.directoryLocationHistory.pop();

				if (FileEditorStates.directoryLocationHistory.length === 0){
					this.backButton.classList.remove("is-active");
				}

				// Go to the now, new latest location
				let nextLocation = FileEditorStates.directoryLocationHistory[FileEditorStates.directoryLocationHistory.length - 1];
				if (nextLocation === undefined){
					// The next location is the root directory
					nextLocation = FileEditorStates.rootThemeDirectory;
				}

				FileEditorStates.currentDirectory = Utils.getThemeFileRelativeLocationFromFullPath(nextLocation);
				this.loadDirectory(nextLocation);
			}
		}
	}

	/**
	* Fetches and renders directory contents
	* @param {string} directory
	*/
	async loadDirectory(directory){

		if (this.isLoadingDirectory){
			return;
		}

		// Clear the filter input
		document.querySelector("#directory-filter").value = "";

		// Fetch the relative directory to show the user and fetch from the API
		const relativePathFromThemeRoot = Utils.getThemeFileRelativeLocationFromFullPath(directory);

		if (relativePathFromThemeRoot !== ""){
			document.querySelector("#directory-location-input").value = relativePathFromThemeRoot;
		}else{
			document.querySelector("#directory-location-input").value = "/";
		}

		// Switch flag
		this.isLoadingDirectory = true;
		SidebarLoadingOverlay.show();

		// Clear the existing entries
		this.directoryListingDOM.innerHTML = "";

		const searchParams = new URLSearchParams();
		searchParams.append("directory-requested", directory);

		const endpoint = `/uplift/theme-manager/list-directory?${searchParams.toString()}`;
		const response = await fetch(endpoint, {
			cache:"no-cache",
			credentials:"same-origin"
		});

		// Minimum wait time
		await Scheduler.wait(250);

		/** @type {{status: int, error: ?string, fsDirectory: Object, applicationRoot: string}} */
 		const data = await response.json();
		this.isLoadingDirectory = false;
		if (data.status === 1){
			// Set the application root from the response
			FileEditorStates.applicationRoot = data.applicationRoot;

			/** @type {{childDirectories: [], childFiles:[], directoryName: string, fullDirectoryPath: string }} */
			const contents = data.fsDirectory;
			const directories = contents.childDirectories;
			const files = contents.childFiles;

			for (/** @type {{directoryName: string, fullDirectoryPath: string}} */ const fsDirectory of directories){
				this.directoryListingDOM.append(
					this.getDirectoryDOM(fsDirectory.directoryName, fsDirectory.fullDirectoryPath)
				);
			}

			for (/** @type {{fileExtension: string, fullFilePath: string, fileName: string}} */ const fsFile of files){
				this.directoryListingDOM.append(
					this.getFileDOM(fsFile.fileName, fsFile.fullFilePath)
				);
			}

			SidebarLoadingOverlay.hide();
		}else if (data.status === -1){
			alert(data.error);
		}
	}

	/**
	 * @param {string} name
	 * @param {string} fullPath
	 * @returns {HTMLDivElement}
	 */
	getDirectoryDOM(name, fullPath){
		const template = document.createElement("div");
		template.classList.add("theme-manager-fm-entry");
		template.classList.add("theme-manager-directory-button-container");
		template.setAttribute("listing-type", "directory");
		template.setAttribute("location", fullPath);
		template.innerHTML = `
			<button type="button">
				<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-folder-fill" viewBox="0 0 16 16">
					<path d="M9.828 3h3.982a2 2 0 0 1 1.992 2.181l-.637 7A2 2 0 0 1 13.174 14H2.826a2 2 0 0 1-1.991-1.819l-.637-7a1.99 1.99 0 0 1 .342-1.31L.5 3a2 2 0 0 1 2-2h3.672a2 2 0 0 1 1.414.586l.828.828A2 2 0 0 0 9.828 3zm-8.322.12C1.72 3.042 1.95 3 2.19 3h5.396l-.707-.707A1 1 0 0 0 6.172 2H2.5a1 1 0 0 0-1 .981l.006.139z"/>
				</svg>
				<span class="directory-name item-label">${name}</span>
			</button>
		`;

		const button = template.querySelector("button");

		// Double click
		let lastClickTime = 0;
		button.addEventListener("click", () => {

			if (!this.backButton.classList.contains("is-active")){
				this.backButton.classList.add("is-active");
			}

			const now = (new Date()).getTime();
			if (now - lastClickTime <= 350){
				FileEditorStates.directoryLocationHistory.push(fullPath);
				FileEditorStates.currentDirectory = Utils.getThemeFileRelativeLocationFromFullPath(fullPath);
				this.loadDirectory(fullPath);
			}

			lastClickTime = now;
		});

		// Context menu
		button.addEventListener("contextmenu", e => {
			e.preventDefault();
			e.stopPropagation();
			this.onDirectoryContextMenu(e, template);
		});

		return template;
	}

	/**
	* @returns {HTMLDivElement}
	*/
	getFileDOM(name, fullPath){
		const template = document.createElement("div");
		template.classList.add("theme-manager-fm-entry");
		template.classList.add("theme-manager-file-button-container");
		template.setAttribute("listing-type", "file");
		template.setAttribute("location", fullPath);
		template.innerHTML = `
			<button type="button">
				<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-file-text" viewBox="0 0 16 16">
					<path d="M5 4a.5.5 0 0 0 0 1h6a.5.5 0 0 0 0-1H5zm-.5 2.5A.5.5 0 0 1 5 6h6a.5.5 0 0 1 0 1H5a.5.5 0 0 1-.5-.5zM5 8a.5.5 0 0 0 0 1h6a.5.5 0 0 0 0-1H5zm0 2a.5.5 0 0 0 0 1h3a.5.5 0 0 0 0-1H5z"/>
					<path d="M2 2a2 2 0 0 1 2-2h8a2 2 0 0 1 2 2v12a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V2zm10-1H4a1 1 0 0 0-1 1v12a1 1 0 0 0 1 1h8a1 1 0 0 0 1-1V2a1 1 0 0 0-1-1z"/>
				</svg>
				<span class="file-name item-label">${name}</span>
			</button>
		`;

		const button = template.querySelector("button");

		// Double click
		let lastClickTime = 0;
		button.addEventListener("click", async () => {

			const now = (new Date()).getTime();
			if (now - lastClickTime <= 350){
				// Open the file editor for this file

				// If there is a currently selected button, deselect it
				const currentlySelectedButton = document.querySelector(".fm-editor-selected-file-button");
				if (currentlySelectedButton){
					currentlySelectedButton.classList.remove("fm-editor-selected-file-button");
				}

				button.classList.add("fm-editor-selected-file-button");
				const didOpen = await FileEditorWrapper.openFile(fullPath, name);
				if (!didOpen){
					button.classList.remove("fm-editor-selected-file-button");
				}
			}

			lastClickTime = now;
		});

		// Context menu
		button.addEventListener("contextmenu", e => {
			e.preventDefault();
			e.stopPropagation();
			this.onFileContextMenu(e, template, fullPath);
		});

		return template;
	}

	/**
	* Filters currently loaded sidebar items by text
	* @param {string} text
	*/
	filterByText(text){
		text = text.toLowerCase();
		const items = document.querySelectorAll(".theme-manager-fm-entry");
		for (let item of items){
			const name = item.querySelector(".item-label").textContent.trim().toLowerCase();
			if (text !== ""){
				if (name.includes(text)){
					item.style.display = "inline";
				}else{
					item.style.display = "none";
				}
			}else{
				item.style.display = "inline";
			}
		}
	}
}

export default (new SidebarFileLoader);
