import BSModal from "../../../_external/bootstrap/modal.js";
import {PageSectionComponentRow} from "../components/PageSectionComponentRow.js";

export class LayoutSectionsModal{
	/**
	 * The current file path of the active layout file this modal is representing.
	 * @type {string|null}
	 */
	currentLayoutOSFilePath = null;
	/**
	 * The current PageLayoutSectionDefinition object that is actively being represented by this
	 * modal.
	 * @type {{id: number, layoutFileName: string, creationTimestamp: number}|null}
	 */
	currentPageLayoutSectionDefinition = null;
	/**
	 * A flag to determine if this modal is actively fetching content from the back-end
	 * @type {boolean}
	 */
	isLoading = false;
	/**
	 * A flag to determine if this modal is actively creating a new section
	 * @type {boolean}
	 */
	isCreatingANewSection = false;
	/**
	 * The Bootstrap modal instance
	 * @type {BSModal}
	 */
	bsModal;
	/**
	 * The actual HTML div that is this modal
	 * @type {HTMLDivElement}
	 */
	modal;
	/**
	 * The loader element for this modal
	 * @type {HTMLDivElement}
	 */
	loader;
	/**
	 * The container for the "this page has no section" message
	 * @type {HTMLDivElement}
	 */
	noSectionsMessage;
	/**
	 * The container for individual layout section components
	 * @type {HTMLDivElement}
	 */
	sectionComponentsContainer;
	/**
	 * The button to create a new section for the currently selected layout file
	 * @type {HTMLButtonElement}
	 */
	newSectionButton;
	newSectionFormContainer;
	newSectionForm;
	/**
	 * A current list of the PageSectionComponentRow component instances that are rendered.
	 * @type {PageSectionComponentRow[]}
	 */
	currentPageLayoutSectionComponentRows = [];

	constructor(){
		this.build();
	}

	/**
	* Forces the modal to be shown
	*/
	showModal(){
		this.bsModal.show();
	}

	/**
	* Forces the modal to be hidden
	*/
	hideModal(){
		this.bsModal.hide();
	}

	/**
	 * Builds the modal's HTML. Returns itself.
	 * @returns {this}
	 */
	build(){
		const template = document.createElement("div");
		template.classList.add("modal");
		template.classList.add("fade");
		template.setAttribute("tabindex", "-1");
		template.innerHTML = `
			<div class="modal-dialog modal-lg">
				<div class="modal-content">
					<div class="modal-header">
						<h5 class="modal-title">Layout sections</h5>
						<button type="button" class="btn-close" data-bs-dismiss="modal"></button>
					</div>
					<div class="modal-body">
						<p>
							These are section definitions for the layout file <strong class="current-layout-file-name"></strong>. When layout sections are defined for a layout file, the Page Editor will load individual editors per-section instead of a single body editor for that page. The PHP $bodyContents variable will no longer be populated.
						</p>
						<button type="button" class="new-section-button btn btn-primary btn-sm">
							<span>New Section</span>
						</button>
						<hr>
						<div class="loader">
							<div class="p-3 text-center">
								<div class="spinner-border text-primary" style="width:4rem; height: 4rem;"></div><br>
								<em>Loading...</em>
							</div>
						</div>
						<div style="display: none;" class="new-section-form-container">
							<form id="create-new-layout-section-form">
								<div class="input-group">
									<label for="new-layout-section-name-input" class="input-group-text">Section name</label>
									<input id="new-layout-section-name-input" type="text" class="form-control">
									<button type="submit" form="create-new-layout-section-form" class="btn btn-success">Save</button>
									<button type="button" class="cancel-new-section-button btn btn-text">Cancel</button>
								</div>
							</form>
						</div>
						<div style="display:none;" class="section-components-container"></div>
						<div style="display: none;" class="no-sections-message">
							<div class="text-center p-2">
								<em>This layout file does not currently have any defined sections.</em>
							</div>
						</div>
					</div>
				</div>
			</div>
		`;

		this.modal = template;
		this.bsModal = new BSModal(template);
		this.loader = template.querySelector(".loader");
		this.sectionComponentsContainer = template.querySelector(".section-components-container");
		this.noSectionsMessage = template.querySelector(".no-sections-message");
		this.newSectionButton = template.querySelector(".new-section-button");
		this.newSectionFormContainer = template.querySelector(".new-section-form-container");
		this.newSectionForm = template.querySelector("#create-new-layout-section-form");

		this.newSectionForm.addEventListener("submit", e => {
			e.preventDefault();
			this.onNewSectionFormSubmitted();
		});

		this.newSectionButton.addEventListener("click", () => {
			this.onNewSectionButtonClicked();
		});

		this.newSectionFormContainer.querySelector(".cancel-new-section-button").addEventListener("click", () => {
			this.onNewSectionCancelButtonClicked();
		});

		return this;
	}

	/**
	 * When the button to show the new section form is clicked.
	 */
	onNewSectionButtonClicked(){
		this.newSectionFormContainer.style.display = null;

		// Hide the no sections message?
		if (this.sectionComponentsContainer.childNodes.length === 0){
			this.noSectionsMessage.style.display = "none";
		}
	}

	/**
	 * When the user clicks the button to cancel new section creation
	 */
	onNewSectionCancelButtonClicked(){
		this.newSectionFormContainer.style.display = "none";

		// Show the no sections message or the components?
		if (this.sectionComponentsContainer.childNodes.length === 0){
			this.noSectionsMessage.style.display = null;
		}
	}

	async onNewSectionFormSubmitted(){
		if (this.isCreatingANewSection){
			return;
		}

		this.isCreatingANewSection = true;

		// Determine if we must send the current section definition Id or a full file path.
		// Sending a full file path creates a definition for this layout file, if it doesn't exist.
		const payload = new FormData();
		if (this.currentPageLayoutSectionDefinition === null){
			payload.set("pageLayoutFullFilePath", this.currentLayoutOSFilePath);
			payload.set("sectionName", this.newSectionForm.querySelector("#new-layout-section-name-input").value);

			const response = await fetch(`/uplift/theme-manager/page-layouts/sections-definition/sections`, {
				method: "POST",
				body: payload
			});

			if (response.status === 200){
				/** @var {{status: number}} apiData */
				const apiData = await response.json();
				if (apiData.status === 1){
					/** @var {{status: number, definition: Object, section: Object}} apiData */

					/** @type {{id: number, layoutFileName: string, creationTimestamp: number}} */
					const newDefinition = apiData.definition;
					this.currentPageLayoutSectionDefinition = newDefinition;

					// Render the new section
					const newComponent = new PageSectionComponentRow(apiData.section)
						.build()
						.renderInto(this.sectionComponentsContainer);

					newComponent.onFinishedSaving(() => {
						this.onComponentFinishedSaving(newComponent);
					});

					this.currentPageLayoutSectionComponentRows.push(newComponent);

					this.sectionComponentsContainer.style.display = null;

					// Hide the form by clicking the cancel button
					this.onNewSectionCancelButtonClicked();

					// Hide the no results message
					this.noSectionsMessage.style.display = "none";

					this.newSectionForm.reset();
				}else{
					/** @var {{status: number, error: string}} apiData */
					alert(apiData.error);
				}
			}else{
				alert("Internal server error.");
			}
		}else{
			// Send the definition Id, and not the full file path
			payload.set("pageLayoutSectionDefinitionId", String(this.currentPageLayoutSectionDefinition.id));
			payload.set("sectionName", this.newSectionForm.querySelector("#new-layout-section-name-input").value);

			const response = await fetch(`/uplift/theme-manager/page-layouts/sections-definition/sections`, {
				method: "POST",
				body: payload
			});

			if (response.status === 200){
				/** @var {{status: number}} apiData */
				const apiData = await response.json();
				if (apiData.status === 1){
					/** @var {{status: number, section: Object}} apiData */

					/** @type {{id: number, pageLayoutSectionDefinitionId: number, sectionName: string, content: string, creationTimestamp: number}} */
					const newSection = apiData.section;

					// Render the new section
					const newComponent = new PageSectionComponentRow(newSection)
						.build()
						.renderInto(this.sectionComponentsContainer);

					newComponent.onFinishedSaving(() => {
						this.onComponentFinishedSaving(newComponent);
					});

					this.currentPageLayoutSectionComponentRows.push(newComponent);

					// Hide the form by clicking the cancel button
					this.onNewSectionCancelButtonClicked();

					this.newSectionForm.reset();
				}else{
					/** @var {{status: number, error: string}} apiData */
					alert(apiData.error);
				}
			}else{
				alert("Internal server error.");
			}
		}

		this.isCreatingANewSection = false;
	}

	/**
	 * Loads and renders the section definition and sections for
	 * the provided file path. The file path should be a full file path
	 * on the current OS.
	 * @param {string} fullOSFilePath
	 * @returns {Promise<void>}
	 */
	async loadAndRenderFileSections(fullOSFilePath){
		if (this.isLoading){
			return;
		}

		this.currentLayoutOSFilePath = fullOSFilePath;
		this.isLoading = true;
		this.sectionComponentsContainer.innerHTML = "";
		this.sectionComponentsContainer.style.display = "none";
		this.loader.style.display = null;
		this.noSectionsMessage.style.display = "none";

		const searchParams = new URLSearchParams({
			filePath: fullOSFilePath
		});

		const response = await fetch(`/uplift/theme-manager/page-layouts/sections-definition?${searchParams}`);
		this.loader.style.display = "none";
		this.isLoading = false;

		if (response.status === 200){
			/** @type {{status: number}} */
			const apiData = await response.json();
			if (apiData.status === 1){
				/** @var {{status: number, pageLayoutSectionDefinition: object}} apiData */

				/** @type {{id: number, layoutFileName: string, creationTimestamp: number, pageLayoutSections:Object[]}} */
				const sectionDefinition = apiData.pageLayoutSectionDefinition;
				this.currentPageLayoutSectionDefinition = sectionDefinition;
				this.renderSections(sectionDefinition.pageLayoutSections);
				this.sectionComponentsContainer.style.display = null;
			}else{
				/** @var {{status: number, error: string}} apiData */
				// console.warn(apiData.error);
				this.currentPageLayoutSectionDefinition = null;
				this.noSectionsMessage.style.display = null;
			}
		}else{
			console.warn(await response.text());
			this.currentPageLayoutSectionDefinition = null;
			this.noSectionsMessage.style.display = null;
		}
	}

	/**
	 * Renders page sections into the components container
	 * @param {{id: number, pageLayoutSectionDefinitionId: number, sectionName: string, content: string, creationTimestamp: number}[]} dtos
	 */
	renderSections(dtos){
		// Clear array of existing components
		this.currentPageLayoutSectionComponentRows = [];

		// Clear inner HTML of where the existing components were rendered
		this.sectionComponentsContainer.innerHTML = "";

		for (const dto of dtos){
			const newComponent = new PageSectionComponentRow(dto)
				.build()
				.renderInto(this.sectionComponentsContainer);

			newComponent.onFinishedSaving(() => {
				this.onComponentFinishedSaving(newComponent);
			});

			this.currentPageLayoutSectionComponentRows.push(newComponent);
		}
	}

	/**
	 * Fires when any of the PageSectionComponentRow components finish saving.
	 * @param {PageSectionComponentRow} component
	 */
	onComponentFinishedSaving(component){
		for (const existingComponent of this.currentPageLayoutSectionComponentRows){
			if (existingComponent.internalId !== component.internalId){
				if (existingComponent.dto.isDefaultContentSection === 1&& component.dto.isDefaultContentSection === 1){
					// It is no longer the default if the most recently saved component is
					existingComponent.dto.isDefaultContentSection = 0;
					existingComponent.removeDefaultContentBadge();
					existingComponent.editFormContainer.querySelector(".default-content-check-input").checked = false;
				}
			}
		}
	}
}
