import PageEditorStates from "./PageEditorStates.js";
import AceEditorExtension from "../AceEditorExtension.js";
import BreadcrumbsManager from "./BreadcrumbsManager.js";
import ContentChangedHandler from "../ContentChangedHandler.js";
import EditorTabs from "./EditorTabs.js";
import DOMHelper from "../DOMHelper.js";
import SidebarPageLoader from "./SidebarPageLoader.js";
import FeaturedImage from "./FeaturedImage.js";
import ImageManager from "../../image-manager/ImageManager.js";
import SidebarPageNameFilter from "./SidebarPageNameFilter.js";
import ArticleCategoryRow from "./components/ArticleCategoryRow.js";
import ProjectTagRow from "./components/ProjectTagRow.js";

// Shortcode buttons
import PhoneNumbersShortcodeButton from "./shortcode-buttons/PhoneNumbers.js";
import PurePhoneNumbers from "./shortcode-buttons/PurePhoneNumbers.js";
import FaxNumbersShortcodeButton from "./shortcode-buttons/FaxNumbers.js";
import CompanyNameShortcodeButton from "./shortcode-buttons/CompanyName.js";
import CityPageCityShortcodeButton from "./shortcode-buttons/PageCity.js";
import CityPageStateShortcodeButton from "./shortcode-buttons/PageState.js";
import CompanyStreetShortcodeButton from "./shortcode-buttons/CompanyStreet.js";
import CompanyPostalShortcodeButton from "./shortcode-buttons/CompanyPostal.js";
import CompanyCityShortcodeButton from "./shortcode-buttons/CompanyCity.js";
import CompanyStateShortcodeButton from "./shortcode-buttons/CompanyState.js";
import BlogReelShortcodeButton from "./shortcode-buttons/BlogReel.js";
import RecentProjectsShortcodeButton from "./shortcode-buttons/RecentProjects.js";
import IPPReelShortcodeButton from "./shortcode-buttons/IPPReel.js";
import RecentArticlesShortcodeButton from "./shortcode-buttons/RecentArticles.js";
import ReviewsShortcodeButton from "./shortcode-buttons/Reviews.js";
import GetPagesShortcodeButton from "./shortcode-buttons/GetPages.js";

// Modal singletons
import NewPageModal from "./modals/NewPageModal.js";
import DeletePageModal from "./modals/DeletePageModal.js";
import BlogReelModal from "./modals/BlogReelModal.js";
import IPPReelModal from "./modals/IPPReelModal.js";
import RecentProjectsModal from "./modals/RecentProjectsModal.js";
import RecentArticlesModal from "./modals/RecentArticlesModal.js";
import NewBlogCategoryModal from "./modals/NewBlogCategoryModal.js";
import BlogCategoryRenameModal from "./modals/BlogCategoryRenameModal.js";
import InsertGalleryModal from "./modals/InsertGalleryModal.js";
import BlogCategoryManageModal from "./modals/BlogCategoryManageModal.js";
import ProjectTagsManageModal from "./modals/ProjectTagsManageModal.js";

// Forms
import NewPageForm from "./forms/NewPageForm.js";
import DeletePageForm from "./forms/DeletePageForm.js";
import ClonePageForm from "./forms/ClonePageForm.js";
import ConvertPageTypeForm from "./forms/ConvertPageTypeForm.js";
import BlogReelForm from "./forms/BlogReelForm.js";
import RecentProjectsForm from "./forms/RecentProjectsForm.js";
import RecentArticlesForm from "./forms/RecentArticlesForm.js";
import NewBlogCategoryForm from "./forms/NewBlogCategoryForm.js";
import BlogCategoryRenameForm from "./forms/BlogCategoryRenameForm.js";
import NewProjectTagForm from "./forms/NewProjectTagForm.js";
import RenameProjectTagForm from "./forms/RenameProjectTagForm.js";
import IPPReelForm from "./forms/IPPReelForm.js";
import GetPagesForm from "./forms/GetPagesForm.js";
import PageEditForm from "./forms/PageEditForm.js";
import FilterGetPagesResultsForm from "./forms/FilterGetPagesResultsForm.js";

// Input component helpers
// import BlogCategorySelectDropdown from "./inputs/BlogCategorySelectDropdown.js";
import IPPCustomerTestimonial from "./inputs/IPPCustomerTestimonial.js";
import RestoreRevisionButton from "./components/RestoreRevisionButton.js";

// Toolbar buttons
import ImageManagerButton from "./toolbar/ImageManagerButton.js";

// Components
import CustomerTestimonialCheckbox from "./components/CustomerTestimonialCheckbox.js";
import PageCategoryButton from "./components/PageCategoryButton.js";
import PageList from "./components/PageList.js";
import PageCategoryBackButton from "./components/PageCategoryBackButton.js";
import NewPageButton from "./components/NewPageButton.js";
import Breadcrumb from "./components/Breadcrumb.js";
import ArticleCategoryManager from "./ArticleCategoryManager.js";
import ProjectTagManager from "./ProjectTagManager.js";
import PageButton from "./components/PageButton.js";
import Scheduler from "../../utils/Scheduler.js";
import RevisionsButtonsContainer from "./components/RevisionsButtonsContainer.js";
import RevisionLoader from "./RevisionLoader.js";
import PageContentLoader from "./PageContentLoader.js";
import BlogCategoriesManagerButton from "./components/BlogCategoriesManagerButton.js";

// Buttons
import AddAttributeButton from "./buttons/AddAttributeButton.js";
import {PageContentSectionEditor} from "./components/PageContentSectionEditor.js";

class PageEditor {
	sidebarPageLoader;
	sidebar = document.querySelector("#content-editor-sidebar");
	fileButtonContainer = document.querySelector("#editor-file-button-container");
	openImageManagerToolbarButton = document.querySelector("#upload-insert-image-button");
	toolsToolbarButton = document.querySelector("#tools-toolbar-container");
	pagePublicationInput = document.querySelector("#page-publication");
	pagePublicationDateTimeInput = document.querySelector("#page-publication-timestamp");
	pageEditorContainer = document.querySelector("#content-editor-container");
	pageEditorLoaderContainer = document.querySelector("#content-editor-loading-container");
	pageEditorPageToolbarButtons = document.querySelector("#editor-page-buttons-toolbar");
	pageEditorShortcodesToolbarButtonContainer = document.querySelector("#editor-shortcodes-toolbar-button");
	pageEditorGalleryToolbarButtonContainer = document.querySelector("#insert-gallery-button");
	pageEditorRevisionToolbarButton = document.querySelector("#page-revision-history-button");
	convertPageButton = document.querySelector("#convert-page-button");
	clonePageButton = document.querySelector("#clone-page-button");
	deletePageButton = document.querySelector("#delete-page-button");

	/**
	 * @type {?ImageManager}
	 */
	imageManager = null;

	constructor() {
		this.loadImageManager();
		this.setup();
	}

	loadImageManager(){
		const imageManagerPermInput = document.querySelector("#has-image-manager-perms");
		if (imageManagerPermInput.value === "1"){
			this.imageManager = new ImageManager(document.querySelector("#image-manager-container"));
		}
	}

	setup(){
		if (PageEditorStates.viewType === "REVISION"){
			// Hide the sidebar and all toolbar buttons
			this.sidebar.style.display = "none";
			this.hideAllToolbarButtons();
			RevisionsButtonsContainer.show();
		}

		// Get the tabs for each view type
		const editorTabs = DOMHelper.getAllPageEditorTabs();

		// Load the extra, custom commands into the editors
		const headExtension = new AceEditorExtension(document.querySelector("#page-head"), "html");
		const bodyExtension = new AceEditorExtension(document.querySelector("#page-content"), "html");

		AceEditorExtension.pageEditors.body = bodyExtension;
		AceEditorExtension.pageEditors.head = headExtension;

		bodyExtension.loadExtensions();
		headExtension.loadExtensions();

		// Listen for changes on the editors
		bodyExtension.ace.on("change", () => {
			if (!PageEditorStates.isLoadingPage) {
				ContentChangedHandler.showUnsavedChangeIdentifier(editorTabs.body);
				PageEditorStates.markAsHavingUnsavedChanges(false);
			}
		});

		headExtension.ace.on("change", () => {
			if (!PageEditorStates.isLoadingPage) {
				ContentChangedHandler.showUnsavedChangeIdentifier(editorTabs.head);
				PageEditorStates.markAsHavingUnsavedChanges(false);
			}
		});

		// Set the default currentEditor as the body
		PageEditorStates.currentEditor = bodyExtension;

		// Event for if there are unsaved changes. Block browser navigation
		window.addEventListener("beforeunload", e => {
			if (PageEditorStates.unsavedChanges) {
				e.preventDefault();
				e.returnValue = "You have unsaved changes. Are you sure you want to leave?";
				return false;
			}
		});

		// Override Ctrl+S for the window
		window.addEventListener("keydown", e => {
			if (e.ctrlKey && e.key.toLowerCase() === "s") {
				e.preventDefault();
				e.stopPropagation();

				// Submit the page edit/save form
				PageEditForm.onSubmit();
			}
		});

		/**
		 * A bunch of event listeners to show unsaved changes notifications on the
		 * page data editor tab.
		 */
		(() => {
			document.querySelector("#page-name").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});
			document.querySelector("#page-layout").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
				this.onPageLayoutChange();
			});
			document.querySelector("#page-url").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});

			document.querySelector("#exclude-from-sitemap").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});

			this.pagePublicationInput.addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
				const publicationTypeEnumIntegerValue = parseInt(this.pagePublicationInput.value);
				if (publicationTypeEnumIntegerValue === 0){
					// Draft. Hide publication time
					document.querySelector("#page-publication-datetime-container").style.display = "none";
				}else{
					document.querySelector("#page-publication-datetime-container").style.display = "block";
				}
			});

			document.querySelector("#city-page-city-name").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});

			document.querySelector("#city-page-state-name").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});

			document.querySelector("#city-page-state-name").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});

			document.querySelector("#city-page-state-name-shorthand").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});

			document.querySelector("#city-page-city-url").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});

			document.querySelector("#city-name").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});

			document.querySelector("#state-name").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});

			document.querySelector("#state-name-shorthand").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});

			document.querySelector("#brands-products").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});

			document.querySelector("#customer-testimonial-check").addEventListener("change", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});

			document.querySelector("#customer-review-first-name").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});

			document.querySelector("#customer-review-last-name").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});

			document.querySelector("#customer-testimonial-body").addEventListener("input", () => {
				ContentChangedHandler.onPageEditorDataChanged(PageEditorStates);
			});
		})();

		// Load the page type category button instances
		const pageTypeButtons = document.querySelectorAll(".page-type-enum-button");
		for (const button of pageTypeButtons){
			new PageCategoryButton(button);
		}

		// Load the page list instances
		const pageListContainers = document.querySelectorAll(".page-list");
		for (const container of pageListContainers){
			new PageList(container);
		}

		// Preload the necessary data for the right type of viewing mode
		if (PageEditorStates.viewType === "STANDARD") {
			// Initiate all pages in the sidebar
			this.sidebarPageLoader = new SidebarPageLoader();
			// Get the page ID from the URL, if there is one
			const pageIDInURL = this.getPageIDFromURL();

			if (pageIDInURL !== null) {
				(async () => {
					const fetchedPageData = await this.loadPage(pageIDInURL);
					if (fetchedPageData !== null) {
						const pageType = fetchedPageData.page.pageType;
						const pageCategorySidebarButton = PageCategoryButton.getButtonForPageType(pageType);

						this.sidebarPageLoader.hideLoader();

						// Fire the click so the correct page list is shown
						this.sidebarPageLoader.showCategoryButtonsContainer();
						pageCategorySidebarButton.onClick();

						// Fetch the page button and set it as active
						const pageButton = PageButton.getPageButtonByPageID(pageIDInURL);
						pageButton.setAsActive();
					}
				})();
			} else {
				// Because there is no page to preload, go ahead and show the page category buttons now
				this.sidebarPageLoader.hideLoader();
				this.sidebarPageLoader.showCategoryButtonsContainer();
			}
		}else if (PageEditorStates.viewType === "REVISION"){
			// This will always have a revision ID in the URL
			const idsFromURL = this.getPageIDAndRevisionIDFromURL();
			RevisionLoader.loadRevision(this, idsFromURL.pageID, idsFromURL.revisionID);
		}
	}

	/**
	 * Fires when the page layout changes. We need to find out if the newly selected layout has sections.
	 * @returns {Promise<void>}
	 */
	async onPageLayoutChange(){
		this.toggleAllFieldsReadOnly(true);

		// Set the local cache of the current page's layout
		PageEditorStates.currentPage.pageLayout = document.querySelector("#page-layout").value;

		// Clear any existing content section edits
		PageContentSectionEditor.clearAll();

		let pageLayoutSectionDefinition = null;
		try{
			pageLayoutSectionDefinition = await this.getSectionDefinitionForLayoutName(PageEditorStates.currentPage.pageLayout);
		}catch(ex){}

		PageEditorStates.currentPageLayoutSectionDefinition = pageLayoutSectionDefinition;

		// Reload the new page content sections
		if (pageLayoutSectionDefinition !== null){
			PageContentLoader.loadPageContentSections(PageEditorStates.lastLoadedPageContentSections, pageLayoutSectionDefinition)
		}

		// On page load, determine if the "body" tab is selected.
		// If so, determine if the primary body window should be visible, or the bodySections window should be.
		this.showHideCorrectBodyEditorIfVisibleForCurrentLayoutSectionDefinition();
		this.toggleAllFieldsReadOnly(false);
	}

	/**
	 * Fetches the page ID from the current page's URL path
	 * @returns {int | null}
	 */
	getPageIDFromURL(){
		const currentPath = window.location.pathname;
		const matches = currentPath.match(/\/uplift\/page-editor\/(\d+)$/);
		if (matches !== null && 1 in matches) {
			return parseInt(matches[1]);
		}

		return null;
	}

	/**
	 * Fetches the revision ID from the current page's URL path
	 * @returns {{pageID: int, revisionID: int} | null}
	 */
	getPageIDAndRevisionIDFromURL(){
		const currentPath = window.location.pathname;
		const matches = currentPath.match(/\/uplift\/page-editor\/revisions\/(\d+)\/(\d+)$/);
		if (matches !== null && 2 in matches) {
			return {
				pageID: parseInt(matches[1]),
				revisionID: parseInt(matches[2])
			};
		}

		return null;
	}

	/**
	 * Hides the top-level toolbar buttons. Used when viewing revisions.
	 */
	hideAllToolbarButtons(){
		this.fileButtonContainer.style.display = "none";
		this.openImageManagerToolbarButton.style.display = "none";
		this.toolsToolbarButton.style.display = "none";
	}

	/**
	 * Hides all utility buttons that can only be used if a page is loaded in
	 */
	hidePageUtilityButtons(){
		this.pageEditorPageToolbarButtons.style.display = "none";
		this.pageEditorShortcodesToolbarButtonContainer.style.display = "none";
		this.pageEditorGalleryToolbarButtonContainer.style.display = "none";
		this.convertPageButton.style.display = "none";
		this.clonePageButton.style.display = "none";
		this.deletePageButton.style.display = "none";

		// If the user doesn't have permission to use the page editor, the button will be null
		if(this.pageEditorRevisionToolbarButton !== null) {
			this.pageEditorRevisionToolbarButton.style.display = "none";
		}
	}

	showPageUtilityButtons(){
		this.pageEditorPageToolbarButtons.style.display = null;
		this.pageEditorShortcodesToolbarButtonContainer.style.display = null;
		this.pageEditorGalleryToolbarButtonContainer.style.display = null;
		this.convertPageButton.style.display = null;
		this.clonePageButton.style.display = null;
		this.deletePageButton.style.display = null;

		// If the user doesn't have permission to use the page editor, the button will be null
		if(this.pageEditorRevisionToolbarButton !== null) {
			this.pageEditorRevisionToolbarButton.style.display = null;
		}
	}

	/**
	 * Loads a page into the editor
	 * @param {int} pageID
	 * @returns {Promise<{page: Object, breadcrumbs: Object, data: Object} | null>} The page that was loaded
	 */
	async loadPage(pageID){

		// Do not load the same page again
		if (PageEditorStates.currentPageID === pageID){
			return null;
		}

		// Do not try loading a new page when one is already being fetched
		if (PageEditorStates.isLoadingPage){
			return null;
		}

		// Loading the same page ID from the URL? If not, make the page state this URL
		if (this.getPageIDFromURL() !== pageID) {
			window.history.pushState({}, "", `/uplift/page-editor/${pageID}`);
		}

		PageEditorStates.isLoadingPage = true;
		this.hidePageEditor();
		this.hidePageUtilityButtons();
		this.showPageEditorLoader();

		// Hide the Ace body editor, because the new page _could_ have sections and not just be a main body content
		PageContentSectionEditor.clearAll();

		const response = await fetch(`/uplift/page-editor/page/${pageID}`, {
			cache:"no-cache",
			credentials:"same-origin"
		});

		let data;
		try{
		    /** @type {{status: int, error: ?string, page: Object, data: Object, breadcrumbs: Array, pageContentSections: Object[]}} **/
		    data = await response.json();
		}catch(jsonSyntaxError){
		    alert("The server responded with invalid JSON.");
		    return null;
		}

		if (data.status === 1){

			this.showPageUtilityButtons();

			/** @type {{id:int, pageName: string, pageType: string, pageRoute: string, pageBody: string, pageHead: string, pageRouteIsRegex: int, pageLayout: string, creationTime: int, excludeFromSitemap: int, excludeSchemaInjection: int, publicationStatus: int, publicationTimestamp: int, isDeleted: int, attributes: Object[]}} **/
			const page = data.page;

			PageEditorStates.currentPage = page;
			PageEditorStates.currentPageID = page.id;
			PageEditorStates.currentPageType = page.pageType;
			PageEditorStates.lastLoadedPageContentSections = data.pageContentSections;

			// Fetch the PageLayoutSectionsDefinition object for the page layout the current page uses
			// It will stay null if there isn't a layout section for the selected page layout
			let pageLayoutSectionDefinition = null;
			try{
				pageLayoutSectionDefinition = await this.getSectionDefinitionForLayoutName(page.pageLayout);
			}catch(ex){
				// alert(ex);
				// return null;
			}

			PageEditorStates.currentPageLayoutSectionDefinition = pageLayoutSectionDefinition;

			// On page load, determine if the "body" tab is selected.
			// If so, determine if the primary body window should be visible, or the bodySections window should be.
			this.showHideCorrectBodyEditorIfVisibleForCurrentLayoutSectionDefinition();

			// Minimum wait time so things don't happen too fast - prevent "blinking" fast loads
			await Scheduler.wait(500);

			// If the user doesn't have permission to use the page editor, the button will be null
			if(this.pageEditorRevisionToolbarButton !== null) {
				// Set revision history href
				this.pageEditorRevisionToolbarButton.setAttribute("href", `/uplift/page-editor/revisions/${page.id}`);
			}

			/** @type {{position: int, label: string, uri: string}[]} */
			const breadcrumbs = data.breadcrumbs;

			/** Will house the page's custom set data. Keys are the data name and values are the page data values. They will either be string values or an array of strings. */
			const pageData = data.data;
			const attributes = data.attributes;

			PageContentLoader.loadPage(page, data.pageContentSections, pageLayoutSectionDefinition);
			PageContentLoader.loadPageBreadcrumbs(breadcrumbs);
			PageContentLoader.loadPageData(page, pageData);
			PageContentLoader.loadPageAttributes(attributes);

			// Show the editor and hide the loader
			this.showPageEditor();
			this.hidePageEditorLoader();

			// Call resize on the main ace editors
			AceEditorExtension.pageEditors.body.ace.resize(true);
			AceEditorExtension.pageEditors.head.ace.resize(true);

			// Done loading, set state to no longer loading a page
			PageEditorStates.isLoadingPage = false;

			return data;
		}else if (data.status === -1){

		}

		// Done loading, set state to no longer loading a page
		PageEditorStates.isLoadingPage = false;

		return null;
	}

	showPageEditor(){
		this.pageEditorContainer.style.display = null;
	}

	showPageEditorLoader(){
		this.pageEditorLoaderContainer.style.display = null
	}

	hidePageEditor(){
		this.pageEditorContainer.style.display = "none";
	}

	hidePageEditorLoader(){
		this.pageEditorLoaderContainer.style.display = "none";
	}

	/**
	 * Hides all data sections
	 */
	hideDataSections(){
		document.querySelector("#page-data-featured-image").style.display = "none";
		document.querySelector("#page-data-blog").style.display = "none";
		document.querySelector("#page-data-project").style.display = "none";
		document.querySelector("#page-data-city").style.display = "none";
		document.querySelector("#page-data-service").style.display = "none";
	}

	/**
	 * Toggles all editable fields as read-only or not.
	 * @param {boolean} flag
	 */
	toggleAllFieldsReadOnly(flag){

		if (flag === undefined){
			flag = true;
		}

		if (flag === true) {
			AceEditorExtension.pageEditors.head.ace.setReadOnly(true);
			AceEditorExtension.pageEditors.body.ace.setReadOnly(true);
			BreadcrumbsManager.hideButton();
			PageContentLoader.pageNameInput.setAttribute("readonly", "");
			PageContentLoader.pageLayoutInput.setAttribute("readonly", "");
			PageContentLoader.pageURLInput.setAttribute("readonly", "");
			PageContentLoader.pagePublicationInput.setAttribute("readonly", "");
			PageContentLoader.pagePublicationDateTimeInput.setAttribute("readonly", "");
			PageContentLoader.excludeFromSitemapCheckbox.setAttribute("readonly", "");
			PageContentLoader.projectCityNameInput.setAttribute("readonly", "");
			PageContentLoader.projectStateNameInput.setAttribute("readonly", "");
			PageContentLoader.projectStateNameShorthandInput.setAttribute("readonly", "");
			PageContentLoader.brandsProductsInput.setAttribute("readonly", "");
			PageContentLoader.customerReviewFirstName.setAttribute("readonly", "");
			PageContentLoader.customerReviewLastName.setAttribute("readonly", "");
			PageContentLoader.customerReviewTestimonial.setAttribute("readonly", "");
			PageContentLoader.customerTestimonialCheckbox.setAttribute("readonly", "");
			PageContentLoader.cityCityNameInput.setAttribute("readonly", "");
			PageContentLoader.cityStateNameInput.setAttribute("readonly", "");
			PageContentLoader.cityStateNameShorthandInput.setAttribute("readonly", "");
			PageContentLoader.cityCityURLInput.setAttribute("readonly", "");
			FeaturedImage.setButton.style.display = "none";
			FeaturedImage.clearButton.style.display = "none";
			BlogCategoriesManagerButton.hide();

			for (const crumb of BreadcrumbsManager.breadcrumbs) {
				crumb.dom.querySelector(".crumb-label").setAttribute("readonly", "");
				crumb.dom.querySelector(".crumb-uri").setAttribute("readonly", "");
				crumb.dom.querySelector(".delete-button").style.display = "none";
				crumb.dom.querySelector(".prefill-crumb-button").style.display = "none";
			}

			for (const component of PageContentSectionEditor.cache){
				component.aceEditorExtension.ace.setReadOnly(true);
			}
		}else{
			AceEditorExtension.pageEditors.head.ace.setReadOnly(false);
			AceEditorExtension.pageEditors.body.ace.setReadOnly(false);
			BreadcrumbsManager.showButton();
			PageContentLoader.pageNameInput.removeAttribute("readonly");
			PageContentLoader.pageLayoutInput.removeAttribute("readonly");
			PageContentLoader.pageURLInput.removeAttribute("readonly");
			PageContentLoader.pagePublicationInput.removeAttribute("readonly");
			PageContentLoader.pagePublicationDateTimeInput.removeAttribute("readonly");
			PageContentLoader.excludeFromSitemapCheckbox.removeAttribute("readonly");
			PageContentLoader.projectCityNameInput.removeAttribute("readonly");
			PageContentLoader.projectStateNameInput.removeAttribute("readonly");
			PageContentLoader.projectStateNameShorthandInput.removeAttribute("readonly");
			PageContentLoader.brandsProductsInput.removeAttribute("readonly");
			PageContentLoader.customerReviewFirstName.removeAttribute("readonly");
			PageContentLoader.customerReviewLastName.removeAttribute("readonly");
			PageContentLoader.customerReviewTestimonial.removeAttribute("readonly");
			PageContentLoader.customerTestimonialCheckbox.removeAttribute("readonly");
			PageContentLoader.cityCityNameInput.removeAttribute("readonly");
			PageContentLoader.cityStateNameInput.removeAttribute("readonly");
			PageContentLoader.cityStateNameShorthandInput.removeAttribute("readonly");
			PageContentLoader.cityCityURLInput.removeAttribute("readonly");
			FeaturedImage.setButton.style.display = null;
			FeaturedImage.clearButton.style.display = null;
			BlogCategoriesManagerButton.show();

			for (const crumb of BreadcrumbsManager.breadcrumbs) {
				crumb.dom.querySelector(".crumb-label").removeAttribute("readonly");
				crumb.dom.querySelector(".crumb-uri").removeAttribute("readonly");
				crumb.dom.querySelector(".delete-button").style.display = null;
				crumb.dom.querySelector(".prefill-crumb-button").style.display = null;
			}

			for (const component of PageContentSectionEditor.cache){
				component.aceEditorExtension.ace.setReadOnly(false);
			}
		}
	}

	/**
	 * Fetches the PageLayoutSectionsDefinition for the provided layoutName, if there is one.
	 * @param {string} layoutName E.g.: "General"
	 * @returns {Promise<{creationTimestamp: number, id: number, layoutFileName: string, pageLayoutSections: Object[]}>}
	 * @throws
	 */
	async getSectionDefinitionForLayoutName(layoutName){
		const params = new URLSearchParams({layoutFileName: layoutName});
		const sectionsDefinitionResponse = await fetch(`/uplift/page-editor/page-layout-sections?${params}`);

		if (sectionsDefinitionResponse.status !== 200){
			throw "Internal server error when loading page layout section definition.";
		}

		/** @type {{status: number}} */
		const sectionsDefinitionApiResponse = await sectionsDefinitionResponse.json();
		if (sectionsDefinitionApiResponse.status === -1){
			/** @var {{status: number, error: string}} sectionsDefinitionApiResponse */
			throw sectionsDefinitionApiResponse.error;
		}

		/** @var {{status: number, definition: {creationTimestamp: number, id: number, layoutFileName: string, pageLayoutSections: Object[]}}} sectionsDefinitionApiResponse */

		return sectionsDefinitionApiResponse.definition;
	}

	/**
	 * Checks if the currently selected window tab is "body"
	 * If it is, checks if there is a defined PageLayoutSectionsDefinition.
	 * If so, then makes sure the section editor is visible and the main body editor is hidden,
	 * else if no layout section definition then it does the reverse.
	 */
	showHideCorrectBodyEditorIfVisibleForCurrentLayoutSectionDefinition(){
		if (PageEditorStates.currentEditorWindowName === "body"){
			const editorTabWindows = DOMHelper.getAllPageEditorWindows();
			if (PageEditorStates.currentPageLayoutSectionDefinition !== null){
				if (editorTabWindows.bodySections.style.display === "none"){
					// It needs to be shown
					editorTabWindows.bodySections.style.display = null;
					editorTabWindows.body.style.display = "none";
				}
			}else{
				if (editorTabWindows.body.style.display === "none"){
					editorTabWindows.bodySections.style.display = "none";
					editorTabWindows.body.style.display = null;
				}
			}
		}
	}
}

export default new PageEditor();
