class PageAttribute{

	/** @type {PageAttribute[]} */
	static cache = [];
	static container = document.querySelector("#page-rendered-attributes-container");

	static clearAll(){
		PageAttribute.cache = [];
		PageAttribute.container.innerHTML = "";
	}

	/**
	 * Retrieves the current cache, but removes the DOM from it so it can be easily
	 * sent as an HTTP payload for requests.
	 */
	static getCacheForHttpPayload(){
		const tempCache = [];

		for (const pageAttribute of PageAttribute.cache){
			tempCache.push({
				id: pageAttribute.id,
				pageAttributeID: pageAttribute.pageAttributeID,
				name: pageAttribute.name,
				value: pageAttribute.value
			});
		}

		return tempCache;
	}

	id;
	pageAttributeID;
	name;
	value;
	dom;
	shortcodeCopyCoolOff = false;

	constructor(id, name, value, pageAttributeID) {
		this.id = id;
		this.pageAttributeID = pageAttributeID;
		this.name = name;
		this.value = value;
		this.dom = this.getDOM();
		this.constructor.cache.push(this);
	}

	render(){
		PageAttribute.container.append(this.dom);
	}

	remove(){
		for (const [index, pageAttributeInCache] of this.constructor.cache.entries()){
			if (pageAttributeInCache.id === this.id){
				this.constructor.cache.splice(index, 1);
				break;
			}
		}

		this.dom.remove();
	}

	async onRemoveButtonClicked(){
		const response = await fetch(`/uplift/page-editor/attributes/page-attribute-value/${this.id}`, {
		    body:null,
		    method:"DELETE",
		    cache:"no-cache",
		    credentials:"same-origin"
		});

		this.remove();

		let data;
		try{
		    /** @type {{status: int, error: ?string}} **/
		    data = await response.json();
		}catch(jsonSyntaxError){
		    alert("The server responded with invalid JSON.");
		    return;
		}

		if (data.status === 1){
			// Check if there are none left and show the empty message
			if (PageAttribute.cache.length === 0){
				document.querySelector("#no-attributes-message-container").style.display = null;
			}
		}else if (data.status === -1){
			alert(data.error);
		}
	}

	async onCopyShortcodeToClipboardClicked(){

		if (this.shortcodeCopyCoolOff){
			return;
		}

		this.shortcodeCopyCoolOff = true;

		const copyButtonText = this.dom.querySelector(".copy-button-text");
		const pageAttributeShortcodeValueInput = this.dom.querySelector(".page-attribute-value-shortcode-input");
		pageAttributeShortcodeValueInput.select();
		pageAttributeShortcodeValueInput.setSelectionRange(0,99999);

		copyButtonText.textContent = "Copied!";
		navigator.clipboard.writeText(pageAttributeShortcodeValueInput.value);

		await new Promise(resolve => {
			setTimeout(() => {
				copyButtonText.textContent = "Copy Shortcode";
				this.shortcodeCopyCoolOff = false;
			}, 1000);
		});
	}

	getDOM(){
		const template = document.createElement("div");
		template.classList.add("page-attribute-value-container");
		template.innerHTML = `
			<div class="d-flex">
				<div class="input-group me-3">
					<label class="input-group-text" for="page-attribute-${this.id}">
						${this.name}
					</label>
					<input id="page-attribute-${this.id}" type="text" class="page-attribute-value-input form-control" value="${this.value}">
					<div class="ms-3">
						<button type="button" class="remove-button btn btn-outline-danger">
							<i class="bi bi-trash-fill"></i>
							<span>Remove</span>
						</button>
					</div>
					<input style="display:none;" readonly type="text" class="disabled page-attribute-value-shortcode-input form-control">
					<div class="ms-1">
						<button type="button" class="copy-to-clipboard btn btn-outline-primary">
							<i class="bi bi-clipboard2-plus-fill"></i>
							<span class="copy-button-text">Copy Shortcode</span>
						</button>
					</div>
				</div>
			</div>
			<div class="ids-container mb-3">
				<small>
					<span class="me-5"><strong>ID: </strong><span>${this.id}</span></span>
					<span><strong>Attribute ID: </strong><span>${this.pageAttributeID}</span></span>
				</small>
			</div>
		`;

		const valueInput = template.querySelector(".page-attribute-value-input");
		const removeButton = template.querySelector(".remove-button");
		const copyShortcodeToClipboard = template.querySelector(".copy-to-clipboard");
		const pageAttributeShortcodeValueInput = template.querySelector(".page-attribute-value-shortcode-input");

		pageAttributeShortcodeValueInput.value = `{{ page-attribute-value id="${this.id}" }}`;

		removeButton.addEventListener("click", () => {
			this.onRemoveButtonClicked();
		});

		valueInput.addEventListener("input", () => {
			this.value = valueInput.value;
		});

		copyShortcodeToClipboard.addEventListener("click", () => {
			this.onCopyShortcodeToClipboardClicked();
		});

		return template;
	}
}

export default PageAttribute;