import ProjectTagRenameModal from "../modals/RenameProjectTagModal.js";
import ProjectTagsManageModal from "../modals/ProjectTagsManageModal.js";
import ConfirmModal from "../../../utils/ConfirmModal.js";
import Endpoints from "../../../Endpoints.js";
import PageEditorStates from "../PageEditorStates.js";
import ProjectTagManager from "../ProjectTagManager.js";

class ProjectTagRow{

	/** @type {ProjectTagRow[]} */
	static rows = [];

	/**
	 * Changes the VISUALS ONLY of all rows to be shown as deselected.
	 */
	static showAllAsDeselected(){
		for (const row of ProjectTagRow.rows){
			row.showAsDeselected();
		}
	}

	/**
	 * Renames a row to a new name
	 * @param {int} projectTagID
	 * @param {string} newName
	 */
	static renameTag(projectTagID, newName){
		console.log(projectTagID);
		const row = ProjectTagRow.getProjectTagByTagID(parseInt(projectTagID));
		console.log(row);
		const label = row.dom.querySelector(".tag-label");
		label.textContent = newName
	}

	/**
	 * Fetches a ProjectTagRow by the tagID
	 * @param {int} tagID
	 * @returns {ProjectTagRow | null}
	 */
	static getProjectTagByTagID(tagID){
		for (const projectTagRow of ProjectTagRow.rows){
			if (projectTagRow.projectTagID === tagID){
				return projectTagRow;
			}
		}

		return null;
	}

	/**
	 * @param {int} projectTagID
	 * @param {string} tagLabel
	 * @param {boolean} isSelected
	 */
	constructor(projectTagID, tagLabel, isSelected){
		// Add self to the static collection
		ProjectTagRow.rows.push(this);
		this.projectTagID = projectTagID;
		this.tagLabel = tagLabel;
		this.row = this.getDOM();
		this.dom = this.row; // For consistency. Other classes use "dom"
		this.tagNotSelected = this.row.querySelector(".tag-not-selected");
		this.tagSelected = this.row.querySelector(".tag-selected");

		this.renameButton = this.row.querySelector(".rename-tag-button");
		this.deleteButton = this.row.querySelector(".delete-tag-button");

		this.row.addEventListener("click", e => {
			// Do not fire on columns that have buttons in them
			if (e.target.closest("column").querySelector("button") === null){
				this.onRowClicked();
			}
		});

		this.renameButton.addEventListener("click", e => {
			this.onRenameButtonClicked();
		});

		this.deleteButton.addEventListener("click", e => {
			this.onDeleteButtonClicked();
		});

		if (isSelected){
			this.showAsSelected();
			ProjectTagManager.addTagIDToSelected(this.projectTagID);
		}
	}

	showAsSelected(){
		this.row.classList.add("selected");
		this.tagNotSelected.style.display = "none";
		this.tagSelected.style.display = "inline";
	}

	showAsDeselected(){
		this.row.classList.remove("selected");
		this.tagNotSelected.style.display = "inline";
		this.tagSelected.style.display = "none";
	}

	onRowClicked(){
		if (this.tagNotSelected.style.display === "none"){
			// Linked, so unlink it
			this.showAsDeselected();
			ProjectTagManager.removeTagIDFromSelected(this.projectTagID);
		}else{
			// Unlinked, so link it
			this.showAsSelected();
			ProjectTagManager.addTagIDToSelected(this.projectTagID);
		}
	}

	async onDeleteButtonClicked(){
		const confirmModal = new ConfirmModal("confirm-delete-project-tag-modal");
		confirmModal.setConfirmButtonHTML(`
			<i class="bi bi-trash-fill"></i>
			<span>Delete</span>
		`);
		confirmModal.setTitle("Confirm project tag deletion");
		confirmModal.setContent("Deleting a project tag will also remove its association with any IPPs (project posts) that currently have it assigned.");

		ProjectTagsManageModal.hideModal();
		confirmModal.showModal();
		const wantsToDelete = await confirmModal.actionTaken();

		if (wantsToDelete){
			this.row.remove();
			fetch(`/uplift/page-editor/project-tag/${this.projectTagID}`, {
				credentials: "same-origin",
				cache: "no-cache",
				method: "delete"
			});
		}

		confirmModal.cleanup();
		ProjectTagsManageModal.showModal();
	}

	onRenameButtonClicked(){
		// BlogCategoryRenameForm.oldCategoryName = this.tagLabel;
		ProjectTagRenameModal.projectTagID = this.projectTagID;
		ProjectTagRenameModal.renameInput.value = this.tagLabel;
		ProjectTagsManageModal.hideModal();
		ProjectTagRenameModal.showModal();
	}

	/**
	 * @returns {HTMLElement}
	 */
	getDOM(){
		const template = document.createElement("project-tag-row");
		template.setAttribute("row-id", String(this.projectTagID));
		template.innerHTML = `
			<column>
				<span class="tag-not-selected"><i class="bi bi-dash-lg"></i></span>
				<span style="display:none;" class="tag-selected text-success"><i class="bi bi-check-lg"></i></span>
			</column>
			<column class="tag-label selector-column">
				<span class="tag-label">${this.tagLabel}</span>
			</column>
			<column>
				<div class="btn-group">
					<button type="button" class="rename-tag-button btn btn-sm btn-primary">
						<i class="bi bi-input-cursor-text"></i>
						<span>Rename</span>
					</button>
					<button type="button" class="delete-tag-button btn btn-sm btn-danger">
						<i class="me-0 bi bi-x-square-fill"></i>
					</button>
				</div>
			</column>
		`;

		document.querySelector("#manage-project-tag-rows").append(template);

		return template;
	}
}

export default ProjectTagRow;