import BlogReelShortcodeButton from "../shortcode-buttons/BlogReel.js";
import PageEditorStates from "../PageEditorStates.js";

class BlogReelForm{
	constructor(){
		// Ignore logic in this code for revision views
		if (PageEditorStates.viewType === "REVISION"){
			return;
		}

		this.form = document.querySelector("#blog-reel-form");

		/** @property {bool} isSubmitting The current status of submission */
		this.isSubmitting = false;

		this.form.addEventListener("submit", e => {
			e.preventDefault();
			this.onSubmit();
		});
	}

	/**
	* When the form is submitted
	*/
	async onSubmit(){
		const fData = new FormData(this.form);
		const submitButton = this.form.querySelector(`button[type="submit"]`);
		const errorContainer = this.form.querySelector(".submit-error");

		if (this.isSubmitting){
			return;
		}

		// Show the submit button as disabled
		submitButton.classList.add("disabled");
		errorContainer.style.display = "none";

		// Get all categories included and excluded
		const includedCategoryIDs = [];
		const excludedCategoryIDs = [];
		const includeCategoryButtons = this.form.querySelectorAll(".blog-reel-category-include-button");
		const excludeCategoryButtons = this.form.querySelectorAll(".blog-reel-category-exclude-button");

		for (let button of includeCategoryButtons){
			if (button.classList.contains("selected")){
				includedCategoryIDs.push(parseInt(button.getAttribute("data-category-id")));
			}
		}

		for (let button of excludeCategoryButtons){
			if (button.classList.contains("selected")){
				excludedCategoryIDs.push(parseInt(button.getAttribute("data-category-id")));
			}
		}

		// Get the number of articles per page and columns
		const numArticlesPerPage = parseInt(String(fData.get("articles-reel-num-articles-per-page")));
		const numColumns = parseInt(String(fData.get("articles-reel-num-columns")));

		if (isNaN(numArticlesPerPage)){
			this.onFormattedError("Number of articles is in a non-numerical format.");
			return;
		}

		if (isNaN(numColumns)){
			this.onFormattedError("Number of columns is in a non-numerical format.");
			return;
		}

		this.onSuccess(numArticlesPerPage, numColumns, includedCategoryIDs, excludedCategoryIDs);
	}

	/**
	* When the form succeeds without error
	*/
	onSuccess(numArticlesPerPage, numColumns, includedCategoryIDs, excludedCategoryIDs){
		const submitButton = this.form.querySelector(`button[type="submit"]`);
		this.isSubmitting = false;
		submitButton.classList.remove("disabled");
		
		// Inject the shortcode with the known parameters
		BlogReelShortcodeButton.injectShortcode(numArticlesPerPage, numColumns, includedCategoryIDs, excludedCategoryIDs);
	}

	/**
	* A readable error happened on the server in the JSON response.
	* (A status of -1 was read)
	* @param {string} err
	*/
	onFormattedError(err){
		const errorContainer = this.form.querySelector(".submit-error");
		const submitButton = this.form.querySelector(`button[type="submit"]`);

		this.isSubmitting = false;
		submitButton.classList.remove("disabled");
		errorContainer.textContent = err;
		errorContainer.style.display = "block";
	}
}

export default (new BlogReelForm);
