import ProjectTagRow from "../components/ProjectTagRow.js";
import ProjectTagManager from "../ProjectTagManager.js";
import PageEditorStates from "../PageEditorStates.js";

class NewProjectTagForm{
	constructor(){
		// Ignore logic in this code for revision views
		if (PageEditorStates.viewType === "REVISION"){
			return;
		}

		this.form = document.querySelector("#new-project-tag-form");

		/** @type {boolean} The current status of submission */
		this.isSubmitting = false;

		this.form.addEventListener("submit", e => {
			e.preventDefault();
			this.onSubmit();
		});
	}

	/**
	* When the form is submitted
	*/
	async onSubmit(){
		const fData = new FormData(this.form);
		const submitButton = this.form.querySelector(`button[type="submit"]`);
		const errorContainer = this.form.querySelector(".submit-error");

		if (this.isSubmitting){
			return;
		}

		// Show the submit button as disabled
		submitButton.classList.add("disabled");
		errorContainer.style.display = "none";

		const response = await fetch("/uplift/page-editor/project-tag", {
			cache:"no-cache",
			method:"POST",
			credentials:"same-origin",
			body:fData
		});

		/** @type {{status: number, error: ?string, projectTagID: ?number, projectTagLabel: ?string}} */
		const data = await response.json();

		if (data.status === 1){
			this.onSuccess(data.projectTagID, data.projectTagLabel);
		}else if (data.status === -1){
			this.onFormattedError(data.error);
		}
	}

	/**
	* When the form receives a success (status code 1) from
	* the server
	* @param {number} projectTagID
	* @param {string} projectTagLabel
	*/
	onSuccess(projectTagID, projectTagLabel){
		const submitButton = this.form.querySelector(`button[type="submit"]`);

		// Create a new row and select it
		const newRow = new ProjectTagRow(projectTagID, projectTagLabel);
		newRow.showAsSelected();
		ProjectTagManager.addTagIDToSelected(projectTagID);

		this.isSubmitting = false;
		submitButton.classList.remove("disabled");
		this.form.reset();
	}

	/**
	* A readable error happened on the server in the JSON response.
	* (A status of -1 was read)
	* @param {string} err
	*/
	onFormattedError(err){
		const errorContainer = this.form.querySelector(".submit-error");
		const submitButton = this.form.querySelector(`button[type="submit"]`);

		this.isSubmitting = false;
		submitButton.classList.remove("disabled");
		errorContainer.textContent = err;
		errorContainer.style.display = "block";
	}
}

export default (new NewProjectTagForm);
