import AvailableAttributeItem from "../components/AvailableAttributeItem.js";
import Scheduler from "../../../utils/Scheduler.js";
import PageEditorStates from "../PageEditorStates.js";
import PageAttribute from "../components/PageAttribute.js";

class AddAttributeModal{
	modal = document.querySelector("#add-attribute-modal");
	loader = document.querySelector("#add-attribute-loader");
	resultsContainer = document.querySelector("#available-attributes");
	noResultsMessage = document.querySelector("#no-available-attributes-message");
	isLoading = false;
	isAdding = false;

	constructor() {
		if (PageEditorStates.viewType === "STANDARD") {
			this.bsModal = bootstrap.Modal.getOrCreateInstance(this.modal);
			this.modal.addEventListener("show.bs.modal", () => {
				this.onShow();
			});
		}
	}

	onShow(){
		this.loadPageAttributes(PageEditorStates.currentPageID);
	}

	hideLoader(){
		this.loader.style.display = "none";
	}

	showLoader(){
		this.loader.style.display = null;
	}

	/**
	 * Fetches and renders the available attributes for this pageID.
	 * @param pageID
	 */
	async loadPageAttributes(pageID){

		if (this.isLoading){
			return;
		}

		this.isLoading = true;
		AvailableAttributeItem.clearAll();
		this.showLoader();
		this.noResultsMessage.style.display = "none";
		this.resultsContainer.style.display = "none";

		const response = await fetch(`/uplift/page-editor/page-attributes/${pageID}`, {
		    body:null,
		    method:"GET",
		    cache:"no-cache",
		    credentials:"same-origin"
		});

		await Scheduler.wait(250);
		this.resultsContainer.style.display = null;

		let data;
		try{
		    /** @type {{status: int, error: ?string, availableAttributes: Object[]}} **/
		    data = await response.json();
		}catch(jsonSyntaxError){
		    alert("The server responded with invalid JSON.");
			this.isLoading = false;
			this.hideLoader();
		    return;
		}

		if (data.status === 1){

			// Filter out attributes that are already assigned to this page
			const attributeIDsToIgnore = [];

			for (const pageAttributeValue of PageAttribute.cache){
				attributeIDsToIgnore.push(pageAttributeValue.pageAttributeID);
			}

			// Only keep entries that are not found in the attribute IDs to ignore
			data.availableAttributes = data.availableAttributes.filter(pageAttribute => {
				return attributeIDsToIgnore.indexOf(pageAttribute.id) === -1;
			});

			if (data.availableAttributes.length > 0) {
				for (const pageAttribute of data.availableAttributes) {
					const newAvailableAttribute = new AvailableAttributeItem(pageAttribute.id, pageAttribute.name);
					newAvailableAttribute.render();
				}
			}else{
				this.noResultsMessage.style.display = null;
			}
		}else if (data.status === -1){

		}

		this.isLoading = false;
		this.hideLoader();
	}

	async addAttributeToPage(attributeID, pageID){

		if (this.isAdding){
			return;
		}

		this.isAdding = true;
		this.resultsContainer.style.display = "none";
		this.showLoader();

		const fData = new FormData();
		fData.set("attributeID", attributeID);

		const response = await fetch(`/uplift/page-editor/page-attributes/${pageID}`, {
		    body:fData,
		    method:"PUT",
		    cache:"no-cache",
		    credentials:"same-origin"
		});

		let data;
		try{
		    /** @type {{status: int, error: ?string, newPageAttribute: Object}} **/
		    data = await response.json();
		}catch(jsonSyntaxError){
		    alert("The server responded with invalid JSON.");
			this.isAdding = false;
			this.hideLoader();
			this.resultsContainer.style.display = null;
		    return;
		}

		if (data.status === 1){
			this.bsModal.hide();

			const newPageAttribute = new PageAttribute(
				data.newPageAttribute.id,
				data.newPageAttribute.name,
				data.newPageAttribute.value,
				data.newPageAttribute.pageAttributeID,
			)

			newPageAttribute.render();

			document.querySelector("#no-attributes-message-container").style.display = "none";
		}else if (data.status === -1){
			alert(data.error);
			this.resultsContainer.style.display = null;
		}

		this.isAdding = false;
		this.hideLoader();
	}

	/**
	 * @param {AvailableAttributeItem} availableAttributeItem
	 */
	onAvailableAttributeClicked(availableAttributeItem){
		this.addAttributeToPage(availableAttributeItem.id, PageEditorStates.currentPageID);
	}
}

export default new AddAttributeModal();