import PageEditorStates from "../PageEditorStates.js";
import EditorEndpoints from "../../EditorEndpoints.js";
import EditorWrapper from "../EditorWrapper.js";
import FilterInsertGalleryModal from "../forms/FilterInsertGalleryModal.js";
import BSModal from "../../../_external/bootstrap/modal.js";
import Scheduler from "../../../utils/Scheduler.js";

class InsertGalleryModal{
	constructor(){

		// Ignore logic in this code for revision views
		if (PageEditorStates.viewType === "REVISION"){
			return;
		}

		if (PageEditorStates.currentPageType !== ""){
			this.modal = document.querySelector("#insert-gallery-modal");
			this.bsModal = new BSModal(this.modal);
			this.loadingIconContainer = document.querySelector("#insert-gallery-loading-container");
			this.renderedGalleriesContainer = document.querySelector("#galleries-container");
			this.noGalleriesMessage = document.querySelector("#no-galleries-message");
			this.activatorButton = document.querySelector("#insert-gallery-button");

			/** @property {bool} isFetching Whether or not galleries are currently being fetched */
			this.isFetching = false;

			this.activatorButton.addEventListener("click", () => {
				this.onActivatorButtonClicked();
			});
		}
	}

	/**
	* Filters current gallery items by text
	* @param {string} text
	*/
	filterByText(text){
		text = text.toLowerCase();
		const items = this.renderedGalleriesContainer.querySelectorAll(".insert-gallery-modal-item");
		for (let item of items){
			const name = item.querySelector(".gallery-name-span").textContent.trim().toLowerCase();
			if (text !== ""){
				if (name.includes(text)){
					item.style.display = null;
				}else{
					item.style.display = "none";
				}
			}else{
				item.style.display = null;
			}
		}
	}

	/**
	* Fetches new galleries and renders them
	*/
	async fetchAndRenderGalleries(){

		if (this.isFetching){
			return;
		}

		this.loadingIconContainer.style.display = "flex";
		this.renderedGalleriesContainer.innerHTML = "";
		this.noGalleriesMessage.style.display = "none";
		this.isFetching = true;
		this.renderedGalleriesContainer.style.display = "none";
		document.querySelector("#galleries-modal-headers-container").style.display = "none";

		const response = await fetch(`/uplift/gallery-manager/galleries`, {
			credentials:"same-origin",
			cache:"no-cache"
		});

		/**
		 * @type {{status: int, error: ?string, galleries: Object[]}}
		 */
		const data = await response.json();
		if (data.status === 1){
			const galleries = data.galleries;
			if (galleries.length > 0){
				for (/** @type {{id: int, name: string, creationTime: int, numImagesInGallery: int}} */ let galleryItem of galleries){
					const dom = this.buildDOM(galleryItem.id, galleryItem.name, galleryItem.numImagesInGallery);
					this.renderedGalleriesContainer.append(dom);
				}
			}else{
				this.noGalleriesMessage.style.display = "block";
			}
		}else{
			// Can't happen for this API endpoint
		}

		// Artificial wait time
		await Scheduler.wait(330);

		document.querySelector("#galleries-modal-headers-container").style.display = null;
		this.renderedGalleriesContainer.style.display = null;
		this.loadingIconContainer.style.display = "none";
		this.isFetching = false;
	}

	/**
	 *
	 * @param {int} galleryID
	 * @param {string} galleryName
	 * @param {int} numItemsInGallery
	 * @returns {HTMLDivElement}
	 */
	buildDOM(galleryID, galleryName, numItemsInGallery){
		const container = document.createElement("div");
		container.classList.add("insert-gallery-modal-item");
		container.classList.add("gallery-modal-column-headers");
		container.setAttribute("data-gallery-id", String(galleryID));
		container.innerHTML = `
			<div class="icon-column">
				<div class="corner-icon">
					<i class="bi bi-images"></i>
				</div>
			</div>
			<div class="gallery-name-column">
				<span class="gallery-name-span">${galleryName}</span>
			</div>
			<div class="num-items-in-gallery-column">
				<span>${numItemsInGallery}</span>
			</div>
		`;

		container.addEventListener("click", () => {
			let galleryName_sanitized = galleryName;

			// Remove double quotes, {, and } characters
			galleryName_sanitized = galleryName_sanitized.replace("\"", "");
			galleryName_sanitized = galleryName_sanitized.replace("{", "");
			galleryName_sanitized = galleryName_sanitized.replace("}", "");

			const shortcode = `{{ gallery id="${galleryID}" gallery-name="${galleryName_sanitized}" }}`;
			EditorWrapper.insertTextAtCursor(shortcode);
			this.hideModal();
			EditorWrapper.focusEditor(shortcode);
		});

		return container;
	}

	/**
	* When the activator button for this modal is clicked
	*/
	onActivatorButtonClicked(){
		this.fetchAndRenderGalleries();
		this.showModal();
	}

	/**
	* Forces the modal to be shown
	*/
	showModal(){
		this.bsModal.show();
	}

	/**
	* Forces the modal to be hidden
	*/
	hideModal(){
		this.bsModal.hide();
	}
}

export default (new InsertGalleryModal);
