import BodyEditor from "../components/BodyEditor.js";
import HeadEditor from "../components/HeadEditor.js";
import Breadcrumb from "../components/Breadcrumb.js";
import SuccessBox from "../components/SuccessBox.js";
import Loader from "../components/Loader.js";
import ImportWebsiteContainer from "../components/ImportWebsiteContainer.js";
import Scheduler from "../../utils/Scheduler.js";

class BuildPageForm{
	form = document.querySelector("#page-build-form");
	isProcessing = false;

	constructor(){
		this.form.addEventListener("submit", e => {
			e.preventDefault();
			this.onSubmit();
		});
	}

	resetFields(){
		const pageNameInput = document.querySelector('#page-name-input');
		const pageRouteInput = document.querySelector('#page-route-input');
		const pageCityNameInput = document.querySelector("#page-city-name-input");
		const pageCityUrlInput = document.querySelector("#page-city-url-input");
		pageNameInput.value = "";
		pageRouteInput.value = "";
		pageCityNameInput.value = "";
		pageCityUrlInput.value = "";
	}

	async onSubmit(){
		if (this.isProcessing){
			return;
		}

		this.isProcessing = true;
		Loader.show();
		ImportWebsiteContainer.hide();

		const fData = new FormData(this.form);
		const crumbs = [];

		// Add the body, head, and breadcrumbs
		fData.set("page-body", BodyEditor.aceEditorExtension.ace.getSession().getValue());
		fData.set("page-head", HeadEditor.aceEditorExtension.ace.getSession().getValue());
		fData.set("imported-from-url", document.querySelector("#import-url-input").value);

		for (const crumb of Breadcrumb.cache){
			crumbs.push({
				"label": crumb.labelInput.value.trim(),
				"uri": crumb.uriInput.value.trim()
			});
		}

		fData.set("breadcrumbs", JSON.stringify(crumbs));

		const response = await fetch(`/uplift/external-importer/build`, {
			body:fData,
		    method:"POST",
		    cache:"no-cache",
		    credentials:"same-origin"
		});

		let data;
		try{
		    /** @type {{status: int, error: ?string, newPageURL: string}} **/
		    data = await response.json();
		}catch(jsonSyntaxError){
		    alert("The server responded with invalid JSON.");
		    return;
		}

		await Scheduler.wait(500);
		Loader.hide();

		if (data.status === 1){
			SuccessBox.setNewPageURL(data.newPageURL);
			SuccessBox.show();
		}else if (data.status === -1){
			alert(data.error);
			ImportWebsiteContainer.show();
		}

		this.isProcessing = false;
	}
}

export default new BuildPageForm()