import BSModal from "../../_external/bootstrap/modal.js";
import Endpoints from "../../Endpoints.js";
import GalleryMember from "../components/GalleryMember.js";
import RenameGalleryModal from "./RenameGalleryModal.js";
import GalleryRow from "../components/GalleryRow.js";
import ConfirmModal from "../../utils/ConfirmModal.js";
import WindowManager from "../../image-manager/WindowManager.js";

class ManageGalleryModal{

	/**
	 * @type {ImageGallery}
	 */
	imageGallery;

	/**
	 * @param {ImageGallery} imageGallery
	 */
	constructor(imageGallery){
		this.imageGallery = imageGallery;
		this.isProcessing = false;
		this.modal = document.querySelector("#manage-gallery-modal");
		this.bsModal = new BSModal(this.modal);
		this.titleContainer = this.modal.querySelector(".modal-title");
		this.membersContainer = this.modal.querySelector("#manage-gallery-members-container");
		this.selectImagesButton = this.modal.querySelector(".manage-gallery-select-images");
		this.renameGallery = this.modal.querySelector(".manage-gallery-rename-gallery");
		this.deleteGallery = this.modal.querySelector(".manage-gallery-delete-gallery");
		this.currentGalleryID = null;

		if (imageGallery.imageManager === null){
			this.selectImagesButton.style.display = "none";
		}

		this.selectImagesButton.addEventListener("click", () => {
			this.onSelectImagesClicked();
		});

		this.renameGallery.addEventListener("click", () => {
			this.onRenameButtonClicked();
		});

		this.deleteGallery.addEventListener("click", () => {
			this.onDeleteButtonClicked();
		});
	}

	async onDeleteButtonClicked(){
		const galleryRow = GalleryRow.getByGalleryID(this.currentGalleryID);
		if (galleryRow !== null) {
			this.hideModal();

			const deleteGalleryModal = new ConfirmModal("delete-gallery-confirm-modal");
			deleteGalleryModal.setTitle("Delete this gallery?");
			deleteGalleryModal.setConfirmButtonText("Delete");
			deleteGalleryModal.setContent("This will remove this gallery. The image files themselves will not be deleted - just the gallery itself. Any pages using this gallery will no longer show the gallery's images.");
			deleteGalleryModal.showModal();

			const didAccept = await deleteGalleryModal.actionTaken();
			deleteGalleryModal.hideModal();

			if (didAccept){
				// Delete request
				const response = await fetch(`${Endpoints.galleries.deleteGallery}/${this.currentGalleryID}`, {
					method:"DELETE",
					cache:"no-cache",
					credentials:"same-origin"
				});

				let data;
				try{
					data = await response.json();
				}catch(jsonSyntaxError){
					alert("Server responded with invalid JSON.");
					return;
				}

				if (data.status === 1){
					const row = GalleryRow.getByGalleryID(this.currentGalleryID);
					row.dom.remove();
				}else if (data.status === -1){
					alert(data.error);
				}
			}else{
				// Show the manager modal again
				this.showModal();
			}
		}else{
			alert(`Fatal error: Cannot find GalleryRow instance with gallery ID ${this.currentGalleryID}`);
		}
	}

	async onRenameButtonClicked(){
		const galleryRow = GalleryRow.getByGalleryID(this.currentGalleryID);
		if (galleryRow !== null) {
			this.imageGallery.renameGalleryModal.modal.querySelector(`[name="gallery-name"]`).value = galleryRow.name;
			this.hideModal();
			this.imageGallery.renameGalleryModal.galleryIDInput.value = String(this.currentGalleryID);
			this.imageGallery.renameGalleryModal.showModal();

			// Wait for it to be hidden
			await new Promise(resolve => {
				this.imageGallery.renameGalleryModal.modal.addEventListener("hide.bs.modal", () => {
					resolve();
				});
			});

			this.showModal();
		}else{
			alert(`Fatal error: Cannot find GalleryRow instance with gallery ID ${this.currentGalleryID}`);
		}
	}

	async onSelectImagesClicked(){
		this.hideModal();
		this.imageGallery.imageManager.imageManagerState.setImageMode(this.imageGallery.imageManager.imageManagerState.IMAGE_MODES.SELECT);
		this.imageGallery.imageManager.windowManager.show(WindowManager.WINDOWS.MAIN);
		/** @type {ImageComponent[]} */
		const imageComponents = await this.imageGallery.imageManager.imagesChosen;
		this.showModal();
		this.imageGallery.imageManager.windowManager.closeImageManager();

		// PUT the images into the gallery so that IDs can be obtained
		const payload = {"images":[]};
		for (const component of imageComponents){
			payload.images.push({
				position:0,
				altText:"",
				imageURI:component.uri,
				thumbURI:component.thumbURI
			});
		}

		const headers = new Headers();
		headers.set("content-type", "application/json");
		const response = await fetch(`${Endpoints.galleries.addMembersToGallery}/${this.currentGalleryID}/members`, {
			credentials:"same-origin",
			body:JSON.stringify(payload),
			method:"PUT",
			headers:headers
		});

		if (response.status === 200) {
			let data;
			try {
				data = await response.json();
			} catch (jsonSyntaxError) {
				alert("Server sent back invalid JSON when adding images.");
				return;
			}

			if (data.status === 1) {
				// Just reload them all, but data _does_ contain `newMembers` for the ones that got added
				await this.loadMembers(this.currentGalleryID);
				await this.imageGallery.loadGalleries();
			} else if (data.status === -1) {
				alert(data.error);
			}
		}else{
			let data;
			try {
				/** @type {error: string, status: int} */
				data = await response.json();
				alert(data.error);
			} catch (jsonSyntaxError) {
				alert("Server sent back invalid JSON when adding images.");
			}

			// Reload this gallery and other to reflect that some of the images may have been added even if an error occured
			await this.loadMembers(this.currentGalleryID);
			await this.imageGallery.loadGalleries();
		}
	}

	/**
	 * @param {int} galleryID
	 * @returns {Promise<void>}
	 */
	async loadMembers(galleryID){
		if (this.isProcessing){
			return;
		}

		// Clear current members
		GalleryMember.clearMembers();
		this.isProcessing = true;

		const response = await fetch(`${Endpoints.galleries.fetchGalleryMembers}/${galleryID}/members`, {
			credentials:"same-origin",
			cache:"no-cache",
		});
		if (response.status === 200){
			/** @type {{status: int, error: ?string, members: Object}} */
			let data;
			try {
				data = await response.json();
			}catch(jsonSyntaxError){
				alert("Response was not in valid JSON.");
				this.isProcessing = false;
				return;
			}

			if (data.status === 1){
				// Render each
				const members = data.members;

				// Sort by position
				members.sort( (a,b) => {
					return a.position > b.position ? 1 : -1;
				});

				for (/** @type {{id: int, galleryID, imageURI: string, thumbURI: string, altAttribute: string, position: int}} */const member of members){
					new GalleryMember(
						member.id,
						member.imageURI.match(/\/([^\/]+)$/)[1],
						member.thumbURI,
						member.altAttribute,
					);
				}
			}else if (data.status === -1){
				alert(data.error);
			}
		}else{
			alert("Internal server error when fetching gallery members.");
		}

		this.isProcessing = false;
	}

	/**
	* Forces the modal to be shown
	*/
	showModal(){
		this.bsModal.show();
	}

	/**
	* Forces the modal to be hidden
	*/
	hideModal(){
		this.bsModal.hide();
	}
}

export default ManageGalleryModal;
