import ImageComponent from "./components/ImageComponent.js";
import BreadcrumbHandler from "./BreadcrumbHandler.js";
import Scheduler from "../utils/Scheduler.js";

class DirectoryImagesFetcher{

	/**
	 * @type {ImageManager}
	 */
	imageManager;

	/**
	 * @type {string}
	 */
	directory;

	isBusy = false;

	/**
	 * @param {ImageManager} imageManager
	 * @param {string} directory
	 */
	constructor(imageManager, directory){
		this.imageManager = imageManager;
		this.directory = directory;
	}

	/**
	 * @param {string} directory
	 */
	setDirectory(directory){
		this.directory = directory;
	}

	/**
	 * Fetches images in the current directory
	 */
	async fetchAndRender(){

		// Ignore if things are being fetched currently
		if (this.isBusy){
			return;
		}

		// Clear the current hash map of DOMs linked to ImageComponents
		ImageComponent.imageComponentDOMLookupHash = {};

		this.imageManager.windowManager.mainWindow.hideNoImagesMessage();
		this.imageManager.windowManager.mainWindow.showImagesLoading();

		this.isBusy = true;

		// clear current images
		this.imageManager.getImageComponentsContainer().innerHTML = "";

		const urlParams = new URLSearchParams();
		urlParams.set("directory", this.directory);

		const response = await fetch(`/uplift/image-manager/fetch-directory-images?${urlParams.toString()}`, {
			credentials:"same-origin",
			cache:"no-cache"
		});

		/** @type {{status:int, error:?string, imageFiles:Object[]}} */
		const data = await response.json();
		if (data.status === 1){
			const imageFiles = data.imageFiles;
			if (imageFiles.length > 0) {
				for (const imageFileData of imageFiles) {
					if(imageFileData.fileExtension !== "gitkeep") {
						const imageComponent = new ImageComponent(this.imageManager, imageFileData);
						imageComponent.getDOM();
						imageComponent.render();
					}
				}
			}else{
				this.imageManager.windowManager.mainWindow.showNoImagesMessage();
			}
		}

		// Clear the image filter, if any
		this.imageManager.filterImages.queryInput.value = "";

		this.imageManager.imageManagerState.currentDirectory = this.directory;
		this.imageManager.breadcrumbsHandler.refreshLocation();
		this.imageManager.windowManager.mainWindow.hideImagesLoading();
		this.isBusy = false;
	}
}

export default DirectoryImagesFetcher;