import CropWindow from "./windows/CropWindow.js";
import CloneWindow from "./windows/CloneWindow.js";
import ResizeWindow from "./windows/ResizeWindow.js";
import MainWindow from "./windows/MainWindow.js";

/**
 * Responsible for managing the current modal window visibility and swapping between them.
 */
class WindowManager{

	/**
	 *
	 * @type {{RESIZE: number, MAIN: number, CROP: number, CLONE: number, QUICK_GALLERY: number}}
	 */
	static WINDOWS = {
		MAIN: 0,
		RESIZE: 1,
		CROP: 2,
		CLONE: 3,
		QUICK_GALLERY: 4
	};

	/**
	 * @type {ImageManager}
	 */
	imageManager;

	/**
	 * @type {CloneWindow}
	 */
	cloneWindow;

	/**
	 * @type {CropWindow}
	 */
	cropWindow;

	/**
	 * @type {MainWindow}
	 */
	mainWindow;

	/**
	 * @type {ResizeWindow}
	 */
	resizeWindow;

	/**
	 * @param {ImageManager} imageManager
	 */
	constructor(imageManager){
		this.imageManager = imageManager;

		this.container = imageManager.container;

		/** @type {?HTMLElement} */
		this.currentWindow = null;

		/** @type {null|number} */
		this.currentWindowNumber = null;

		/** @type {?HTMLElement} */
		this.lastMouseDownElement = null;

		this.cloneWindow = new CloneWindow(imageManager);
		this.cropWindow = new CropWindow(imageManager);
		this.mainWindow = new MainWindow(imageManager);
		this.resizeWindow = new ResizeWindow(imageManager);

		/**
		 * The index of each window should correspond to the static WINDOWS property
		 */
		this.windows = [
			this.imageManager.container.querySelector("image-manager-window"),
			this.imageManager.container.querySelector(".im-resize-window"),
			this.imageManager.container.querySelector(".im-crop-window"),
			this.imageManager.container.querySelector(".im-clone-window"),
			this.imageManager.container.querySelector(".im-quick-gallery-menu")
		];

		this.container.addEventListener("mousedown", e => {
			this.lastMouseDownElement = e.target;
		});

		this.container.addEventListener("mouseup", e => {
			if (e.target === this.container) {
				if (this.lastMouseDownElement === e.target) {
					if (this.currentWindow === this.windows[WindowManager.WINDOWS.MAIN]) {
						this.closeImageManager();
					} else {
						this.show(WindowManager.WINDOWS.MAIN);
					}
				}
			}
		});
	}

	/**
	 * Transitions to the provided window enum
	 * @param {number} windowID
	 */
	show(windowID){
		/** @type {HTMLElement} */
		const element = this.windows[windowID];

		// Check if the container is visible
		if (this.container.style.display === "none"){
			this.container.style.display = "block";
		}

		if (this.currentWindow){
			this.currentWindow.style.display = "none";
		}else{
			// There is no current window, so the image manager is being open for the first time
			this.imageManager.imagesChosen = new Promise(resolve => {

				// Leak the resolver out of scope, so we can call it from other functions
				this.imageManager.imagesChosenPromiseResolveFunction = resolve;
			});
		}

		element.style.display = "flex";
		this.currentWindow = element;
		this.currentWindowNumber = windowID;
	}

	/**
	 * Closes the entire image manager
	 */
	closeImageManager(){
		if (this.currentWindow) {
			this.currentWindow.style.display = "none";
			this.currentWindow = null;
			this.currentWindowNumber = null;

			if (this.container.style.display !== "none") {
				this.container.style.display = "none";

				// Clear promise stuff, resolve the promise if it still exists
				if (this.imageManager.imagesChosen !== null) {
					// Resolve without a selection
					this.imageManager.imagesChosenPromiseResolveFunction(null);
				}

				this.imageManager.imagesChosen = null;
				this.imageManager.imagesChosenPromiseResolveFunction = null;
			}
		}
	}
}

export default WindowManager;