import ContextMenu from "../../utils/ContextMenu.js";
import SidebarFolderComponent from "./SidebarFolderComponent.js";
import ImageManagerState from "../ImageManagerState.js";

class FoldersSidebar {
	/**
	 * @type {ImageManager}
	 */
	imageManager = null;

	/**
	 * @param {ImageManager} imageManager
	 */
	constructor(imageManager){
		this.imageManager = imageManager;
		this.sidebar = document.querySelector("im-window-body .folder-column");

		// Also allow the folders container to be right-clicked to create new folders
		// in the root.
		this.overflowColumn = document.querySelector("im-window-body .folders-container");

		this.sidebar.addEventListener("contextmenu", e => {
			if (e.target === this.sidebar || e.target === this.overflowColumn){
				e.preventDefault();
				e.stopPropagation();
				this.onContextMenu(e.pageX, e.pageY);
			}
		});
	}

	/**
	 *
	 * @param {number} posX
	 * @param {number} posY
	 */
	onContextMenu(posX, posY){
		const cm = new ContextMenu(posX, posY);
		cm.buildDOM();
		const addFolderButton = cm.addButton(`
			<i class="bi bi-folder-plus"></i>
			<span>New Folder</span>
		`);

		addFolderButton.addEventListener("click", () => {
			cm.cleanup();
			this.addNewFolder(this.imageManager.imageManagerState.baseImagesDirectory)
		});
		cm.render();
	}

	/**
	 * Adds a new folder to the parent location
	 * and enables naming of the folder
	 * @param {string} parentDirectory
	 */
	async addNewFolder(parentDirectory){
		const fData = new FormData();
		fData.set("parent-directory", parentDirectory);
		const response = await fetch(`/uplift/image-manager/new-directory`, {
			credentials:"same-origin",
			method:"post",
			cache:"no-cache",
			body:fData
		});

		/** @type {{status: number, newFolderName: string}} */
		const data = await response.json();
		const newFolderName = data.newFolderName;

		const sidebarFolder = new SidebarFolderComponent(this.imageManager, {
			directoryName: newFolderName,
			fullDirectoryPath: `${parentDirectory}/${newFolderName}`,
			childDirectories:[]
		});
		sidebarFolder.getDOM();
		sidebarFolder.render();
		sidebarFolder.enableRenaming(true);
	}
}

export default FoldersSidebar;