import ContextMenu from "../../utils/ContextMenu.js";
import WindowManager from "../WindowManager.js";
import Scheduler from "../../utils/Scheduler.js";

class ImageComponent{

	/**
	 * @type {ImageManager}
	 */
	imageManager;

	/** @type {number} */
	static currentImageComponentHashNumber = 0;

	/** @type {{int: ImageComponent}} */
	static imageComponentDOMLookupHash = {};

	/**
	 * @param {HTMLElement} dom
	 * @returns {?ImageComponent}
	 */
	static getComponentByDOM(dom){
		const hashID = parseInt(dom.getAttribute("hash-lookup-id"));
		if (dom.tagName.toLowerCase() === "im-image"){
			if (hashID in ImageComponent.imageComponentDOMLookupHash){
				return ImageComponent.imageComponentDOMLookupHash[hashID];
			}
		}

		return null;
	}

	/**
	 * @param {ImageManager} imageManager
	 * @param {Object} imageFileData From the backend API
	 */
	constructor(imageManager, imageFileData){
		this.imageManager = imageManager;
		this.container = imageManager.getImageComponentsContainer();
		this._hashIndex = ++ImageComponent.currentImageComponentHashNumber;
		this.fileName = imageFileData.fileName;
		this.fullFilePath = imageFileData.fullFilePath;
		this.fileNameWithoutExtension = imageFileData.fileNameWithoutExtension;
		this.fileSize = imageFileData.fileSize;
		this.fileSizeHumanReadable = imageFileData.fileSizeHumanReadable;
		this.fileExtension = imageFileData.fileExtension.toLowerCase();
		this.thumbURI = "thumbURI" in imageFileData ? imageFileData.thumbURI : null;
		this.uri = imageFileData.uri;
		this.imageWidth = "gdHelper" in imageFileData ? imageFileData.gdHelper.width : null;
		this.imageHeight = "gdHelper" in imageFileData ? imageFileData.gdHelper.height : null;
		this.imageTypePHPGDInteger = "gdHelper" in imageFileData ? imageFileData.gdHelper.imageType : null;
		this.fallbackImage = imageFileData.fallbackImage;

		/**
		 * The last time the mouse clicked on this element. Helps to determine double clicks
		 * @type {number}
		 */
		this.lastMouseClickTime = 0;

		/**
		 * The <im-image> element for this component
		 * @type {HTMLElement}
		 */
		this.dom = null;

		/**
		 * Flag for whether a renamed submission is processing
		 * @type {boolean}
		 */
		this.isRenameProcessing = false;
		ImageComponent.imageComponentDOMLookupHash[this._hashIndex] = this;
	}

	/**
	 * Tries to use the image name to create an alt text
	 * @return {string}
	 */
	tryToCreateAlt(){
		let alt = this.fileNameWithoutExtension;
		alt = alt.replace(new RegExp("-", "g"), " ")
			.replace(new RegExp("_", "g"), " ")
			.replace(new RegExp("\\.", "g"), " ")
			.replace(new RegExp("\\+", "g"), " ")
			.replace(new RegExp("\\)", "g"), "")
			.replace(new RegExp("\\(", "g"), "")
			.replace(/\s{2,}/g, " ") // More than 1 space to a single space
			.replace(/[\s\d]+$/, "") // Remove the ending if it's just spaces and numbers
			.trim()
			.toLowerCase();

		return alt.substring(0,1).toUpperCase() + alt.substring(1);
	}

	getDOM(){
		const template = document.createElement("im-image");
		template.setAttribute("tabindex", "0");
		template.setAttribute("draggable", "true");
		template.setAttribute("image-type", this.fileExtension.toLowerCase());

		// Helps find this image component by its DOM
		template.setAttribute("hash-lookup-id", this._hashIndex);
		template.innerHTML = `
			<div class="image-rename-container" style="display:none;">
				<form class="image-rename-form">
					<input type="text" class="form-control" class="image-rename-input-field" value="${this.fileNameWithoutExtension}">
				</form>
			</div>
			<div class="image-name-contents">
				<i class="bi bi-image-fill" image-type="${this.fileExtension.toLowerCase()}"></i>
				<span class="image-full-file-name">
					<span class="image-file-name">${this.fileNameWithoutExtension}</span>.<span class="image-file-extension">${this.fileExtension}</span>
				</span>
			</div>
			<div class="image-type-container">
				<span class="image-type">${this.fileExtension.toUpperCase()}</span>
			</div>
			<div class="file-size-container">
				<span class="image-file-size">${this.fileSizeHumanReadable}</span>
			</div>
		`;

		template.addEventListener("click", e => {
			this.onClicked(e);
		});

		template.addEventListener("focusin", async () => {
			this.imageManager.metaDataPane.hideImageData();
			this.imageManager.metaDataPane.hidePreview(this.thumbURI);
			this.imageManager.windowManager.mainWindow.showImagePreviewLoader();
			await this.loadFileDimensions();
			await Scheduler.wait(50);
			this.imageManager.windowManager.mainWindow.hideImagePreviewLoader();

			// Not all images have thumbs - thumbs don't have thumbs, for instance
			if (this.thumbURI !== null) {
				this.imageManager.metaDataPane.showPreview(this.thumbURI);
			}else{
				this.imageManager.metaDataPane.showPreview(this.uri);
			}

			this.imageManager.metaDataPane.showImageData(this.fileSizeHumanReadable, this.fileExtension, this.imageWidth, this.imageHeight);
		});


		template.addEventListener("dragstart", e => {
			if (this.isRenameFormShown()){
				return true;
			}else {
				if (this.imageManager.imageManagerState.isDraggingComponent === false) {

					// Add to select if it isn't there
					if (!this.imageManager.selectionListener.isImageComponentInSelection(this)){
						this.imageManager.selectionListener.addImageComponentToSelection(this);
					}

					this.imageManager.imageManagerState.isDraggingComponent = true;
					this.imageManager.imageManagerState.currentlyDraggedImageComponent = this;
					e.dataTransfer.setData("text/plain", this.fullFilePath);
					e.dataTransfer.setData("cms/droptype", "image");
				}
			}
		});

		template.addEventListener("contextmenu", e => {

			// Check if the rename field is currently open
			if (this.isRenameFormShown()){
				return;
			}

			e.preventDefault();
			e.stopPropagation();

			this.onContextMenu(e);
		});

		this.dom = template;
		this.hookEvents();

		return template;
	}

	/**
	 * @date 3/16/2023
	 * Added to allow image dimensions to not be available when fetching initial image directories. Loading the
	 * dimensions was causing significant load times.
	 *
	 * Fetches the image dimensions from the server and loads them into the imageWidth and imageHeight properties,
	 * if they are currently null and the image extension is not SVG.
	 */
	async loadFileDimensions(){
		if (this.imageWidth === null && this.imageHeight === null && this.fileExtension.toLowerCase() !== "svg") {
			const urlParams = new URLSearchParams();
			urlParams.set("image-path", this.fullFilePath);
			const response = await fetch(`/uplift/image-manager/get-image-dimensions?${urlParams.toString()}`, {
				method: "GET",
				cache: "no-cache",
				credentials: "same-origin"
			});

			let data;
			try {
				/** @type {{status: int, error: ?string, width: int, height: int}} **/
				data = await response.json();
			} catch (jsonSyntaxError) {
				alert("The server responded with invalid JSON.");
				return;
			}

			if (data.status === 1) {
				this.imageWidth = data.width;
				this.imageHeight = data.height;
			} else if (data.status === -1) {

			}
		}
	}

	/**
	 * When this image component is clicked
	 * @param {PointerEvent} e
	 */
	onClicked(e){
		const now = (new Date()).getTime();
		const differenceSinceLastClick = now - this.lastMouseClickTime;
		const selectionListener = this.imageManager.selectionListener;

		if (differenceSinceLastClick > 400 && differenceSinceLastClick < 700){
			if (!this.isRenameFormShown()) {
				selectionListener.clearImageComponentSelection();
				this.toggleRenameField();
			}
		}else{
			if(selectionListener.hasImageComponentsSelected()){
				// Add or overwrite?
				if (e.ctrlKey) {
					if (selectionListener.isImageComponentInSelection(this)) {
						selectionListener.removeImageComponentFromSelection(this);
					}else{
						selectionListener.addImageComponentToSelection(this);
					}
				}else if (e.shiftKey){
					selectionListener.addImageComponentsToSelectionFromRange(
						selectionListener.currentImageComponentSelection[0],
						this
					);
				}else {
					// Neither, clear selection then add this one
					selectionListener.clearImageComponentSelection();
					selectionListener.addImageComponentToSelection(this);
				}
			}else{
				selectionListener.addImageComponentToSelection(this);
			}

			if (differenceSinceLastClick <= 400){
				// Double click
				this.imageManager.imagesChosenPromiseResolveFunction(selectionListener.currentImageComponentSelection);
			}
		}

		this.lastMouseClickTime = now;
	}

	/**
	 * When the context menu is requested on this image component
	 * @param {MouseEvent} e
	 */
	onContextMenu(e){
		const selectionListener = this.imageManager.selectionListener;
		if (selectionListener.hasImageComponentsSelected()){
			if (selectionListener.currentImageComponentSelection.length === 1){
				// Only one
				this.onSingleSelectionContextMenu(e);
			}else{
				if (selectionListener.isImageComponentInSelection(this)) {
					this.onMultipleSelectionContextMenu(e);
				}else{
					// This function will clear the selection and select this component
					this.onSingleSelectionContextMenu(e);
				}
			}
		}else{
			this.onSingleSelectionContextMenu(e);
		}
	}

	/**
	 * When a context menu is requested and this is the only ImageComponent
	 * in the selection
	 * @param {MouseEvent} e
	 */
	onSingleSelectionContextMenu(e){
		const selectionListener = this.imageManager.selectionListener;

		// Clear selection if there is one and select this
		selectionListener.clearImageComponentSelection();
		selectionListener.addImageComponentToSelection(this);

		const cm = new ContextMenu(e.pageX, e.pageY);

		const renameButton = cm.addButton(`
				<div class="cm-img-component-button-splitter">
					<i class="bi bi-input-cursor-text"></i>
					<span>Rename</span>
				</div>
			`);

		cm.addSeparator();

		if (this.fileExtension !== "svg") {
			const resizeButton = cm.addButton(`
				<div class="cm-img-component-button-splitter">
					<i class="bi bi-bounding-box"></i>
					<span>Resize</span>
				</div>
			`);

			resizeButton.addEventListener("click", () => {
				cm.cleanup();

				this.imageManager.windowManager.show(WindowManager.WINDOWS.RESIZE);
				this.imageManager.windowManager.resizeWindow.onWindowShown(this.fullFilePath, this.uri, this.imageWidth, this.imageHeight);
			});

			const cropButton = cm.addButton(`
				<div class="cm-img-component-button-splitter">
					<i class="bi bi-person-bounding-box"></i>
					<span>Crop</span>
				</div>
			`);

			cropButton.addEventListener("click", () => {
				cm.cleanup();

				this.imageManager.windowManager.show(WindowManager.WINDOWS.CROP);
				this.imageManager.windowManager.cropWindow.onWindowShown(this.fullFilePath, this.uri, this.imageWidth, this.imageHeight);
			});

			cm.addSeparator();
		}

		const cloneButton = cm.addButton(`
			<div class="cm-img-component-button-splitter">
				<i class="bi bi-images"></i>
				<span>Clone</span>
			</div>
		`);

		cloneButton.addEventListener("click", () => {
			cm.cleanup();

			this.imageManager.windowManager.show(WindowManager.WINDOWS.CLONE);
			this.imageManager.windowManager.cloneWindow.onWindowShown(this.fullFilePath, this.fileNameWithoutExtension, this.fileExtension);
		});

		if (this.fileExtension !== "svg") {
			const regenThumbButton = cm.addButton(`
				<div class="cm-img-component-button-splitter">
					<i class="bi bi-image-fill"></i>
					<span>Regen Thumb</span>
				</div>
			`);

			regenThumbButton.addEventListener("click", e => {
				cm.cleanup();

				this.onRegenButtonClicked();
			});
		}

		cm.addSeparator();

		const copyImageURI = cm.addButton(`
			<div class="cm-img-component-button-splitter">
				<i class="bi bi-pin-map-fill"></i>
				<span>Copy URI</span>
			</div>
		`);

		copyImageURI.addEventListener("click", e => {
			cm.cleanup();

			this.onCopyURIButtonClicked();
		});

		cm.addSeparator();

		const deleteButton = cm.addButton(`
				<div class="cm-img-component-button-splitter">
					<i class="text-danger bi bi-x-circle-fill"></i>
					<span>Delete</span>
				</div>
			`);

		deleteButton.addEventListener("click", e => {
			cm.cleanup();

			this.onDeleteButtonClicked();
		});

		renameButton.addEventListener("click", () => {
			cm.cleanup();

			this.toggleRenameField();
		});

		cm.render();
	}

	/**
	 * When a context menu is requested and this is the only ImageComponent
	 * in the selection
	 * @param {MouseEvent} e
	 */
	onMultipleSelectionContextMenu(e){
		const cm = new ContextMenu(e.pageX, e.pageY);

		const resizeButton = cm.addButton(`
			<div class="cm-img-component-button-splitter">
				<i class="bi bi-bounding-box"></i>
				<span>Resize All</span>
			</div>
		`);

		resizeButton.addEventListener("click", () => {
			cm.cleanup();

			this.imageManager.windowManager.show(WindowManager.WINDOWS.RESIZE);
			this.imageManager.windowManager.resizeWindow.onWindowShown();
		});

		cm.addSeparator();

		const cloneAndConvertTypesButton = cm.addButton(`
			<div class="cm-img-component-button-splitter">
				<i class="bi bi-images"></i>
				<span>Clone &amp; Convert</span>
			</div>
			`);

		cloneAndConvertTypesButton.addEventListener("click", () => {
			cm.cleanup();

			this.imageManager.windowManager.show(WindowManager.WINDOWS.CLONE);
			this.imageManager.windowManager.cloneWindow.onWindowShown(this.fullFilePath, this.fileNameWithoutExtension, this.fileExtension);
		});

		const regenThumbButton = cm.addButton(`
				<div class="cm-img-component-button-splitter">
					<i class="bi bi-image-fill"></i>
					<span>Regen Thumbs</span>
				</div>
			`);

		regenThumbButton.addEventListener("click", e => {
			cm.cleanup();

			this.onRegenButtonClicked();
		});

		cm.addSeparator();

		const deleteButton = cm.addButton(`
			<div class="cm-img-component-button-splitter">
				<i class="text-danger bi bi-x-circle-fill"></i>
				<span>Delete All</span>
			</div>
		`);

		deleteButton.addEventListener("click", e => {
			cm.cleanup();

			this.onDeleteButtonClicked();
		});

		cm.render();
	}

	/**
	 * Submits a thumb regeneration request for this image component
	 */
	async onRegenButtonClicked(){
		for (/** @type {ImageComponent} */ const component of this.imageManager.selectionListener.currentImageComponentSelection) {
			const fData = new FormData();
			fData.set("image-path", component.fullFilePath);

			const response = await fetch(`/uplift/image-manager/regenerate-thumb`, {
				credentials: "same-origin",
				cache: "no-cache",
				method: "patch",
				body: fData
			});

			/** @type {{status: number}} */
			const data = await response.json();
			if (data.status === 1) {

			}
		}
	}

	/**
	 * Copies the URI to the clipboard
	 */
	async onCopyURIButtonClicked(){
		try {
			// Check if the Clipboard API is supported
			if (!navigator.clipboard) {
				throw new Error("Clipboard API not supported.");
			}

			// Replace spaces with %20
			const uriToCopy = this.uri.replace(/ /g, "%20");

			// Attempt to write text to the clipboard
			await navigator.clipboard.writeText(uriToCopy);
			console.log("URI copied to clipboard successfully.");
		} catch (error) {
			console.error("Failed to copy URI: ", error);
		}
	}

	/**
	 * Submits a request to delete all images in the current selection
	 */
	async onDeleteButtonClicked(){
		for (/** @type {ImageComponent} */ const component of this.imageManager.selectionListener.currentImageComponentSelection) {
			const fData = new FormData();
			fData.set("image-path", component.fullFilePath);

			const response = await fetch(`/uplift/image-manager/delete-image`, {
				credentials: "same-origin",
				cache: "no-cache",
				method: "delete",
				body: fData
			});

			/** @type {{status: number}} */
			const data = await response.json();
			if (data.status === 1) {
				// Remove the DOM representing this image component
				component.dom.remove();
			}
		}
	}

	/**
	 * Hooks necessary events to elements in the component
	 */
	hookEvents(){
		const renameForm = this.dom.querySelector(".image-rename-form");

		renameForm.addEventListener("submit", e => {
			e.preventDefault();
			this.submitRenameForm();
		});

		renameForm.querySelector("input").addEventListener("blur", e => {
			this.submitRenameForm();
		});

		renameForm.querySelector("input").addEventListener("focusout", e => {
			this.submitRenameForm();
		});
	}

	/**
	 * Submits a rename request
	 */
	async submitRenameForm(){
		if (this.isRenameProcessing){
			return;
		}

		const renameForm = this.dom.querySelector(".image-rename-form");
		const fData = new FormData();
		const inputField = renameForm.querySelector("input");
		const newFileNameWithoutExtension = inputField.value.trim();

		fData.set("new-file-name-without-extension", newFileNameWithoutExtension);
		fData.set("file-path", this.fullFilePath);

		// Check if a renaming is actually happening
		if (newFileNameWithoutExtension.toLowerCase() === this.fileNameWithoutExtension.toLowerCase()){
			this.toggleRenameField();
			return;
		}

		this.isRenameProcessing = true;
		inputField.setAttribute("disabled", "true");

		const response = await fetch(`/uplift/image-manager/rename-image`, {
			credentials:"same-origin",
			cache:"no-cache",
			method:"patch",
			body:fData
		});

		if (response.status === 200){
			/** @type {{status:int, error:?string, newImageFilePath: string, newFileNameWithoutExtension:string, newImageURI: string, newImageThumbURI: string}} */
			const data = await response.json();
			if (data.status === 1){
				// Success
				this.renameSuccess(
					data.newImageFilePath,
					data.newFileNameWithoutExtension,
					data.newImageURI,
					data.newImageThumbURI
				);
			}else if (data.status === -1){
				inputField.removeAttribute("disabled");
				this.isRenameProcessing = false;
				alert(data.error);
			}
		}else{
			inputField.removeAttribute("disabled");
			this.isRenameProcessing = false;
			alert("Non-200 OK response. Check the inspector network tab.");
		}
	}

	/**
	 * Successful image renaming response from the backend
	 * @param {string} newImageFilePath
	 * @param {string} newImageNameWithoutExtension
	 * @param {string} newImageURI
	 * @param {string} newImageThumbURI
	 */
	renameSuccess(newImageFilePath, newImageNameWithoutExtension, newImageURI, newImageThumbURI){
		this.fileNameWithoutExtension = newImageNameWithoutExtension;
		this.fullFilePath = newImageFilePath;
		this.fileName = `${newImageNameWithoutExtension}.${this.fileExtension}`;
		this.uri = newImageURI;
		this.thumbURI = newImageThumbURI;

		this.dom.querySelector(".image-file-name").textContent = newImageNameWithoutExtension;
		this.isRenameProcessing = false;
		this.toggleRenameField();

		const renameForm = this.dom.querySelector(".image-rename-form");
		const inputField = renameForm.querySelector("input");
		inputField.removeAttribute("disabled");
	}

	/**
	 * Checks if the image rename field is active
	 */
	isRenameFormShown(){
		return this.dom.querySelector(".image-rename-container").style.display !== "none";
	}

	/**
	 * Toggles the image rename field for this component
	 */
	toggleRenameField(){
		/** @type {HTMLDivElement} */
		const renameDiv = this.dom.querySelector(".image-rename-container");
		const nameContents = this.dom.querySelector(".image-name-contents");

		if (!this.isRenameFormShown()) {
			this.dom.removeAttribute("draggable");
			nameContents.style.display = "none";
			renameDiv.style.display = "block";
			renameDiv.querySelector("form input").focus();
		}else{
			this.dom.setAttribute("draggable", "true");
			nameContents.style.display = "block";
			renameDiv.style.display = "none";
		}
	}

	render(){
		this.container.append(this.dom);
	}
}

export default ImageComponent;