import DirectoryImagesFetcher from "../DirectoryImagesFetcher.js";
import ImageManagerState from "../ImageManagerState.js";
import ImageMoveOperation from "../ImageMoveOperation.js";
import Endpoints from "../../Endpoints.js";
import ContextMenu from "../../utils/ContextMenu.js";
import ConfirmModal from "../../utils/ConfirmModal.js";
import SelectionListener from "../SelectionListener.js";
import Tooltip from "../../_external/bootstrap/tooltip.js";

class SidebarFolderComponent{

	/** @type {{[HTMLElement] : SidebarFolderComponent}} */
	static folderDOMCache = {};

	/**
	 * @type {?ImageManager}
	 */
	imageManager = null;

	/** @type {Tooltip} */
	bsTooltip;

	/**
	 * @type {HTMLElement}
	 */
	container;

	/**
	 * @param {ImageManager} imageManager
	 * @param {{directoryName: string, fullDirectoryPath: string, childDirectories: array}} childFolder The directory this sidebar folder component will represent
	 */
	constructor(imageManager, childFolder){
		this.imageManager = imageManager;
		this.container = imageManager.getSidebarFoldersContainer();
		this.folderName = childFolder.directoryName;
		this.filesystemPath = childFolder.fullDirectoryPath;

		/** @property {array} childFolders */
		this.childFolders = childFolder.childDirectories;

		/** @property {number} lastMouseClickTime The last unix time in ms that this folder was clicked */
		this.lastMouseClickTime = null;

		/** @property {number} dragEnterEventDepth How many drag enter events were fired and not "exited" */
		this.dragEnterEventDepth = 0;

		/** @property {boolean} isProcessing If a asynchronous event is currently processing, like a rename request. */
		this.isProcessing = false;

		this.dom = null;
	}

	/**
	 * @returns {HTMLElement}
	 */
	getDOM(){
		const template = document.createElement("im-folder");
		template.setAttribute("tabindex", "0");

		template.innerHTML = `
			<div class="main-button-contents">
				<div class="folder-rename-container" style="display:none;">
					<form>
						<input type="text" class="im-folder-directory-name-input form-control form-control-sm" name="directory-name">
					</form>
				</div>
				<div class="folder-button-container">
					<button type="button" class="expand-child-folders-button">
						<i class="bi bi-chevron-right"></i>
					</button>
					<i class="bi bi-folder-fill"></i>
					<span class="folder-name">${this.folderName}</span>
				</div>
			</div>
			<div class="child-folders"></div>
		`;

		const mainButtonContents = template.querySelector(".main-button-contents");
		const expandButton = template.querySelector(".expand-child-folders-button");
		const childFoldersContainer = template.querySelector(".child-folders");
		const folderNameForm = template.querySelector(".folder-rename-container form");
		const folderButtonContainer = template.querySelector(".folder-button-container");
		const folderNameSpan = template.querySelector(".folder-name");

		this.bsTooltip = new Tooltip(folderNameSpan, {
			trigger:"hover",
			animation: false,
			title:this.folderName
		});

		folderButtonContainer.setAttribute("draggable", true);

		if (this.childFolders.length > 0){
			// Has children
			for (const childFolder of this.childFolders){
				const component = new SidebarFolderComponent(this.imageManager, childFolder);
				const dom = component.getDOM();
				childFoldersContainer.append(dom);
			}
		}else{
			template.classList.add("no-child-folders");
		}

		folderButtonContainer.addEventListener("dragstart", e => {
			if (ImageManagerState.isDraggingComponent === false) {
				ImageManagerState.isDraggingComponent = true;
				ImageManagerState.currentlyDraggedFolderComponent = this;
				e.dataTransfer.setData("text/plain", this.filesystemPath);
				e.dataTransfer.setData("cms/droptype", "directory");
			}
		});

		folderNameForm.addEventListener("submit", e => {
			e.preventDefault();
			this.submitRename();
		});

		folderNameForm.querySelector(".im-folder-directory-name-input").addEventListener("blur", e => {
			this.submitRename();
		});

		expandButton.addEventListener("click", () => {
			template.classList.toggle("expanded");
		});

		mainButtonContents.addEventListener("click", () => {
			this.onFolderClick();
		});

		mainButtonContents.addEventListener("dragenter", e => {
			e.preventDefault();
			++this.dragEnterEventDepth;

			if (!mainButtonContents.classList.contains("drag-enter")){
				mainButtonContents.classList.add("drag-enter");
			}
		});

		mainButtonContents.addEventListener("dragleave", e => {
			e.preventDefault();
			--this.dragEnterEventDepth;

			if (mainButtonContents.classList.contains("drag-enter") && this.dragEnterEventDepth === 0){
				mainButtonContents.classList.remove("drag-enter");
			}
		});

		mainButtonContents.addEventListener("drop", e => {
			e.preventDefault();
			this.dragEnterEventDepth = 0;

			if (mainButtonContents.classList.contains("drag-enter")){
				mainButtonContents.classList.remove("drag-enter");
				const fileSystemPath = e.dataTransfer.getData("text/plain");
				const dropType = e.dataTransfer.getData("cms/droptype");
				if (dropType !== null && dropType !== ""){
					if (dropType === "image") {
						// Attempt to move it
						this.onImageDropped();
					}else if (dropType === "directory"){
						this.onDirectoryDropped(fileSystemPath)
					}
				}
			}
		});

		mainButtonContents.addEventListener("dragover", e => {
			e.preventDefault();
		});

		folderButtonContainer.addEventListener("contextmenu", e => {
			e.preventDefault();
			e.stopPropagation();
			this.onContextMenu(e.pageX, e.pageY);
		});

		this.dom = template;
		SidebarFolderComponent.folderDOMCache[this.dom] = this;

		return template;
	}

	render(){
		this.container.append(this.dom);
	}

	/**
	 * Reloads the child content. Does not re-fetch the child content from
	 * the backend API. Will read from this.childFolders
	 */
	reloadChildFolders(){
		const childFoldersContainer = this.dom.querySelector(".child-folders");
		childFoldersContainer.innerHTML = "";
		for (const childFolder of this.childFolders){
			const component = new SidebarFolderComponent(this.imageManager, childFolder);
			const dom = component.getDOM();
			childFoldersContainer.append(dom);
		}
	}

	/**
	 * When a directory is dropped on this component
	 * @param {string} directoryPath The directory path of the folder dropped on this component
	 */
	async onDirectoryDropped(directoryPath){
		const fData = new FormData();
		fData.set("current-full-path", directoryPath);
		fData.set("new-parent-directory", this.filesystemPath);

		const response = await fetch(`/uplift/image-manager/move-directory`, {
			credentials:"same-origin",
			cache:"no-cache",
			method:"patch",
			body:fData
		});

		/** @type {{status: number, destinationNewChildContents: {directoryName: string, fullDirectoryPath: string, childDirectories: Array}}} */
		const data = await response.json();
		if (data.status === 1){
			ImageManagerState.currentlyDraggedFolderComponent.dom.remove();
			this.childFolders = data.destinationNewChildContents.childDirectories;
			this.reloadChildFolders();
		}else if (data.status === -1){

		}
	}

	/**
	 * When a context menu is called on this folder item
	 * @param {number} posX
	 * @param {number} posY
	 */
	onContextMenu(posX, posY){
		const cm = new ContextMenu(posX, posY);
		cm.buildDOM();
		const renameButton = cm.addButton(`
			<i class="bi bi-input-cursor-text"></i>
			<span>Rename</span>
		`);

		const newFolderButton = cm.addButton(`
			<i class="bi bi-folder-plus"></i>
			<span>New Folder</span>
		`);

		cm.addSeparator();

		const deleteButton = cm.addButton(`
			<i class="bi bi-x-octagon-fill text-danger"></i>
			<span>Delete</span>
		`);

		renameButton.addEventListener("click", e => {
			cm.cleanup();
			this.enableRenaming(true);
		});

		newFolderButton.addEventListener("click", e => {
			cm.cleanup();
			this.addChildFolder();
		});

		deleteButton.addEventListener("click", e => {
			cm.cleanup();
			this.onFolderDeleteButtonClicked();

		});

		cm.render();
	}

	/**
	 * When the delete option in the context menu is clicked
	 */
	async onFolderDeleteButtonClicked(){
		const confirmModal = new ConfirmModal("confirm-folder-deletion");
		confirmModal.setHTMLContent(`
				<p>
					This will delete this directory and <strong>all of its contents</strong> permanently. This is a file system operation and there is no way to reverse this.
				</p>
			`);
		confirmModal.setConfirmButtonHTML(`
			<i class="bi bi-emoji-frown-fill"></i>
			<span>Delete Directory</span>
		`);
		confirmModal.setTitle("Confirm directory deletion");
		confirmModal.showModal();
		const didConfirm = await confirmModal.actionTaken();
		confirmModal.cleanup();

		if (didConfirm){
			// Delete the directory
			const fData = new FormData();
			fData.set("directory-path", this.filesystemPath);
			const response = await fetch(`/uplift/image-manager/delete-directory`, {
				credentials:"same-origin",
				cache:"no-cache",
				method:"delete",
				body:fData
			});

			const data = await response.json();
			if (data.status === 1){
				this.dom.remove();
			}else if (data.status === -1){
				alert(data.error);
			}
		}
	}

	/**
	 * Adds a new child folder to this one
	 */
	async addChildFolder(){
		// Fetch a new folder name
		const fData = new FormData();
		fData.set("parent-directory", this.filesystemPath);
		const response = await fetch(`/uplift/image-manager/new-directory`, {
			credentials:"same-origin",
			cache:"no-cache",
			method:"post",
			body:fData
		});

		/** @type {{status: number, newFolderName: string, error: ?string}} */
		const data = await response.json();
		if (data.status === 1) {

			// Create a new child folder
			const newFolder = new SidebarFolderComponent(this.imageManager, {
				directoryName:data.newFolderName,
				fullDirectoryPath:`${this.filesystemPath}/${data.newFolderName}`,
				childDirectories:[]
			});
			const newFolderDOM = newFolder.getDOM();
			this.dom.querySelector(".child-folders").append(newFolderDOM);

			// Check if this folder is marked as having no children.
			// Because it damn sure does now!
			if (this.dom.classList.contains("no-child-folders")){
				this.dom.classList.remove("no-child-folders");
			}

			if (!this.dom.classList.contains("expanded")){
				this.dom.classList.add("expanded");
			}

			// Enable renaming of the new folder
			newFolder.enableRenaming(true);
		}else if (data.status === -1){
			alert(data.error);
		}
	}

	/**
	 * When an image is dropped on this folder, move all image components
	 * in the current selection
	 */
	async onImageDropped(){
		const imageComponentsToMove = this.imageManager.selectionListener.currentImageComponentSelection;
		for (/** @type {ImageComponent} */const component of imageComponentsToMove) {
			const moveImageOperation = new ImageMoveOperation(
				component.fullFilePath,
				this.filesystemPath,
				component
			);
			this.imageManager.imageManagerState.currentlyDraggedImageComponent = null;
			await moveImageOperation.moveImageRequest();
		}
	}

	onFolderClick(){
		const now = (new Date()).getTime();
		if (this.lastMouseClickTime !== null){
			const difference = now - this.lastMouseClickTime;
			if (difference < 500){
				this.onFolderDoubleClick();
			}
		}

		this.lastMouseClickTime = now;
	}

	async onFolderDoubleClick(){
		if (!this.imageManager.directoryImagesFetcher.isBusy){
			const newDirectory = this.filesystemPath;
			this.imageManager.imageManagerState.currentDirectory = newDirectory;
			this.imageManager.directoryImagesFetcher.setDirectory(newDirectory);
			await this.imageManager.directoryImagesFetcher.fetchAndRender();
		}
	}

	async submitRename(){

		if (this.isProcessing){
			return;
		}

		this.isProcessing = true;

		const inputField = this.dom.querySelector(".im-folder-directory-name-input");
		const newFolderName = inputField.value.trim();
		if (newFolderName.toLowerCase() === this.folderName){
			// Ignore
		}else{
			// Submit rename
			const fData = new FormData(this.dom.querySelector(".folder-rename-container form"));
			fData.set("current-full-path", this.filesystemPath);
			const response = await fetch(`/uplift/image-manager/rename-directory`, {
				credentials:"same-origin",
				cache:"no-cache",
				method:"PATCH",
				body:fData
			});

			/** @type {{newDirectoryPath: string, childContents: {childDirectories: Array}, status: number}} */
			const data = await response.json();
			if (data.status === 1){
				// Successful rename
				this.folderName = newFolderName;
				this.bsTooltip._config.title = newFolderName;
				this.dom.querySelector(".folder-name").textContent = this.folderName;
				this.filesystemPath = data.newDirectoryPath;
				this.childFolders = data.childContents.childDirectories;

				// Re-render child folders
				this.reloadChildFolders();
			}
		}

		this.isProcessing = false;
		this.disableRenaming();
	}

	/**
	 * Enables the elements to rename the directory
	 * @param {boolean} focusInput
	 */
	enableRenaming(focusInput){
		const inputField = this.dom.querySelector(".im-folder-directory-name-input");
		inputField.value = this.folderName;
		this.dom.querySelector(".folder-rename-container").style.display = "block";
		this.dom.querySelector(".folder-button-container").style.display = "none";
		inputField.focus();
	}

	/**
	 * Disables the elements to rename the directory
	 */
	disableRenaming(focusInput){
		this.dom.querySelector(".folder-rename-container").style.display = "none";
		this.dom.querySelector(".folder-button-container").style.display = "flex";
	}
}

export default SidebarFolderComponent;