import WindowManager from "../WindowManager.js";

class CloneWindow{

	/**
	 * @type {ImageManager}
	 */
	imageManager;

	/**
	 * @type {HTMLElement}
	 */
	window;

	constructor(imageManager){
		this.imageManager = imageManager;
		this.window = this.imageManager.container.querySelector(".im-clone-window");
		this.dismissButton = this.window.querySelector(".im-modal-dismiss-button");
		this.form = this.window.querySelector(".im-clone-fields");
		this.formSubmitButton = this.window.querySelector(".im-clone-submit-button");
		this.imageRenameFieldContainer = this.window.querySelector(".im-clone-new-image-name-field");
		this.optionalMessage = this.window.querySelector(".im-clone-optional-message");
		this.singleHeading = this.window.querySelector(".im-clone-singular-heading");
		this.multiHeading = this.window.querySelector(".im-clone-multi-heading");
		this.multiNotice = this.window.querySelector(".im-clone-multi-notice");
		this.loader = this.window.querySelector(".im-clone-modal-loading-spinner");
		this.loaderFinishedNumber = this.window.querySelector(".im-finished-clone-number");
		this.loaderTotalNumber = this.window.querySelector(".im-clone-in-queue-upload-number");
		this.imagePath = this.window.querySelector(".im-clone-image-path");
		this.newNameInput = this.window.querySelector(".im-clone-new-name");
		this.imageTypeInput = this.window.querySelector(".im-clone-image-type");
		this.queuedImageComponents = [];

		/** @type {boolean} Flag for when the form is submitting and being processed */
		this.isProcessing = false;

		this.formSubmitButton.addEventListener("click", () => {
			this.onSubmit();
		});

		this.form.addEventListener("submit", e => {
			e.preventDefault();
			this.onSubmit();
		});

		this.dismissButton.addEventListener("click", () => {
			if (!this.isProcessing){
				this.imageManager.windowManager.show(WindowManager.WINDOWS.MAIN);
			}
		});
	}

	/**
	 * Gather valid cloneable images
	 */
	getAcceptableImagesFromSelection(){
		this.queuedImageComponents = [];
		for (/** @type {ImageComponent} */ const component of this.imageManager.selectionListener.currentImageComponentSelection){
			if (component.fileExtension !== "svg"){
				this.queuedImageComponents.push(component);
			}
		}
	}

	/**
	 * Function called when this window gets shown
	 */
	onWindowShown(){
		// Filter the current image component selection to those that can be converted
		this.getAcceptableImagesFromSelection();
		if (this.queuedImageComponents.length > 1){
			// Hide the rename field
			this.imageRenameFieldContainer.style.display = "none";
			this.singleHeading.style.display = "none";
			this.multiHeading.style.display = "block";
			this.multiNotice.style.display = "block";
			// It ain't optional here, bucko
			this.optionalMessage.style.display = "none";
		}else if (this.queuedImageComponents.length === 1){
			/** @type {ImageComponent} */
			const primaryImage = this.queuedImageComponents[0];
			this.singleHeading.style.display = "block";
			this.multiHeading.style.display = "none";
			this.multiNotice.style.display = "none";
			this.imageRenameFieldContainer.style.display = "block";
			this.optionalMessage.style.display = "block";
			this.newNameInput.value = primaryImage.fileNameWithoutExtension;
		}

		// TODO Prefill option value if single image
	}

	/**
	 * When the clone form is submitted
	 * @return {string[]}
	 */
	async onSubmit(){

		if (this.isProcessing){
			return;
		}

		this.isProcessing = true;
		this.loader.style.display = "block";
		this.loaderTotalNumber.textContent = String(this.queuedImageComponents.length);
		this.loaderFinishedNumber.textContent = "0";
		let totalFinished = 0;
		const errors = [];

		for (/** @type {ImageComponent} */const component of this.queuedImageComponents) {
			let newImageName = component.fileNameWithoutExtension;
			if (this.queuedImageComponents.length  === 1){
				// Only one image, change the name to what is in the image name input
				newImageName = this.newNameInput.value;
			}
			await new Promise(async resolve => {
				const fData = new FormData();
				fData.set("image-path", component.fullFilePath);
				fData.set("new-image-name", newImageName);
				fData.set("image-type", this.imageTypeInput.value);

				const response = await fetch(`/uplift/image-manager/clone-image`, {
					credentials: "same-origin",
					cache: "no-cache",
					method: "post",
					body: fData
				});

				if (response.status === 200) {
					const data = await response.json();
					if (data.status === 1) {
						// That's cool
					} else if (data.status === -1) {
						// Oof
						errors.push(data.error);
					}
				} else {
					errors.push("Non-200 server error.");
				}

				resolve();
			});
			++totalFinished;
			this.loaderFinishedNumber.textContent = String(totalFinished);
		}

		this.isProcessing = false;
		this.loader.style.display = "none";
		await this.imageManager.windowManager.mainWindow.refreshCurrentLocation();
		this.imageManager.windowManager.show(WindowManager.WINDOWS.MAIN);

		return errors;
	}

}

export default CloneWindow;