import PageComponent from "../components/PageComponent.js";
import VirtualParent from "../components/VirtualParent.js";

class ContentBuildForm{
	isProcessing = false;
	form = document.querySelector("#content-build-form");
	loader = document.querySelector("#content-build-loader");
	successContainer = document.querySelector("#content-build-success");
	noPagesMessages = {
		"General":document.querySelector("#no-general-pages-message"),
		"Service":document.querySelector("#no-service-pages-message"),
		"City":document.querySelector("#no-city-pages-message"),
		"Blog":document.querySelector("#no-blog-pages-message"),
		"Project":document.querySelector("#no-project-pages-message")
	}

	constructor() {
		this.form.addEventListener("submit", e => {
			e.preventDefault();
			this.onSubmit();
		});
	}

	async onSubmit(){
		if (this.isProcessing){
			return;
		}

		this.isProcessing = true;

		// Verify all virtual parents have a forward slash
		for (const component of VirtualParent.cache){
			if (!component.uri.startsWith("/")){
				alert("One of your virtual parents is missing a forward slash in the URI. All URIs in the virtual parent definitions must begin with a forward slash.");
				this.isProcessing = false;
				return;
			}
		}

		this.loader.style.display = null;
		this.form.style.display = "none";

		const payload = {};
		const pageBuildPayload = this.getContentPayload();
		payload.senzuSiteID = parseInt(document.querySelector("#site-id-input").value.trim());
		payload.pages = pageBuildPayload;
		const payloadJSON = JSON.stringify(payload);

		const headers = new Headers();
		headers.set("Content-Type", "application/json");
		const response = await fetch(`/uplift/senzu-importer/import-content`, {
		    body:payloadJSON,
		    method:"POST",
			headers: headers,
		    cache:"no-cache",
		    credentials:"same-origin"
		});

		let data;
		try{
		    /** @type {{status: int, error: ?string}} **/
		    data = await response.json();
		}catch(jsonSyntaxError){
		    alert("The server responded with invalid JSON.");
			this.isProcessing = false;
			this.loader.style.display = "none";
			this.form.style.display = null;
		    return;
		}

		if (data.status === 1){
			this.loader.style.display = "none";
			this.successContainer.style.display = null;
		}else if (data.status === -1){
			alert(data.error);
			this.loader.style.display = "none";
			this.form.style.display = null;
		}

		this.isProcessing = false;
	}

	show(){
		this.form.style.display = null;
	}

	hide(){
		this.form.style.display = "none";
	}

	/**
	 * Packages up the PageComponents into a payload to be sent to import into the system
	 * @returns array
	 */
	getContentPayload(){
		const pageImportPayload = [];

		for (const component of PageComponent.cache){
			const breadcrumbs = [];

			// Check if it's the home page. If so, don't add the home page crumb
			if (component.slug !== ""){
				breadcrumbs.push({
					label:"Home",
					uri:"/"
				});
			}

			// Variable will be appended to each iteration of a virtual parent
			let completeURISoFar = "";
			if (component.pageParentID !== null) {
				const ancestors = component.getAncestorArray();

				for (const virtualParent of ancestors){
					completeURISoFar += virtualParent.uri;
					breadcrumbs.push({
						label:virtualParent.name,
						uri:completeURISoFar
					});
				}

			}

			// Add itself as a breadcrumb
			const fullURI = `${completeURISoFar}/${component.slug}`;
			breadcrumbs.push({
				label:component.pageName,
				uri:fullURI
			});

			// Add page datas
			const pageDatas = {};
			for (const pageDataComponent of component.pageDatas){
				const dataType = pageDataComponent.dataType;
				if (dataType in pageDatas){
					// Convert to array
					if (!Array.isArray(pageDatas[dataType])){
						pageDatas[dataType] = [pageDatas[dataType], pageDataComponent.dataValue]
					}else{
						// Add to array
						pageDatas[dataType].push(pageDataComponent.dataValue);
					}
				}else{
					pageDatas[dataType] = pageDataComponent.dataValue;
				}
			}

			pageImportPayload.push({
				pageName: component.pageName,
				slug: component.slug,
				pageRoute: fullURI,
				pageType: component.pageType,
				title: component.title,
				description: component.description,
				pageBody: component.content,
				featuredImageURI: component.featuredImageURI,
				featuredImageThumbURI: component.featuredImageThumbURI,
				breadcrumbs:breadcrumbs,
				pageDatas:pageDatas
			});
		}

		return pageImportPayload;
	}
}

export default new ContentBuildForm();