import BSModal from "../_external/bootstrap/modal.js";

/**
* Programmatic confirmation modal for general action confirming use.
* Such as confirming a deletion act after the button is pressed.
* Uses asynchronous programming to send confirm/cancel events back
* to the calling scope.
*/
class ConfirmModal{
	constructor(id){
		this.modal = this.build(id);

		/** @type {Modal} */
		this.bsModal = new BSModal(this.modal);

		/** @property {function} acceptPromise */
		this.acceptPromise = null;

		/** @property {Promise} actionPromise */
		this.actionPromise = new Promise( (accept, reject) => {
			// Leak the "accept" function out of the scope
			// so other functions can call it
			this.acceptPromise = accept;
		});

		// Connect events
		this.modal.addEventListener("hide.bs.modal", () => {
			this.acceptPromise(false);
			this.modal.remove();
		});

		this.modal.querySelector(".modal-confirm-button").addEventListener("click", () => {
			this.acceptPromise(true);
		});
	}

	/**
	* Creates a new promise that can be awaited by a calling function
	* for when action is taken on the modal
	* @returns {Promise} Boolean when accepted. False if the modal is dismissed.
	*/
	actionTaken(){
		return this.actionPromise;
	}

	/**
	* Sets the modal's title
	* @param {string} title
	*/
	setTitle(title){
		this.modal.querySelector(".modal-title").textContent = title;
	}

	/**
	* Sets the modal's textual content. Does not accept HTML
	* @param {string} content
	*/
	setContent(content){
		this.modal.querySelector(".modal-body-inner").textContent = content;
	}

	/**
	* Sets the modal's HTML content.
	* @param {string} htmlContent
	*/
	setHTMLContent(htmlContent){
		this.modal.querySelector(".modal-body-inner").innerHTML = htmlContent;
	}

	/**
	* Sets the text of the confirm button
	* @param {string} text
	*/
	setConfirmButtonText(text){
		this.modal.querySelector(".modal-confirm-button").textContent = text;
	}

	/**
	 * Sets the HTML of the confirm button
	 * @param {string} html
	 */
	setConfirmButtonHTML(html){
		this.modal.querySelector(".modal-confirm-button").innerHTML = html;

		// Confirm event needs to be reconnected
		this.modal.querySelector(".modal-confirm-button").addEventListener("click", () => {
			this.acceptPromise(true);
		});
	}

	/**
	* Shows the modal
	*/
	showModal(){
		this.bsModal.show();
	}

	/**
	* Hides the modal
	*/
	hideModal(){
		this.bsModal.hide();
	}

	/**
	* Hides the cancel button
	*/
	hideSecondaryButton(){
		this.modal.querySelector(".modal-cancel-button").style.display = "none";
	}

	/**
	* Builds the DOM modal
	* @param {string} id
	* @returns {HTMLDivElement} The modal
	*/
	build(id){
		const modal = document.createElement("div");
		modal.classList.add("modal");
		modal.setAttribute("id", id);
		modal.setAttribute("tabindex", "-1");
		modal.setAttribute("role", "dialog");
		modal.setAttribute("aria-hidden", "true");
		modal.innerHTML = `
			<div class="modal-dialog">
				<div class="modal-content">
					<div class="modal-header bg-danger text-white">
						<h4 class="modal-title"></h4>
						<button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
					</div>
					<div class="modal-body">
						<p class="modal-body-inner"></p>
						<div class="submit-error my-2 p-2 text-white bg-danger" style="display:none;"></div>
					</div>
					<div class="modal-footer">
						<button type="button" class="modal-confirm-button btn btn-danger"></button>
						<button type="button" class="modal-cancel-button btn btn-gray" data-bs-dismiss="modal">
							<span>Cancel</span>
						</button>
					</div>
				</div>
			</div>
		`;

		document.body.append(modal);

		return modal
	}

	cleanup(){
		this.bsModal.hide();
	}
}

export default ConfirmModal;
