export default class ContentLengthIndicator{
    public projectDescriptionBodyTextField: HTMLTextAreaElement = document.querySelector("#project-description-input");
    public textIndicator: HTMLSpanElement = document.querySelector("#content-length-text");
    public barIndicator: HTMLDivElement = document.querySelector("#content-length-inner-optimization-bar");
    public primaryServiceSelectMenu: HTMLSelectElement = document.querySelector("#service-page-id-input");
    public cityPageIDSelectMenu: HTMLSelectElement = document.querySelector("#city-page-id-input");
    public projectCityLocationInput: HTMLInputElement = document.querySelector("#project-location-city-input");
    public wordCountLabelMarks = {
        met: document.querySelector("#word-count-label-column .met") as HTMLSpanElement,
        notMet: document.querySelector("#word-count-label-column .not-met") as HTMLSpanElement,
    };
    public serviceMentionedLabelMarks = {
        met: document.querySelector("#primary-service-label-column .met") as HTMLSpanElement,
        notMet: document.querySelector("#primary-service-label-column .not-met") as HTMLSpanElement,
    };
    public cityMentionedLabelMarks = {
        met: document.querySelector("#city-label-column .met") as HTMLSpanElement,
        notMet: document.querySelector("#city-label-column .not-met") as HTMLSpanElement,
    };

    constructor() {
        this.projectDescriptionBodyTextField.addEventListener("input", () => {
            this.onProjectDescriptionInput();
        });

        this.cityPageIDSelectMenu.addEventListener("change", () => {
           this.updateContentLengthQualityFields();
        });

        this.projectCityLocationInput.addEventListener("change", () => {
            this.updateContentLengthQualityFields();
        });
    }

    public onProjectDescriptionInput(): void{
        this.updateContentLengthQualityFields();
    }

    /**
     * Called when refreshing of the quality bar and text is needed.
     */
    public updateContentLengthQualityFields(){
        let percentage: number = this.getQualityPercentage();

        // Min 10%
        if (percentage < 0.1){
            percentage = 0.1;
        }

        this.barIndicator.style.width = `${percentage * 100}%`;

        // Handle verbiage
        if (percentage < 0.5){
            this.textIndicator.textContent = "Poor";
            this.barIndicator.setAttribute("quality", "poor");
        }else if (percentage < 0.75){
            this.textIndicator.textContent = "Fair";
            this.barIndicator.setAttribute("quality", "fair");
        }else{
            this.textIndicator.textContent = "Good";
            this.barIndicator.setAttribute("quality", "good");
        }
    }

    /**
     * Quality is determined by how much content and how keyword optimized the project description is.
     * Key phrase optimization gains points when a service and location are selected in the project form, and the
     * service and location are also present in the description content.
     */
    private getQualityPercentage(): number{
        // Maximum weight for just content with no key phrase optimization is 50%.
        // 25% for service area optimization
        // 25% for city and state optimization

        // Word count ranges
        const minWordCount = 20;
        const maxWordCountToConsider = 100;

        // Define the scoring weights
        const maxWordCountWeight = 0.5;
        const locationOptimizationWeight = 0.25;
        const mainServiceOptimizationWeight = 0.25;

        // Accumulator for total percentage
        let totalWeight = 0; // 0 to 1 percentage

        // Interpolate percentage from 20-100 words.
        const numWords = this.countWordsInProjectDescription();
        if (numWords > minWordCount){
            const wordCountRange = maxWordCountToConsider - minWordCount;
            let interpolationPercentage = numWords / wordCountRange;

            // Clamp it at 100%
            if (interpolationPercentage > 1){
                interpolationPercentage = 1;
            }

            this.updateWordCountLabel(interpolationPercentage >= 1);

            totalWeight += (maxWordCountWeight * interpolationPercentage);
        }else{
            this.updateWordCountLabel(false);
        }

        // Consider if the primary service is in the content
        const projectDescription = this.projectDescriptionBodyTextField.value.trim();
        if (this.primaryServiceSelectMenu.value !== "-1" && this.primaryServiceSelectMenu.value !== "0"){
            const primaryServiceName = this.primaryServiceSelectMenu.options[this.primaryServiceSelectMenu.selectedIndex];
            if (projectDescription.toLowerCase().includes(primaryServiceName.textContent.toLowerCase())){
                totalWeight += mainServiceOptimizationWeight;
                this.updateServiceLabel(true);
            }else{
                this.updateServiceLabel(false);
            }
        }else{
            this.updateServiceLabel(false);
        }

        // Consider if the city is in the content
        if (this.cityPageIDSelectMenu.value !== "-1" && this.cityPageIDSelectMenu.value !== "0"){
            const cityPageName = this.cityPageIDSelectMenu.options[this.cityPageIDSelectMenu.selectedIndex].textContent;
            const city = this.getCityFromCityPageName(cityPageName);
            if (projectDescription.toLowerCase().includes(city.toLowerCase())){
                totalWeight += locationOptimizationWeight;
                this.updateCityLabel(true);
            }else{
                this.updateCityLabel(false);
            }
        }else if(this.cityPageIDSelectMenu.value === "-1"){
            // Custom city
            const city = this.projectCityLocationInput.value.trim();
            if (city.length > 0){
                if (projectDescription.toLowerCase().includes(city.toLowerCase())){
                    totalWeight += locationOptimizationWeight;
                    this.updateCityLabel(true);
                }else{
                    this.updateCityLabel(false);
                }
            }else{
                this.updateCityLabel(false);
            }
        }else{
            this.updateCityLabel(false);
        }

        return totalWeight;
    }

    private countWordsInProjectDescription(): number{
        const text: string = this.projectDescriptionBodyTextField.value.trim();
        return text.split(" ").length;
    }

    /**
     * City pages usually have the format "City, ST" and we only want the City.
     */
    private getCityFromCityPageName(cityPageName: string): string{
        if (cityPageName.includes(",")){
            return cityPageName.split(",")[0];
        }else{
            return cityPageName;
        }
    }

    /**
     * Shows or hides the word count met checkmark
     */
    private updateWordCountLabel(isWordCountMet: boolean): void{
        if (isWordCountMet){
            this.wordCountLabelMarks.met.style.display = null;
            this.wordCountLabelMarks.notMet.style.display = "none";
        }else{
            this.wordCountLabelMarks.met.style.display = "none";
            this.wordCountLabelMarks.notMet.style.display = null;
        }
    }

    /**
     * Shows or hides the service mentioned checkmark
     */
    private updateServiceLabel(isServiceMentioned: boolean): void{
        if (isServiceMentioned){
            this.serviceMentionedLabelMarks.met.style.display = null;
            this.serviceMentionedLabelMarks.notMet.style.display = "none";
        }else{
            this.serviceMentionedLabelMarks.met.style.display = "none";
            this.serviceMentionedLabelMarks.notMet.style.display = null;
        }
    }

    /**
     * Shows or hides the city mentioned checkmark
     */
    private updateCityLabel(isCityMentioned: boolean): void{
        if (isCityMentioned){
            this.cityMentionedLabelMarks.met.style.display = null;
            this.cityMentionedLabelMarks.notMet.style.display = "none";
        }else{
            this.cityMentionedLabelMarks.met.style.display = "none";
            this.cityMentionedLabelMarks.notMet.style.display = null;
        }
    }
}