import { IFindResult } from "../../Interfaces/FindAndReplace/IFindResult";
import { IFindResultItem } from "../../Interfaces/FindAndReplace/IFindResultItem";
import { IFoundFileDto } from "../../Interfaces/FindAndReplace/IFoundFile";
import { SearchForm } from "../Forms/SearchForm";
import { MakeReplacementsDto } from "../Interfaces/MakeReplacementsDto";

type FileFindResultItems = HTMLDivElement[];

export class FileResult {
	private SearchForm: SearchForm;
	public static ResultsCache: FileResult[] = [];
	public LocalID: string;
	public OriginalResult: IFoundFileDto;
	public FilePath: string;
	public FileName: string;
	public FindResults: IFindResultItem[];
	public Dom: HTMLElement;
	public IsProcessing: boolean = false;

	public constructor(findResult: IFoundFileDto, searchForm: SearchForm) {

		this.LocalID = findResult.filePath;
		this.OriginalResult = findResult;
		this.FilePath = findResult.filePath;
		this.FileName = findResult.fileName;
		this.FindResults = findResult.findResultItems;
		this.SearchForm = searchForm;
		this.Dom = this.GetFileResultDOM();

		FileResult.ResultsCache.push(this);
	}

	// Creates the DOM structure for displaying results within the UI/Page.
	private GetFileResultDOM() {
		let title = `Theme File: ${this.FileName}`;

		const resultItems: FileFindResultItems = [];

		// Loop through all search results and create their DOM elements.
		for (const findResult of this.FindResults){
			const stub = findResult.stub;
			const resultItem = document.createElement("div");
			const beforeSpan = document.createElement("span");
			const resultSpan = document.createElement("span");
			const afterSpan = document.createElement("span");

			resultSpan.classList.add("find-replace-search-result-result-container");

			// Use innerHTML and not textContent here so the html entities render as their display characters
			beforeSpan.innerHTML = `${stub.before.length > 0 ? "... " : ""}${stub.beforeSanitized}`;
			resultSpan.innerHTML = stub.resultSanitized;
			afterSpan.innerHTML = `${stub.afterSanitized}${stub.after.length > 0 ? "... " : ""}`;

			// Add the spans to the result item container.
			resultItem.append(beforeSpan);
			resultItem.append(resultSpan);
			resultItem.append(afterSpan);

			resultItems.push(resultItem);
		}

		const template = document.createElement("div");
		template.classList.add("find-and-replace-find-result-item");
		template.innerHTML = `
			<div class="card">
				<div class="card-header">
					<h4>${title}</h4>
				</div>
				<div class="card-body">
					<div>
						<div class="result-stub-container"></div>
					</div>
					<div class="replace-all-button-container">
						<div class="text-end">
							<button type="button" class="btn btn-primary replace-all-occurrences-button" id="replace-all">
								<span>Replace All</span>
								<i class="bi bi-arrow-repeat"></i>
							</button>
						</div>
					</div>
				</div>
			</div>
		`;

		// Add the result items to the stub container in the card.
		const stubContainer = template.querySelector(".result-stub-container");
		for (const item of resultItems){
			stubContainer.append(item);
		}

		// Attaches event listener to the "Replace All Occurrences" button.
		const replaceAllButton = template.querySelector(".replace-all-occurrences-button") as HTMLElement;
		replaceAllButton.addEventListener("click", () => {
			this.ReplaceAllOccurrenceButtonClicked();
		});

		return template;
	}

	// Handles the "Replace All Occurrences" functionality when clicked.
	public async ReplaceAllOccurrenceButtonClicked() {
		if (this.IsProcessing) return;

		// Checks to see if the search input has not changed since the last search.
		if (this.SearchForm.SearchInputChangedSinceLastSubmission) {
			alert("You have changed your search input since the last Find operation. Re-run the Find operation before making replacements.");
			return;
		}

		this.IsProcessing = true;

		// Gets replacement text from the input field.
		const replacementInput = document.querySelector("#replacement-input");
		const replacementText = replacementInput instanceof HTMLInputElement ? replacementInput.value : "";

		// Prepare the payload for server request.
		const payload: MakeReplacementsDto = {
			findAndReplaceFoundDto: {
				foundPagesMatches: {
					foundPages: []
				},
				foundFiles: [this.OriginalResult]
			},
			originalQuery: this.SearchForm.FindAndReplaceEntryInstance.LastQueryUsed,
			replacement: replacementText,
		};
		// PATCH request to replace all occurrences.
		const response = await fetch(`/uplift/find-and-replace/make-replacements`, {
			body: JSON.stringify(payload),
			method: "PATCH",
			cache: "no-cache",
			credentials: "same-origin",
			headers: { "Content-Type": "application/json" },
		});

		let data;
		try {
			data = await response.json();
		} catch (error) {
			alert("The server responded with invalid JSON.");
			this.IsProcessing = false;
			return;
		}

		// Handles server response.
		if (data.status === 1) {
			this.Remove();
		} else if (data.status === -1) {
			alert(data.error);
		}

		this.IsProcessing = false;
	}

	// Removes all results from the UI/Page and clears cache. 
	public static ClearAll(): void {
		for(const result of FileResult.ResultsCache){
			result.Dom.remove();
		}
		FileResult.ResultsCache = [];
	}

	// Appends result's DOM element to a container.
	public RenderInto(container: HTMLElement): void {
		container.append(this.Dom);
	}

	// Removes result from the UI/Page.
	public Remove(): void {
		this.Dom.remove();
		const index = FileResult.ResultsCache.findIndex(result => result.LocalID === this.LocalID);
		if (index !== -1) {
			FileResult.ResultsCache.splice(index, 1);
		}
	}
}
