import { IFindResult } from "../../Interfaces/FindAndReplace/IFindResult";
import { IFindResultItem } from "../../Interfaces/FindAndReplace/IFindResultItem";
import { SearchForm } from "../Forms/SearchForm";

export class Result{
	private SearchForm: SearchForm;
    public static ResultsContainer = document.querySelector("#search-results-container");
    public static ResultsCache: Result[] = [];
    public static LocalCounter = 0;
    public LocalID: number;
    public OriginalResult: IFindResult;
    public FilePath: string;
    public FileName: string;
    public FindResults: IFindResultItem[];
    public PageID: number;
    public PageName: string;
    public Type: string;
    public Dom: HTMLElement;
    public IsProcessing: boolean = false;

    public constructor(findResult: IFindResult, searchForm: SearchForm){
        this.LocalID = Result.LocalCounter++;
        this.OriginalResult = findResult;
        this.FilePath = findResult.filePath;
        this.FileName = findResult.fileName;
        this.FindResults = findResult.findResults;
        this.PageID = findResult.pageID;
        this.PageName = findResult.pageName;
        this.Type = findResult.type;
        this.SearchForm = searchForm;

        if (this.Type === "PAGE") {
            this.Dom = this.GetPageResultDOM();
          } else {
            this.Dom = this.GetFileResultDOM();
          }
          
          Result.ResultsContainer.append(this.Dom);
          Result.ResultsCache.push(this);
          
    }

    public static ClearAll(): void {
        for(const result of Result.ResultsCache){
            result.Dom.remove();
        }
        Result.ResultsCache = [];
    }

    public Remove(): void {
        this.Dom.remove();
        for (const resultIndex in Result.ResultsCache){
			const iteratedResult = Result.ResultsCache[resultIndex];
			if (iteratedResult.LocalID === this.LocalID){
				Result.ResultsCache.splice(parseInt(resultIndex), 1);
			}
		}
    }

    public GetPageResultDOM(){
		let title = `Page: ${this.PageName} (ID: ${this.PageID})`;

		const headResultItems = [];
		const bodyResultItems = [];
		for (const findResult of this.FindResults){
			/** @type {{before: string, beforeSanitized: string, result: string, resultSanitized: string, after: string, afterSanitized: string}} */
			const stub = findResult.stub;
			const resultItem = document.createElement("div");
			const beforeSpan = document.createElement("span");
			const resultSpan = document.createElement("span");
			const afterSpan = document.createElement("span");

			resultSpan.classList.add("find-replace-search-result-result-container");

			// Use innerHTML and not textContent here so the html entities render as their display characters
			beforeSpan.innerHTML = `${stub.beforeSanitized.length > 0 ? "... " : ""}${stub.beforeSanitized}`;
			resultSpan.innerHTML = stub.resultSanitized;
			afterSpan.innerHTML = `${stub.afterSanitized}${stub.afterSanitized.length > 0 ? "... " : ""}`;

			resultItem.append(beforeSpan, resultSpan, afterSpan);

			if (findResult.isPageHead){
				headResultItems.push(resultItem);
			}else{
				bodyResultItems.push(resultItem);
			}
		}

		const template = document.createElement("div");
		template.classList.add("find-and-replace-find-result-item");
		template.innerHTML = `
			<div class="card">
				<div class="card-header">
					<div class="d-flex justify-content-between align-items-center">
						<h4>${title}</h4>
					</div>
				</div>
				<div class="result-card-body">
					<div class="page-tabs">
						<div class="d-flex justify-content-between align-items-center">
							<ul class="nav nav-underline">
								<li class="nav-item">
									<a class="nav-link active" aria-current="page" href="#" data-target="body-container">Page Body</a>
								</li>
								<li class="nav-item">
									<a class="nav-link" href="#" data-target="head-container">Page Head</a>
								</li>
							</ul>
							<div class="edit-button">
								<a target="_blank" href="/uplift/page-editor/${this.PageID}" class="text-decoration-none">
									<span>Edit</span>
									<i class="bi bi-pencil-square"></i>
								</a>
							</div>
						</div>
					</div>
					<div class="container-fluid">
						<div class="row">
							<div class="col-xl-6 page-body-column" id="body-container">
								<div class="stub-container body-result-stub-container"></div>
							</div>
							<div class="col-xl-6 page-head-column d-none" id="head-container">
								<div class="stub-container head-result-stub-container"></div>
							</div>
						</div>
					</div>
					<div class="replace-all-button-container">
						<div class="text-end">
							<button type="button" class="btn btn-primary replace-all-occurrences-button" id="replace-all">
								<span>Replace All</span>
								<i class="bi bi-arrow-repeat"></i>
							</button>
						</div>
					</div>
				</div>
			</div>
		`;

		const headStubContainer = template.querySelector(".head-result-stub-container");
		const bodyStubContainer = template.querySelector(".body-result-stub-container");

		for (const item of headResultItems){
			headStubContainer.append(item);
		}

		for (const item of bodyResultItems){
			bodyStubContainer.append(item);
		}

		const replaceAllButton = template.querySelector(".replace-all-occurrences-button") as HTMLElement;

		replaceAllButton.addEventListener("click", () => {
			this.ReplaceAllOccurrenceButtonClicked();
		});

		const pageTabs = template.querySelectorAll(".page-tabs .nav-link");
		pageTabs.forEach(tab => {
			tab.addEventListener("click", (event) => {
				event.preventDefault();
				const targetId = tab.getAttribute("data-target");
	
				template.querySelectorAll(".page-body-column, .page-head-column").forEach(el => {
					if (!el.classList.contains("d-none")) {
						el.classList.add("d-none");
					}
				});
	
				pageTabs.forEach(t => {
					if (t.classList.contains("active")) {
						t.classList.remove("active");
					}
				});
	
				const targetContainer = template.querySelector(`#${targetId}`);
				if (targetContainer) {
					targetContainer.classList.remove("d-none");
				}
				tab.classList.add("active");
			});
		});

		return template;
	}

    private GetFileResultDOM(){
		let title = `Theme File: ${this.FileName}`;

		const resultItems = [];
		for (/** @type {{isPageHead: boolean, start: int, end: int, stub: Object}} */ const findResult of this.FindResults){
			/** @type {{before: string, beforeSanitized: string, result: string, resultSanitized: string, after: string, afterSanitized: string}} */
			const stub = findResult.stub;
			const resultItem = document.createElement("div");
			const beforeSpan = document.createElement("span");
			const resultSpan = document.createElement("span");
			const afterSpan = document.createElement("span");

			resultSpan.classList.add("find-replace-search-result-result-container");

			// Use innerHTML and not textContent here so the html entities render as their display characters
			beforeSpan.innerHTML = `${stub.before.length > 0 ? "... " : ""}${stub.beforeSanitized}`;
			resultSpan.innerHTML = stub.resultSanitized;
			afterSpan.innerHTML = `${stub.afterSanitized}${stub.after.length > 0 ? "... " : ""}`;

			resultItem.append(beforeSpan);
			resultItem.append(resultSpan);
			resultItem.append(afterSpan);

			resultItems.push(resultItem);
		}

		const template = document.createElement("div");
		template.classList.add("find-and-replace-find-result-item");
		template.innerHTML = `
			<div class="card">
				<div class="card-header">
					<h4>${title}</h4>
				</div>
				<div class="card-body">
					<div>
						<div class="result-stub-container"></div>
					</div>
					<div class="replace-all-button-container">
						<div class="text-end">
							<button type="button" class="btn btn-primary replace-all-occurrences-button" id="replace-all">
								<span>Replace All</span>
								<i class="bi bi-arrow-repeat"></i>
							</button>
						</div>
					</div>
				</div>
			</div>
		`;

		const stubContainer = template.querySelector(".result-stub-container");
		for (const item of resultItems){
			stubContainer.append(item);
		}

		const replaceAllButton = template.querySelector(".replace-all-occurrences-button");
		replaceAllButton.addEventListener("click", () => {
			this.ReplaceAllOccurrenceButtonClicked();
		});

		return template;
	}

    public async ReplaceAllOccurrenceButtonClicked(){

		if (this.IsProcessing){
			return;
		}

		if (this.SearchForm.SearchInputChangedSinceLastSubmission){
			alert("You have changed your search input since the last Find operation. You should re-run the Find operation before trying to make a replacement.");
			return;
		}

		this.IsProcessing = true;

		const payload: any = {};
        const resultClone = structuredClone(this.OriginalResult);
        const newFindResultItems: IFindResultItem[] = [];
        const leftOverFindResultItems: IFindResultItem[] = [];

        for (const findResultItem of resultClone.findResults) {
            if (findResultItem.isPageHead) {
                newFindResultItems.push(findResultItem);
            } else {
                leftOverFindResultItems.push(findResultItem);
            }
        }

        resultClone.findResults = newFindResultItems;
        payload.findResult = resultClone;

        const replacementInput = document.querySelector("#replacement-input");
        if (replacementInput && replacementInput instanceof HTMLInputElement) {
            payload.replacement = replacementInput.value;
        } else {
            payload.replacement = "";
        }

		const response = await fetch(`/uplift/find-and-replace/replace`, {
			body:JSON.stringify(payload),
			method:"PATCH",
			cache:"no-cache",
			credentials:"same-origin",
			headers:{
				"Content-Type":"application/json"
			}
		});

		let data;
		try{
			/** @type {{status: int, error: ?string}} **/
			data = await response.json();
		}catch(jsonSyntaxError){
			alert("The server responded with invalid JSON.");
			this.IsProcessing = false;
			return;
		}

		if (data.status === 1){
			this.Remove();
		}else if (data.status === -1){
			alert(data.error);
		}

		this.IsProcessing = false;
	}

    private async ReplaceAllBodyOccurrenceButtonClicked(){

		if (this.IsProcessing){
			return;
		}

		if (this.SearchForm.SearchInputChangedSinceLastSubmission){
			alert("You have changed your search input since the last Find operation. You should re-run the Find operation before trying to make a replacement.");
			return;
		}

		this.IsProcessing = true;

		const payload: any = {};
        const resultClone = structuredClone(this.OriginalResult);
        const newFindResultItems: IFindResultItem[] = [];
        const leftOverFindResultItems: IFindResultItem[] = [];

        for (const findResultItem of resultClone.findResults) {
            if (findResultItem.isPageHead) {
                newFindResultItems.push(findResultItem);
            } else {
                leftOverFindResultItems.push(findResultItem);
            }
        }

        resultClone.findResults = newFindResultItems;
        payload.findResult = resultClone;

        const replacementInput = document.querySelector("#replacement-input");
        if (replacementInput && replacementInput instanceof HTMLInputElement) {
            payload.replacement = replacementInput.value;
        } else {
            payload.replacement = "";
        }

		const response = await fetch(`/uplift/find-and-replace/replace`, {
			body:JSON.stringify(payload),
			method:"PATCH",
			cache:"no-cache",
			credentials:"same-origin",
			headers:{
				"Content-Type":"application/json"
			}
		});

		let data;
		try{
			/** @type {{status: int, error: ?string}} **/
			data = await response.json();
		}catch(jsonSyntaxError){
			alert("The server responded with invalid JSON.");
			this.IsProcessing = false;
			return;
		}

		this.IsProcessing = false;
	}

    private async ReplaceAllHeadOccurrenceButtonClicked(){

		if (this.IsProcessing){
			return;
		}

		if (this.SearchForm.SearchInputChangedSinceLastSubmission){
			alert("You have changed your search input since the last Find operation. You should re-run the Find operation before trying to make a replacement.");
			return;
		}

		this.IsProcessing = true;

		const payload: any = {};
        const resultClone = structuredClone(this.OriginalResult);
        const newFindResultItems: IFindResultItem[] = [];
        const leftOverFindResultItems: IFindResultItem[] = [];

        for (const findResultItem of resultClone.findResults) {
            if (findResultItem.isPageHead) {
                newFindResultItems.push(findResultItem);
            } else {
                leftOverFindResultItems.push(findResultItem);
            }
        }

        resultClone.findResults = newFindResultItems;
        payload.findResult = resultClone;

        const replacementInput = document.querySelector("#replacement-input");
        if (replacementInput && replacementInput instanceof HTMLInputElement) {
            payload.replacement = replacementInput.value;
        } else {
            payload.replacement = "";
        }

		const response = await fetch(`/uplift/find-and-replace/replace`, {
			body:JSON.stringify(payload),
			method:"PATCH",
			cache:"no-cache",
			credentials:"same-origin",
			headers:{
				"Content-Type":"application/json"
			}
		});

		let data;
		try{
			/** @type {{status: int, error: ?string}} **/
			data = await response.json();
		}catch(jsonSyntaxError){
			alert("The server responded with invalid JSON.");
			this.IsProcessing = false;
			return;
		}

		this.IsProcessing = false;
	}
}