import { EndpointsProvider } from "../../EndpointsProvider";
import { ConfirmModal } from "../../Utils/ConfirmModal";
import { ManageRedirects } from "../ManageRedirects";

export class Redirect{
	private ManageRedirectsInstance: ManageRedirects;
    public Dom: HTMLElement;
    public IsRegex: number;
    public Id: number;
    public From: string;
    public To: string;
    public Status: number;
    public Timestamp: number;
    public PreserveQueryString: number;

    public constructor(
		manageRedirectsInstance: ManageRedirects,
        Id: number,
        IsRegex: number,
        From: string,
        To: string,
        Status: number,
        Timestamp: number,
        PreserveQueryString: number,
     ){
		this.ManageRedirectsInstance = manageRedirectsInstance;
        this.Id = Id;
		this.IsRegex = IsRegex;
		this.From = From;
		this.To = To;
		this.Status = Status;
		this.Timestamp = Timestamp;
		this.PreserveQueryString = PreserveQueryString;
		this.Dom = this.GetDOM();

		this.ManageRedirectsInstance.RedirectsComponentContainer.append(this.Dom);
    }
	
	/**
	 * Builds the HTML for a redirect row entry
	 * @returns 
	 */
    private GetDOM(): HTMLElement{
        // Set the icon HTML for regular expression symbolism
		let regexIcon = "";
		if (this.IsRegex === 1){
			regexIcon = `<i class="text-success bi bi-check-circle-fill"></i>`;
		}else if (this.IsRegex === 0){
			regexIcon = `<i class="text-danger bi bi-x-lg"></i>`;
		}

		// Set icon for preserving query string
		let preserveQueryStringIcon = "";
		if (this.PreserveQueryString === 1){
			preserveQueryStringIcon = `<i class="text-success bi bi-check-circle-fill"></i>`;
		}else if (this.PreserveQueryString === 0){
			preserveQueryStringIcon = `<i class="text-danger bi bi-x-lg"></i>`;
		}

		const dateString = (new Date(this.Timestamp * 1000)).toLocaleDateString();

		const template = document.createElement("redirect-row");
		template.classList.add("redirect-columns");
		template.innerHTML = `
			<div class="regex">
				${regexIcon}
			</div>
			<div class="from">
				${this.From}
			</div>
			<div class="to">
				${this.To}			
			</div>
			<div class="status">
				${this.Status}
			</div>
			<div class="preserve-query-string">
				${preserveQueryStringIcon}
			</div>
			<div class="timestamp">
				${dateString}
			</div>
			<div class="management-buttons">
				<button type="button" class="edit-redirect-button btn btn-sm btn-primary">
					<i class="bi bi-pencil-square"></i>
					<span>Edit</span>
				</button>
				<button type="button" class="delete-redirect-button btn btn-sm btn-danger">
					<i class="bi bi-trash-fill"></i>
				</button>
			</div>
		`;

		const deleteButton = template.querySelector(".delete-redirect-button");
		const editButton = template.querySelector(".edit-redirect-button");

		deleteButton.addEventListener("click", () => {
			this.OnDelete();
		});


		editButton.addEventListener("click", () => {
			this.OnEditButtonClicked();
		});

		return template;
    }

	/**
	 * Handles the edit button click event
	 */
    private OnEditButtonClicked(): void{
		this.ManageRedirectsInstance.EditRedirectForm.CurrentRedirectID = this.Id;

		this.ManageRedirectsInstance.EditRedirectModal.GetModal().querySelector<HTMLInputElement>("#edit-redirect-is-regex").checked = this.IsRegex === 1;
		this.ManageRedirectsInstance.EditRedirectModal.GetModal().querySelector<HTMLInputElement>("#edit-redirect-from").value = this.From;
		this.ManageRedirectsInstance.EditRedirectModal.GetModal().querySelector<HTMLInputElement>("#edit-redirect-to").value = this.To;
		const StatusElement = this.ManageRedirectsInstance.EditRedirectModal.GetModal().querySelector<HTMLSelectElement>("#edit-redirect-status");
		if (StatusElement) {
			StatusElement.value = String(this.Status);
		}
		this.ManageRedirectsInstance.EditRedirectModal.GetModal().querySelector<HTMLInputElement>("#edit-redirect-preserve-query-string").checked = this.PreserveQueryString === 1;
		this.ManageRedirectsInstance.EditRedirectModal.ShowModal();
    }

	/**
	 * Handles delete button click event
	 * @returns 
	 */
    private async OnDelete(): Promise<void>{
        const deleteModal = new ConfirmModal("remove-redirect-modal");
		deleteModal.SetTitle("Delete this redirect");
		deleteModal.SetContent("Are you sure you want to delete this redirect? If, after deletion, the page still redirects then clear your browser's cache. Optionally, you can use your browser's DevTools window and disable caching while that window is open.");
		deleteModal.SetConfirmButtonText("Delete");
		deleteModal.ShowModal();
		const didConfirm = await deleteModal.ActionTaken();
		deleteModal.Cleanup();

		if (didConfirm){
			const response = await fetch(`${EndpointsProvider.EndPoints.Redirects.DeleteRedirect}/${this.Id}`, {
			    method:"DELETE",
			    cache:"no-cache",
			    credentials:"same-origin"
			});

			let data: {status: number, error?: string};
			try{
			    data = await response.json();
			}catch(jsonSyntaxError){
			    alert("The server responded with invalid JSON.");
			    return;
			}

			if (data.status === 1){
				this.ManageRedirectsInstance.LoadRedirects(
					this.ManageRedirectsInstance.CurrentPage,
					this.ManageRedirectsInstance.CurrentLimit,
					this.ManageRedirectsInstance.CurrentQuery
				);
			}else if (data.status === -1){
				alert(data.error);
			}
		}
    }
}