import { EndpointsProvider } from "../EndpointsProvider";
import { Paginator } from "../Utils/Paginator";
import { Scheduler } from "../Utils/Scheduler";
import { NewReviewModal } from "./Modals/NewReviewModal";
//import { NewReviewForm } from "./Forms/NewReviewForm";
import { SearchReviewsForm } from "./Forms/SearchReviewsForm";
import { Review } from "./Components/Review";
import { IGetReviewsResponse } from "../Interfaces/ManageReviews/IGetReviewsResponse";
import { DeleteReviewModal } from "./Modals/DeleteReviewModal";
import { EditReviewModal } from "./Modals/EditReviewModal";
import {IReviewsOverviewData} from "../Interfaces/ManageReviews/IReviewsOverviewData";

export class ManageReviews {
    /**
     * Elements to show/hide loader and no results message.
     */
    private LoaderElement = document.querySelector<HTMLElement>("#reviews-loader");
    private NoResultsMessageContainer = document.querySelector("#no-reviews-message-container") as HTMLElement;
    private ReviewsContainer = document.querySelector<HTMLElement>(".review-info-rows");
    private OverviewEarliestYear: HTMLSpanElement = document.querySelector("#reviews-overview-earliest-year");
    private OverviewAverageRating: HTMLHeadingElement = document.querySelector("#reviews-overview-average-rating");
    private OverviewAverageRatingStarsContainer: HTMLDivElement = document.querySelector("#reviews-overview-average-rating-stars-container");
    private OverviewTotalReviews: HTMLHeadingElement = document.querySelector("#reviews-overview-total-reviews");
    private Overview30DaysAgoDate: HTMLSpanElement = document.querySelector("#reviews-overview-30-days-ago");
    private OverviewReviewsInLast30Days: HTMLHeadingElement = document.querySelector("#reviews-overview-reviews-in-30-days");
    private OverviewLoadedContainers: HTMLElement[] = Array.from(document.querySelectorAll(".overview-loaded-container"));
    private OverviewLoaders: HTMLElement[] = Array.from(document.querySelectorAll(".overview-loader"));
    private NewReviewModal: NewReviewModal;
    private TimeLoadingReviewsStarted: number;
    /**
     * Array for paginator instances.
     */
    private readonly Paginators: Paginator [] = [];
    /**
     * Static default values for pagination.
     */
    public static DEFAULT_PAGE: number = 1;
    public static DEFAULT_LIMIT: number = 10;
    public DeleteReviewModal: DeleteReviewModal;
    public EditReviewModal: EditReviewModal;
    /**
     * Instances to track pagination.
     */
    public CurrentPage: number = 1;
    public CurrentLimit: number = 10;
    public CurrentQuery: string = "";

    public constructor() {
        /**
        * @type {Paginator[]}
        */
        this.Paginators = [];
        /**
         * Selects all the pagination buttons, create paginators for them, and set up click events.
         */
        for (const paginator of Array.from(document.querySelectorAll<HTMLElement>("pagination"))) {
            const newPaginator = new Paginator(paginator);
            this.HookPaginatorEvents(newPaginator);
            this.Paginators.push(newPaginator);
        }
        /**
         * Loads the reviews right away when the class is created.
         */
        (async () => {
            this.LoadReviews(this.CurrentPage, this.CurrentLimit, null);
        })();

        this.NewReviewModal = new NewReviewModal(this)
            .Build()
            .RenderInto(document.body);
        //new NewReviewForm();

        document.querySelector<HTMLButtonElement>("#new-review-modal-trigger-button").addEventListener("click", () => {
            this.NewReviewModal.Show();
        });

        // Search input event listener
        const searchInput = document.querySelector<HTMLInputElement>("#review-search");
        searchInput.addEventListener("input", () => {
            this.CurrentQuery = searchInput.value.trim();
            this.LoadReviews(this.CurrentPage, this.CurrentLimit, this.CurrentQuery);
        });

        this.DeleteReviewModal = new DeleteReviewModal(this)
            .Build()
            .RenderInto(document.body);

        this.EditReviewModal = new EditReviewModal(this)
            .Build()
            .RenderInto(document.body);

        this.LoadOverviewData();
    }   

    /**
     * Adds event listeners for paginators to load reviews when the page is changed.
     * @param {Paginator} paginator
     */
    public HookPaginatorEvents(paginator: Paginator) {
        /**
         * Load reviews for previous page.
         */
        paginator.OnPagePrev(pageNumber => {
            this.CurrentPage = pageNumber;
            this.LoadReviews(
                pageNumber,
                this.CurrentLimit,
                this.CurrentQuery
            )
        });

        /**
         * Load reviews for the next page.
         */
        paginator.OnPageNext(pageNumber => {
            this.CurrentPage = pageNumber;
            this.LoadReviews(
                pageNumber,
                this.CurrentLimit,
                this.CurrentQuery
            )
        });

        /**
         * Load reviews for manually entered page number.
         */
        paginator.OnPageManuallyEntered(pageNumber => {
            this.CurrentPage = pageNumber;
            this.LoadReviews(
                pageNumber,
                this.CurrentLimit,
                this.CurrentQuery
            )
        });
    }

    /**
     * @param {int} page
     * @param {int} limit
     * @param {string|null} query
     * @returns {Promise<void>}
     */
    public async LoadReviews(
        page: number = undefined,
        limit: number = undefined,
        query: string = undefined
    ) {

        const timeCallStarted = (new Date()).getTime();
        this.TimeLoadingReviewsStarted = timeCallStarted;
        this.NoResultsMessageContainer.style.display = "none";

        if(page === undefined){
            page = 1;
        }
        
        if(limit === undefined){
            limit = 10;
        }

        if(query === undefined){
            query = ""; //value of search bar
        }

        const urlParams = new URLSearchParams();
        urlParams.set("page", String(page));
        urlParams.set("limit", String(limit));

        if (query !== null) {
            urlParams.set("query", query);
        }

        this.LoaderElement.style.display = "block";
        /**
         * Clear current reviews from the page.
         */
        this.ReviewsContainer.innerHTML = "";
        
        const response = await fetch(`${EndpointsProvider.EndPoints.Reviews.GetReviews}?${urlParams.toString()}`, {
            body: null,
            cache: "no-cache",
            credentials: "same-origin"
        });

        let data: IGetReviewsResponse;
        try {
            data = await response.json();
        } catch (jsonSyntaxError) {
            alert("The server responded with invalid JSON.");
        }

        /**
         * Force-wait a minimum of 0.35s so things don't flash.
         */
        await Scheduler.Wait(700);

        if(timeCallStarted !== this.TimeLoadingReviewsStarted){
            return;
        }

        if (data.status === 1) {
            this.LoaderElement.style.display = "none";

            // Set paginator stuff
            for (const paginator of this.Paginators) {
                // paginator.setCurrentPage(this.currentPage);
                paginator.SetMaxPages(data.totalPages);
            }

            if (data.reviews.length > 0) {
                for (const review of data.reviews) {
                    const component = new Review(review, this, this.EditReviewModal)
                        .Build()
                        .RenderInto(this.ReviewsContainer);
                }
            } else {
                this.NoResultsMessageContainer.style.display = null;
            }

        } else if (data.status === -1) {
            this.LoaderElement.style.display = "none";
        }
    }

    public async DeleteReview(reviewComponent: Review): Promise<boolean>{
        const response: Response = await fetch(`/uplift/review/${reviewComponent.ReviewObject.id}`, {
            method: "DELETE"
        });

        const apiResponse = await response.json();
        
        if("error" in apiResponse){
            const deleteErrorContainer = document.querySelector<HTMLDivElement>("#delete-review-error");
            const deleteErrorMessage = apiResponse.error;

            deleteErrorContainer.style.display = null;
            deleteErrorContainer.querySelector("div p").textContent = "This review has already been deleted, try refreshing your page.";

            return false;

        } else{
            const deleteErrorContainer = document.querySelector<HTMLDivElement>("#delete-review-error");
            deleteErrorContainer.style.display = "none";
            this.LoadReviews();

            return true;
        }
    }

    /**
     * Fetches the system's review overview data and then renders it into the data containers in the view.
     * @constructor
     */
    public async LoadOverviewData(): Promise<void>{
        const response: Response = await fetch(`/uplift/reviews/overview`);
        if (response.status === 200){
            const apiData: IReviewsOverviewData = await response.json();
            if (apiData.totalReviews > 0){
                const earliestReviewDate = new Date(apiData.firstReviewUnixTimestamp * 1000);
                this.OverviewEarliestYear.textContent = earliestReviewDate.getFullYear().toString();
                this.OverviewAverageRating.textContent = apiData.averageRating.toFixed(1).toString();
                this.OverviewTotalReviews.textContent = apiData.totalReviews.toString();
                this.Overview30DaysAgoDate.textContent = earliestReviewDate.toLocaleDateString("en-US", {month: "long", day: "2-digit", year: "numeric"});
                this.OverviewReviewsInLast30Days.textContent = apiData.newReviewsAddedInLast30Days.toString();

                // Stars
                this.OverviewAverageRatingStarsContainer.innerHTML = ``;
                for (let i = 0; i < Math.floor(apiData.averageRating); i++){
                    const starIcon = document.createElement("i");
                    starIcon.classList.add("bi", "bi-star-fill");
                    this.OverviewAverageRatingStarsContainer.append(starIcon);
                }
            }else{
                // No reviews
                this.OverviewEarliestYear.textContent = "(Never)";
                this.OverviewAverageRating.textContent = "N/A";
                this.OverviewTotalReviews.textContent = "0"
                this.Overview30DaysAgoDate.textContent = "(Never)";
                this.OverviewReviewsInLast30Days.textContent = "N/A";
            }

            this.OverviewLoadedContainers.forEach(element => element.style.display = null);
            this.OverviewLoaders.forEach(element => element.style.display = "none");
        }else{

        }
    }
}

new ManageReviews();