import type bootstrapTypes from "bootstrap";

/**
* Programmatic confirmation modal for general action confirming use.
* Such as confirming a deletion act after the button is pressed.
* Uses asynchronous programming to send confirm/cancel events back
* to the calling scope.
*/
export class ConfirmModal {

    private Modal: HTMLElement;
    private BSModal: bootstrapTypes.Modal;
    private AcceptPromise: ((value: boolean) => void) | null = null;
    private ActionPromise: Promise<boolean>;

    constructor(id: string) {
        this.Modal = this.Build(id);

        /** @type {Modal} */
        this.BSModal = new bootstrap.Modal(this.Modal);

        this.AcceptPromise = null;

        this.ActionPromise = new Promise((accept, reject) => {
            // Leak the "accept" function out of the scope
            // so other functions can call it
            this.AcceptPromise = accept;
        });

        // Connect events
        this.Modal.addEventListener("hide.bs.modal", () => {
            this.AcceptPromise(false);
            this.Modal.remove();
        });

        this.Modal.querySelector(".modal-confirm-button").addEventListener("click", () => {
            this.AcceptPromise(true);
        });

    }

    /**
    * Creates a new promise that can be awaited by a calling function
    * for when action is taken on the modal
    * @returns {Promise} Boolean when accepted. False if the modal is dismissed.
    */
    public ActionTaken() {
        return this.ActionPromise;
    }

    /**
    * Sets the modal's title
    * @param {string} title
    */
    public SetTitle(title: string) {
        this.Modal.querySelector(".modal-title").textContent = title;
    }

    /**
    * Sets the modal's textual content. Does not accept HTML
    * @param {string} content
    */
    public SetContent(content: string) {
        this.Modal.querySelector(".modal-body-inner").textContent = content;
    }

    /**
    * Sets the modal's HTML content.
    * @param {string} htmlContent
    */
    public SetHTMLContent(htmlContent: string) {
        this.Modal.querySelector(".modal-body-inner").innerHTML = htmlContent;
    }

    /**
    * Sets the text of the confirm button
    * @param {string} text
    */
    public SetConfirmButtonText(text: string) {
        this.Modal.querySelector(".modal-confirm-button").textContent = text;
    }

    /**
     * Sets the HTML of the confirm button
     * @param {string} html
     */
    public SetConfirmButtonHTML(html: string) {
        this.Modal.querySelector(".modal-confirm-button").innerHTML = html;

        // Confirm event needs to be reconnected
        this.Modal.querySelector(".modal-confirm-button").addEventListener("click", () => {
            this.AcceptPromise(true);
        });
    }

    /**
    * Shows the modal
    */
    public ShowModal() {
        this.BSModal.show();
    }

    /**
    * Hides the modal
    */
    public HideModal() {
        this.BSModal.hide();
    }

    /**
    * Hides the cancel button
    */
    public HideSecondaryButton() {
        this.Modal.querySelector<HTMLElement>(".modal-cancel-button").style.display = "none";
    }

    /**
    * Builds the DOM modal
    * @param {string} id
    * @returns {HTMLDivElement} The modal
    */
    public Build(id: string) {
        const modal = document.createElement("div");
        modal.classList.add("modal");
        modal.setAttribute("id", id);
        modal.setAttribute("tabindex", "-1");
        modal.setAttribute("role", "dialog");
        modal.setAttribute("aria-hidden", "true");
        modal.innerHTML = `
			<div class="modal-dialog">
				<div class="modal-content">
					<div class="modal-header bg-danger text-white">
						<h4 class="modal-title"></h4>
						<button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
					</div>
					<div class="modal-body">
						<p class="modal-body-inner"></p>
						<div class="submit-error my-2 p-2 text-white bg-danger" style="display:none;"></div>
					</div>
					<div class="modal-footer">
						<button type="button" class="modal-confirm-button btn btn-danger"></button>
						<button type="button" class="modal-cancel-button btn btn-gray" data-bs-dismiss="modal">
							<span>Cancel</span>
						</button>
					</div>
				</div>
			</div>
		`;

        document.body.append(modal);

        return modal
    }

    public Cleanup() {
        this.BSModal.hide();
    }
}