<?php

	namespace ActivityLogs;

	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\RenderEngine\Exceptions\LayoutDoesNotExist;
	use Nox\RenderEngine\Exceptions\ParseError;
	use Nox\RenderEngine\Exceptions\ViewFileDoesNotExist;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Roles\PermissionCategories;

	#[Controller]
	#[RouteBase("/uplift")]
	class ActivityLogController extends BaseController
	{
		/**
		 * @throws ParseError
		 * @throws ViewFileDoesNotExist
		 * @throws LayoutDoesNotExist
		 */
		#[Route("GET", "/view-activity-logs")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::VIEW_ACTIVITY_LOGS)]
		public function mainView(): string
		{
			return Renderer::renderView(
				viewFileName: "activity-logs/main.php",
				viewScope: [
					"activityLogCategories"=>ActivityLogCategories::cases(),
				],
			);
		}

		#[Route("GET", "/activity-logs")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::VIEW_ACTIVITY_LOGS)]
		public function getActivityLogs(Request $request): JSONResult
		{
			$limit = $request->getQueryValue("limit");
			$page = $request->getQueryValue("page");
			$category = $request->getQueryValue("category");
			$user = $request->getQueryValue("user");

			if ($limit === null){
				return new JSONError("Missing 'limit' GET parameter.");
			}

			if ($page === null){
				return new JSONError("Missing 'page' GET parameter.");
			}

			if ($category === null){
				return new JSONError("Missing 'category' GET parameter.");
			}

			if ($user === null){
				return new JSONError("Missing 'user' GET parameter.");
			}

			if($user === "") {
				$totalActivityLogs = ActivityLogsService::getTotalActivityLogs($category);

				return new JSONSuccess([
					"activityLogs"=>ActivityLogsService::getActivityLogs(
						page: $page,
						limit: $limit,
						category: $category,
						order: "DESC",
					),
					"totalActivityLogs"=>$totalActivityLogs,
					"totalPages"=>ceil($totalActivityLogs / $limit),
				]);
			} else {
				$totalActivityLogs = ActivityLogsService::getTotalActivityLogs($category, $user);

				return new JSONSuccess([
					"activityLogs"=>ActivityLogsService::getActivityLogs(
						page: $page,
						limit: $limit,
						category: $category,
						order: "DESC",
						user: $user,
					),
					"totalActivityLogs"=>$totalActivityLogs,
					"totalPages"=>ceil($totalActivityLogs / $limit),
				]);
			}
		}
	}