<?php
	namespace ClientProjectForm;

	use Accounts\Account;
	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use FileSystemUtilities\Exceptions\InvalidFileName;
	use FileSystemUtilities\Exceptions\InvalidFolderName;
	use FileSystemUtilities\exceptions\MaximumNewFolderDepthExceeded;
	use FileSystemUtilities\Exceptions\NewDirectoryWithSameNameExists;
	use FileSystemUtilities\Exceptions\PathDoesntExist;
	use GDHelper\Exceptions\AnimatedWebPNotSupported;
	use GDHelper\exceptions\FileNotFound;
	use GDHelper\Exceptions\InvalidImage;
	use GuzzleHttp\Exception\GuzzleException;
	use ImageGallery\ImageGalleryNameEmpty;
	use ImageGallery\ImageGalleryWithSameNameExists;
	use ImageManager\Exception\InvalidDirectory;
	use ImageManager\Exception\MissingParameter;
	use MonologWrapper\MonologWrapper;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\Request;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Page\Page;
	use ProjectPostTag\ProjectPostTag;
	use Roles\PermissionCategories;
	use Settings\RequireSettingValue;
	use Settings\Settings;
	use System\HttpHelper;
	use Uplift\Exceptions\MalformedValue;
	use Uplift\Exceptions\NoObjectFound;
	use Uplift\ImageManager\ImageFileAlreadyExists;
	use Uplift\ImageManager\ImageIsAThumb;
	use Uplift\ImageManager\ImageMissingThumb;
	use Uplift\ImageProcessing\Exceptions\ImageProcessingException;

	#[Controller]
	#[RouteBase("/uplift")]
	class ClientProjectFormController extends BaseController{

		#[Route("GET", "/ipp-form")]
		#[RequireLogin]
		#[RequireSettingValue(Settings::IPP_FORM_DISABLED, ["0", null])]
		public function formView(): string{

			return Renderer::renderView(
				viewFileName: "client-project-form/main-form.php",
			);
		}

		#[Route("POST", "/ipp-form/validate-image-file-name")]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::ACCESS_CLIENT_IPP_FORM)]
		#[RequireSettingValue(Settings::IPP_FORM_DISABLED, ["0", null])]
		public function validateImageFileName(Request $request): JSONResult{
			$payload = $request->getPayload();
			$logger = MonologWrapper::getLogger();
			$logger->info("Validating image file names.");

			try {
				$image = $payload->getFileUploadPayload("image");
			} catch (NoPayloadFound $e) {
				return new JSONError($e->getMessage());
			}

			try{
				ClientProjectFormService::validateProjectPhotoFileName($image);
			}catch(InvalidFileName $e){
				$logger->warning("Sent IPP form error back to user's browser: " . $e->getMessage());
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess();
		}

		#[Route("GET", "/ipp-form/project-tags")]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::ACCESS_CLIENT_IPP_FORM)]
		#[RequireSettingValue(Settings::IPP_FORM_DISABLED, ["0", null])]
		public function QueryAllProjectTags(Request $request): JSONResult{
			return new JSONSuccess([
				"tags"=>ProjectPostTag::query()
			]);
		}

		#[Route("POST", "/ipp-form/create-post")]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::ACCESS_CLIENT_IPP_FORM)]
		#[RequireSettingValue(Settings::IPP_FORM_DISABLED, ["0", null])]
		public function postIPP(Request $request): JSONResult{
			$payload = $request->getPayload();
			$logger = MonologWrapper::getLogger();
			$logger->info("Uploading project post.");

			try {
				$projectTitle = $payload->getTextPayload("project-title");
				$projectDescription = $payload->getTextPayload("project-description");
				$projectCityPageID = $payload->getTextPayload("city-page-id");
				$projectLocationCity = $payload->getTextPayload("project-location-city");
				$projectLocationStateNameShorthand = $payload->getTextPayload("project-location-state");
				$projectLocationStateName = $payload->getTextPayload("project-location-state-full");
				$projectPrimaryServicePageID = $payload->getTextPayload("service-page-id");
				$projectBudget = $payload->getTextPayload("project-budget");
				$projectBrandsMaterials = $payload->getTextPayload("project-materials");
				$projectCustomerLeftReview = $payload->getTextPayload("customer-review-check"); // "0" or "1"
				$projectCustomerFirstName = $payload->getTextPayload("customer-first-name");
				$projectCustomerLastName = $payload->getTextPayload("customer-last-name");
				$projectCustomerTestimonial = $payload->getTextPayload("project-customer-testimonial");
				$jsonProjectTagIds = $payload->getTextPayload("json-project-tag-ids");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			// Decode the project tag Ids for this post
			$projectTagIds = json_decode($jsonProjectTagIds->contents, true);
			if ($projectTagIds === null){
				$projectTagIds = [];
			}

			try {
				$projectPage = ClientProjectFormService::createProjectPost(
					projectTitle: $projectTitle->contents,
					projectDescription: $projectDescription->contents,
					projectCityPageID: $projectCityPageID->contents,
					projectLocationCity: $projectLocationCity->contents,
					projectLocationStateShorthand: $projectLocationStateNameShorthand->contents,
					projectLocationState: $projectLocationStateName->contents,
					projectPrimaryServicePageID: $projectPrimaryServicePageID->contents,
					projectBudget: $projectBudget->contents,
					projectBrandsMaterials: $projectBrandsMaterials->contents,
					didLeaveCustomerReview: $projectCustomerLeftReview->contents,
					customerReviewFirstName: $projectCustomerFirstName->contents,
					customerReviewLastName: $projectCustomerLastName->contents,
					customerReviewTestimonial: $projectCustomerTestimonial->contents,
					projectTagIds: $projectTagIds
				);
			} catch (Exceptions\ProjectValidationError|NoObjectFound $e) {
				return new JSONError($e->getMessage());
			}

			// Needs to create an image directory to store images in and return the image directory with the new
			// page ID to the front-end. That way all submitted media can be stored in that directory
			try {
				$projectImagesDirectory = ClientProjectFormService::createImageDirectoryForProject($projectTitle->contents);

				ActivityLog::log(
					categoryID: ActivityLogCategories::UPLOADED_IPP_VIA_FORM->value,
					accountID: (Account::getCurrentUser())->id,
					ip: $request->getIP(),
					jsonData: json_encode([
						"pageID"=>$projectPage->id,
						"ippTitle"=>$projectTitle->contents
					]),
				);
			} catch (Exceptions\ProjectValidationError|InvalidFolderName|PathDoesntExist|MaximumNewFolderDepthExceeded|InvalidDirectory|MissingParameter|NewDirectoryWithSameNameExists $e) {
				$logger->critical("IPP error generating image folder: " . $e->getMessage());
				return new JSONError("There was an internal problem generating a directory to store your images. Please email support@footbridgemedia.com with a screenshot of this error, the name of your project, and the images you want to attach it to.");
			}

			$logger->info("Project post created.");

			return new JSONSuccess([
				"newPageID"=>$projectPage->id,
				"projectImagesDirectory"=>realpath($projectImagesDirectory),
			]);
		}

		#[Route("POST", "/ipp-form/cover-photo")]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::ACCESS_CLIENT_IPP_FORM)]
		#[RequireSettingValue(Settings::IPP_FORM_DISABLED, ["0", null])]
		public function uploadCoverPhoto(Request $request): JSONResult{
			$payload = $request->getPayload();
			$logger = MonologWrapper::getLogger();
			$logger->info("Uploading project post cover photo.");

			try {
				$coverPhoto = $payload->getFileUploadPayload("image");
				$projectPostID = $payload->getTextPayload("project-post-id");
				$projectImagesDirectory = $payload->getTextPayload("project-images-directory");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				ClientProjectFormService::uploadCoverPhotoAndPlaceInContent(
					coverPhoto: $coverPhoto,
					projectPageID: $projectPostID->contents,
					projectImagesDirectory: $projectImagesDirectory->contents,
				);

				ActivityLog::log(
					categoryID: ActivityLogCategories::UPLOADED_IMAGE_VIA_FORM->value,
					accountID: (Account::getCurrentUser())->id,
					ip: $request->getIP(),
					jsonData: json_encode([
						"imageFileName"=>$coverPhoto->fileName,
					]),
				);

			} catch (InvalidFileName|AnimatedWebPNotSupported|InvalidImage|FileNotFound|InvalidDirectory|ImageFileAlreadyExists|ImageIsAThumb|ImageProcessingException $e) {
				$logger->error("IPP cover photo upload error: " . $e->getMessage());
				return new JSONError("There was an error uploading your cover photo and your project post was not able to be submitted. Please inform your account rep of this and attach the photos with the name of the project they should go to.");
			}

			return new JSONSuccess();
		}

		#[Route("POST", "/ipp-form/image")]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::ACCESS_CLIENT_IPP_FORM)]
		#[RequireSettingValue(Settings::IPP_FORM_DISABLED, ["0", null])]
		public function uploadProjectImage(Request $request): JSONResult{
			$payload = $request->getPayload();
			$logger = MonologWrapper::getLogger();
			$logger->info("Uploading a project image.");

			try {
				$image = $payload->getFileUploadPayload("image");
				$projectImagesDirectory = $payload->getTextPayload("project-images-directory");
			}catch(NoPayloadFound $e){
				$logger->error("IPP image upload payload error: " . $e->getMessage());
				return new JSONError($e->getMessage());
			}

			$logger->info("Image name being uploaded is: " . $image->fileName);

			try {
				$modifiedFileNameThatWasUploaded = ClientProjectFormService::uploadProjectImage(
					image: $image,
					projectImagesDirectory: $projectImagesDirectory->contents,
				);

				ActivityLog::log(
					categoryID: ActivityLogCategories::UPLOADED_IMAGE_VIA_FORM->value,
					accountID: (Account::getCurrentUser())->id,
					ip: $request->getIP(),
					jsonData: json_encode([
						"imageFileName"=>$image->fileName,
					]),
				);

				return new JSONSuccess([
					"fileName"=>$modifiedFileNameThatWasUploaded,
				]);
			} catch (InvalidFileName|AnimatedWebPNotSupported|InvalidImage|FileNotFound|InvalidDirectory|ImageFileAlreadyExists|ImageIsAThumb|ImageProcessingException $e) {
				$logger->error("Issue uploading image " . $image->fileName . ": " . $e->getMessage());
				return new JSONError("There was an error uploading this photo.");
			}
		}

		#[Route("POST", "/ipp-form/finalize")]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::ACCESS_CLIENT_IPP_FORM)]
		#[RequireSettingValue(Settings::IPP_FORM_DISABLED, ["0", null])]
		public function finalizeProjectPostUpload(Request $request): JSONResult{
			$payload = $request->getPayload();
			$logger = MonologWrapper::getLogger();
//			$logger->info("Finalizing IPP upload and sending to master server for queueing.");
			$logger->info("Finalizing IPP upload.");

			try {
				$projectPostID = $payload->getTextPayload("project-post-id");
				$projectImagesDirectory = $payload->getTextPayload("project-images-directory");
				$jsonListOfUploadedProjectImageFileNames = $payload->getTextPayload("json-list-of-uploaded-image-file-names");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			// Try to parse the JSON list of image file names into an array
			$listOfUploadedProjectImageFileNames = json_decode($jsonListOfUploadedProjectImageFileNames->contents, true);
			if ($listOfUploadedProjectImageFileNames === null){
				$logger->error("Failed to parse the JSON list of image file names. The JSON string that was provided was: " . $jsonListOfUploadedProjectImageFileNames->contents);
				return new JSONError("Internal error. Failed to process the list of JSON image file names. Please report this to a developer.");
			}

			try {
				ClientProjectFormService::addProjectGalleryToPost(
					projectPostID: $projectPostID->contents,
					projectImagesDirectory: $projectImagesDirectory->contents,
					listOfImageFileNames: $listOfUploadedProjectImageFileNames,
				);
			} catch (AnimatedWebPNotSupported|FileNotFound|InvalidImage|ImageGalleryNameEmpty|ImageGalleryWithSameNameExists|ImageMissingThumb $e) {
				$logger->critical("Error inserting the image gallery to the project post: " . $e->getMessage());
				return new JSONError("There was an error inserting the image gallery into your project post. Your IPP has been uploaded and your images are saved. Please let your account rep know of this error and which project it was for and our support team will resolve the issue.");
			}

//			try {
//				ClientProjectFormService::submitProjectPostSubmissionToMasterServer($projectPostID->contents);
//			}catch(MalformedValue $e){
//				$logger->critical("Error submitting the project post to the master control server: " . $e->getMessage());
//				return new JSONError("The project post was submitted and saved to your website's backend system, but the system failed to notify the Footbridge Support team due to a malformed value. Please screenshot this and send it to support@footbridgemedia.com. You do not need to resubmit your project.");
//			}catch(GuzzleException $e){
//				$logger->critical("Error submitting the project post to the master control server. Is the control panel down? " . $e->getMessage());
//				return new JSONError("The project post was submitted and saved to your website's backend system, but the system failed to notify the Footbridge Support team due to an HTTP error when notifying the team. Please screenshot this and send it to support@footbridgemedia.com. You do not need to resubmit your project.");
//			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::FINALIZED_IPP_UPLOAD_VIA_FORM->value,
				accountID: (Account::getCurrentUser())->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"pageID"=>$projectPostID->contents,
				]),
			);

//			$logger->info("IPP finalized. Successfully saved to CMS and published to master control panel.");
			$logger->info("IPP finalized. Successfully saved to CMS.");

			// Get the project route
			/** @var ?Page $projectPage */
			$projectPage = Page::fetch($projectPostID->contents);

			return new JSONSuccess([
				"projectURL"=>$projectPage->pageRoute,
			]);
		}

		#[Route("DELETE", "/\/ipp-form\/undo-ipp-upload\/(?<pageID>\d+)/", true)]
		#[UseJSON]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::ACCESS_CLIENT_IPP_FORM)]
		#[RequireSettingValue(Settings::IPP_FORM_DISABLED, ["0", null])]
		public function undoProjectPost(Request $request): JSONResult{
			$pageID = $request->getParameter("pageID");

			$logger = MonologWrapper::getLogger();
			$logger->info("Undoing a project post. Page ID: $pageID.");


			try {
				ClientProjectFormService::undoProjectPostSubmission(
					projectPostID: $pageID,
				);
			}catch(\Exception $e){
				http_response_code(500);
				$logger->error($e->getMessage());
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess();
		}
	}