<?php
	namespace EasyEditor;

	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\Http\Rewrite;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Page\ArticleCategory;
	use Page\Page;
	use PageArchives\PageArchivesService;
	use PageEditor\PageEditorService;
	use ProjectPostTag\ProjectPostTag;
	use Roles\PermissionCategories;
	use Settings\Setting;

	#[Controller]
	#[RouteBase("/uplift/easy-editor")]
	class EasyEditorController extends BaseController{

		#[Route("GET", "/")]
		#[Route("GET", "@^/(?<pageID>\d+)$@", true)]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::ACCESS_EASY_EDITOR)]
		public function easyEditorView(Request $request): string|Rewrite
		{
			$pageID = (int) $request->getParameter("pageID");

			/** @var ?Page $page */
			$page = null;
			if ($pageID !== 0){
				/** @var Page $page */
				$page = Page::fetch($pageID);

				if ($page === null){
					return new Rewrite(
						path:"/404",
						statusCode: 404,
					);
				}
			}

			return Renderer::renderView(
				viewFileName:"easy-editor/main.php",
				viewScope:[
					"page"=>$page,
					"articleCategories"=>ArticleCategory::query(),
					"projectPostTags"=>ProjectPostTag::query(),
				],
			);
		}

		#[Route("GET", "/phone-and-fax-numbers")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::ACCESS_EASY_EDITOR)]
		public function getPhoneAndFaxNumbers(): JSONResult
		{
			$phoneNumbers = Setting::getSettingValue("companyPhoneNumbers");
			$faxNumbers = Setting::getSettingValue("companyFaxNumbers");

			$phoneNumbersArray = [];
			$faxNumbersArray = [];

			if ($phoneNumbers !== null){
				$phoneNumbersArray = json_decode($phoneNumbers, true);
			}

			if ($faxNumbers !== null){
				$faxNumbersArray = json_decode($faxNumbers, true);
			}

			return new JSONSuccess([
				"phoneNumbers" => $phoneNumbersArray,
				"faxNumbers" => $faxNumbersArray
			]);
		}

		#[Route("GET", "/pages")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::ACCESS_EASY_EDITOR)]
		public function getPages(): JSONResult
		{
			return new JSONSuccess([
				"pages" => Page::query(),
			]);
		}

		#[Route("POST", "/^\/save-page\/(?<pageId>\d+)$/", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::ACCESS_EASY_EDITOR)]
		public function savePage(Request $request): JSONResult
		{
			$pageId = (int) $request->getParameter("pageId");
			$payload = $request->getPayload();

			try{
				$content = $payload->getTextPayload("content");
				$pageContentSectionsJson = $payload->getTextPayload("page-content-sections");
				$pageContentSections = json_decode($pageContentSectionsJson->contents, true);

				/** @var ?Page $page */
				$page = Page::fetch($pageId);

				if($page !== null){
					$page->pageBody = $content->contents;
					$page->save();

					$updatedPageContentSections = PageEditorService::savePageContentSections($page, $pageContentSections);

					if ($page->pageType === "Blog"){
						PageArchivesService::archiveBlogPage($page);
					} elseif ($page->pageType === "General"){
						PageArchivesService::archiveGeneralPage($page);
					} elseif ($page->pageType === "Service"){
						PageArchivesService::archiveServicePage($page);
					} elseif ($page->pageType === "City"){
						PageArchivesService::archiveCityPage($page);
					} elseif ($page->pageType === "Project"){
						PageArchivesService::archiveProjectPage($page);
					}
				}

				return new JSONSuccess([
					"updatedPageContentSections"=>$updatedPageContentSections,
				]);
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}
		}
	}