<?php
	namespace ExternalSiteImporter;

	use DOMNode;
	use GuzzleHttp\Client;
	use GuzzleHttp\Exception\GuzzleException;
	use Page\Page;
	use Page\PageType;
	use Page\PublicationStatus;
	use PageArchives\PageArchivesService;
	use PageEditor\PageEditorService;
	use Redirects\DuplicateFromValue;
	use Redirects\RedirectsService;
	use Uplift\Exceptions\EmptyValue;
	use Uplift\Exceptions\MalformedValue;

	class ExternalSiteImporterService{

		public static function getInnerHTMLOfDOM(DOMNode $element): string{
			$innerHTML = "";
			/** @var DOMNode $child */
			foreach($element->childNodes as $child){
				$innerHTML .= $child->ownerDocument->saveHTML($child);
			}

			return trim($innerHTML);
		}

		/**
		 * @throws NoBodyElement
		 * @throws GuzzleException
		 * @throws NoHeadElement
		 * @throws EmptyValue
		 */
		public static function fetch(string $url): ExternalSiteFetchResult{

			if (empty($url)){
				throw new EmptyValue("The URL provided is empty.");
			}

			$client = new Client();
			$response = $client->request(
				method:"GET",
				uri:$url,
			);

			$result = new ExternalSiteFetchResult();
			$htmlResponseBody = $response->getBody()->getContents();

			libxml_use_internal_errors(true);
			$document = new \DOMDocument();
			$document->preserveWhiteSpace = false;
			$document->loadHTML(
				source: $htmlResponseBody,
			);

			$headElements = $document->getElementsByTagName("head");
			$bodyElements = $document->getElementsByTagName("body");

			if ($headElements->count() === 0){
				throw new NoHeadElement("The response from that URL did not have an HTML head element.");
			}

			if ($bodyElements->count() === 0){
				throw new NoBodyElement("The response from that URL did not have an HTML body element.");
			}

			$headElement = $headElements->item(0);
			$bodyElement = $bodyElements->item(0);

			$result->response = $response;

			ImportFilters::runHeadFilter($headElement);
			ImportFilters::runBodyFilter($bodyElement);

			$bodyString = self::getInnerHTMLOfDom($bodyElement);
			$bodyString = ImportFilters::runStringFilter($bodyString);
			$headString = self::getInnerHTMLOfDom($headElement);
			$headString = ImportFilters::runStringFilter($headString);

			$result->headDOMNode = $headElement;
			$result->bodyDOMNode = $bodyElement;
			$result->head = $headString;
			$result->body = $bodyString;

			return $result;
		}

		/**
		 * @throws \Nox\ORM\Exceptions\NoPrimaryKey
		 * @throws \PageEditor\Exceptions\PageLayoutDoesntExist
		 * @throws \PageEditor\Exceptions\PageRouteIsBlank
		 * @throws \PageEditor\Exceptions\PageLayoutIsBlank
		 * @throws \PageEditor\Exceptions\PageNameIsBlank
		 * @throws \PageEditor\Exceptions\PageNameExists
		 * @throws MalformedValue
		 * @throws EmptyValue
		 * @throws \PageEditor\Exceptions\PublicationStatusDoesntExist
		 * @throws \PageEditor\Exceptions\PageNameExistsWithSamePageType
		 * @throws \PageEditor\Exceptions\PageRouteInUse
		 */
		public static function buildPageFromImport(
			string $urlImportedFrom,
			string $pageName,
			string $pageType,
			string $pageHead,
			string $pageBody,
			string $pageRoute,
			string $pageLayout,
			string $cityName,
			string $stateName,
			string $stateNameShorthand,
			string $cityURL,
			string $breadcrumbsJSON,
		): Page{

			if (empty($pageName)){
				throw new EmptyValue("pageName cannot be empty.");
			}

			if (!PageType::has($pageType)){
				throw new MalformedValue("pageType value doesn't correspond to any known PageType enum.");
			}

			// Check that the page route begins with a forward slash
			if (!str_starts_with($pageRoute, "/")){
				throw new MalformedValue("Page routes must begin with a forward slash.");
			}

			if (empty($pageRoute)){
				throw new EmptyValue("pageRoute cannot be empty.");
			}

			$page = PageEditorService::createPage(
				pageName:$pageName,
				pageType:$pageType,
			);

			PageEditorService::saveBreadcrumbs($page, json_decode($breadcrumbsJSON, true));

			if ($pageType === PageType::General->name){
				// Save the location data
				PageEditorService::saveGeneralPage(
					page: $page,
					pageName: $pageName,
					pageRoute: $pageRoute,
					pageRouteIsRegex: false,
					pageLayout: $pageLayout,
					pageBody: $pageBody,
					pageHead: $pageHead,
					excludedFromSitemap: false,
					excludedFromSchema: false,
					publicationStatus: PublicationStatus::Published->value,
					publicationTimestamp: time(),
				);

				PageArchivesService::archiveGeneralPage($page);
			}elseif ($pageType === PageType::Service->name) {
				PageEditorService::saveServicePage(
					page: $page,
					pageName: $pageName,
					pageRoute: $pageRoute,
					pageRouteIsRegex: false,
					pageLayout: $pageLayout,
					pageBody: $pageBody,
					pageHead: $pageHead,
					excludedFromSitemap: false,
					excludedFromSchema: false,
					publicationStatus: PublicationStatus::Published->value,
					publicationTimestamp: time(),
					featuredImageURI: "",
					featuredImageThumbURI: "",
				);

				PageArchivesService::archiveServicePage($page);
			}elseif ($pageType === PageType::Blog->name) {
				PageEditorService::saveBlogPage(
					page: $page,
					pageName: $pageName,
					pageRoute: $pageRoute,
					pageRouteIsRegex: false,
					pageLayout: $pageLayout,
					pageBody: $pageBody,
					pageHead: $pageHead,
					excludedFromSitemap: false,
					excludedFromSchema: false,
					publicationStatus: PublicationStatus::Published->value,
					publicationTimestamp: time(),
					featuredImageURI: "",
					featuredImageThumbURI: "",
					articleCategoryIDs: [],
				);

				PageArchivesService::archiveBlogPage($page);
			}elseif ($pageType === PageType::Project->name){
				PageEditorService::saveProjectPage(
					page: $page,
					pageName: $pageName,
					pageRoute: $pageRoute,
					pageRouteIsRegex: false,
					pageLayout: $pageLayout,
					pageBody: $pageBody,
					pageHead: $pageHead,
					excludedFromSitemap: false,
					excludedFromSchema: false,
					publicationStatus: PublicationStatus::Published->value,
					publicationTimestamp: time(),
					featuredImageURI: "",
					featuredImageThumbURI: "",
					cityName: $cityName,
					stateName: $stateName,
					stateNameShorthand: $stateNameShorthand,
					brandsProducts: "",
					customerDidLeaveReview: false,
					customerReviewFirstName: "",
					customerReviewLastName: "",
					customerReviewTestimonial: "",
					projectTagIDs: [],
				);

				PageArchivesService::archiveProjectPage($page);
			}elseif ($pageType === PageType::City->name){
				PageEditorService::saveCityPage(
					page: $page,
					pageName: $pageName,
					pageRoute: $pageRoute,
					pageRouteIsRegex: false,
					pageLayout: $pageLayout,
					pageBody: $pageBody,
					pageHead: $pageHead,
					excludedFromSitemap: false,
					excludedFromSchema: false,
					publicationStatus: PublicationStatus::Published->value,
					publicationTimestamp: time(),
					cityName: $cityName,
					stateName: $stateName,
					stateNameShorthand: $stateNameShorthand,
					officialCityURL: $cityURL,
					featuredImageURI: "",
					featuredImageThumbURI: "",
				);

				PageArchivesService::archiveCityPage($page);
			}

			// Save content to the first section of the page, if there is sectioned content
			// for this page's layout
			$page->saveToDefaultContentSectionOrFirstSection($pageBody);

			if (!empty($urlImportedFrom)){
				// Try and make a redirect from it
				$relativeURL = parse_url($urlImportedFrom, PHP_URL_PATH);
				if (!empty($relativeURL) && $relativeURL !== "/"){
					// Check that the relativeURL is not the same as the route for the new page
					if (strtolower($pageRoute) !== strtolower($relativeURL)) {
						// Ignore an exception here
						try {
							RedirectsService::addRedirect(
								from: $relativeURL,
								to: $pageRoute,
								isRegex: false,
								status: 301,
								preserveQueryString: false,
							);
						} catch (DuplicateFromValue) {
						}
					}
				}
			}

			return $page;
		}
	}