<?php

	namespace FileSystemUtilities;

	use GDHelper\exceptions\AnimatedWebPNotSupported;
	use GDHelper\exceptions\FileNotFound;
	use GDHelper\exceptions\InvalidImage;
	use GDHelper\GDHelper;

	class FSImageFile
	{

		public string $fileExtension;
		public string $fileNameWithoutExtension;
		public int $fileSize;
		public string $fileSizeHumanReadable;
		public ?string $thumbURI = null;
		public ?string $thumbFilePath = null;
		public ?FSImageFile $fallbackImage = null;

		/** @var int $imageType The GD image type constant */
		public int $imageType;

		/** @var ?GDHelper The GDHelper for this image */
		public ?GDHelper $gdHelper;

		/**
		 * FSImageFile constructor.
		 * @param string $fileName File's base name with no directory path
		 * @param string $fullFilePath
		 * @param string|null $uri
		 * @param bool $acceptSVG
		 * @throws AnimatedWebPNotSupported
		 * @throws FileNotFound
		 * @throws InvalidImage
		 */
		public function __construct(
			/**
			 * File's base name with no directory path
			 */
			public string $fileName,
			public string $fullFilePath,
			public ?string $uri,
			bool $acceptSVG = false,
			bool $createGDHelper = true,
		)
		{

			// For the image extension to be lowercase
			$this->fileExtension = strtolower(pathinfo($this->fullFilePath, PATHINFO_EXTENSION));

			if ( ($acceptSVG && $this->fileExtension === "svg") || $this->fileExtension !== "svg") {

				$this->fileNameWithoutExtension = pathinfo($this->fullFilePath, PATHINFO_FILENAME);
				$this->fileSize = filesize($this->fullFilePath);
				$this->fileSizeHumanReadable = FileSystemUtilities::bitsToHumanReadable($this->fileSize);

				// There is no guarantee a thumb exists for the image, but the standard is always in
				// the image's directory's folder named "thumbs"
				$thumbFileLocation = sprintf(
					"%s/thumbs/%s",
					dirname($this->fullFilePath),
					$fileName,
				);
				if (file_exists($thumbFileLocation)) {
					if ($uri !== null) {
						$this->thumbURI = sprintf("%s/thumbs/%s", dirname($uri), $fileName);
					}
					$this->thumbFilePath = $thumbFileLocation;
				}

				if ($this->fileExtension !== "svg") {
					if ($createGDHelper) {
						$this->gdHelper = GDHelper::fromFilePath($this->fullFilePath);
					}
				}
			} elseif (!$acceptSVG && $this->fileExtension === "svg") {
				throw new InvalidImage("SVG file is not allowed by this call.");
			}
		}
	}