<?php

	namespace FileSystemUtilities;

	use FileSystemUtilities\exceptions\InvalidFolderName;
	use FileSystemUtilities\exceptions\MaxDirectoryDepthExceeded;
	use FileSystemUtilities\exceptions\MaximumNewFolderDepthExceeded;
	use FileSystemUtilities\exceptions\NewDirectoryWithSameNameExists;
	use FileSystemUtilities\exceptions\PathDoesntExist;
	use function gatherChildren;

	/**
	 * Static methods to manipulate or fetch complex data from a file system
	 */
	class FileSystemUtilities
	{

		/**
		 * Recursively clones a folder and all descendants
		 *
		 * @param string $parentFolderPath
		 * @param string $destination
		 * @return void
		 */
		public static function cloneFolder(
			string $parentFolderPath,
			string $destination
		): void
		{
			$contents = array_diff(scandir($parentFolderPath), ['.', '..']);
			@mkdir($destination, 0777, true);

			foreach ($contents as $fileName) {
				$filePath = $parentFolderPath . "/$fileName";

				if (is_dir($filePath)) {
					self::cloneFolder($filePath, $destination . "/$fileName");
				} else {
					copy($filePath, $destination . "/$fileName");
				}
			}
		}

		/**
		 * Safely renames or moves a directory and properly migrates every child
		 * directory and file. If the folder name at the end of the new $newDirectoryPath
		 * does not exist, then it will be created. But folders in between will not be recursively created.
		 * @throws InvalidFolderName
		 * @throws PathDoesntExist|NewDirectoryWithSameNameExists
		 */
		public static function moveDirectoryToNewLocation(string $currentDirectoryPath, string $newDirectoryPath): void
		{
			if (file_exists($newDirectoryPath)) {
				throw new NewDirectoryWithSameNameExists("A directory with that name already exists.");
			} else {
				$currentDirectoryPath = realpath($currentDirectoryPath);
				$newParentDirectoryPath = realpath(dirname($newDirectoryPath));

				if (!$currentDirectoryPath) {
					throw new PathDoesntExist("The current directory path provided does not exist.");
				}

				if (!$newParentDirectoryPath) {
					throw new PathDoesntExist("The new directory path's parent directory doesn't exist.");
				}

				$newFolderName = basename($newDirectoryPath);
				if (str_starts_with($newFolderName, ".")) {
					throw new InvalidFolderName("The folder name %s is not a valid name for a directory in this system..");
				}

				$newResolvedPath = sprintf("%s/%s", $newParentDirectoryPath, $newFolderName);
				mkdir($newResolvedPath, 0777);

				// Recursively move/rename all the child content to be under the new folder location as well
				self::renameDirectory_iterateChildren($currentDirectoryPath, $newResolvedPath, $currentDirectoryPath);
				self::recursivelyDeleteFolder($currentDirectoryPath);
			}
		}

		/**
		 * Iterator function used in self::renameDirectory
		 * @param string $baseDirectory The base directory that is being renamed. This should never change between iterated calls to this function. Example C:\something\images\folder-1
		 * @param string $newBaseDirectory The new base directory after folder name in the $baseDirectory was renamed. This should never change between iterated calls to this function. Example C:\something\images\folder-new-name-1
		 * @param string $directory The current directory that is being iterated through that is a part of the $baseDirectory. Example C:\something\images\folder-1\projects\house-project
		 */
		private static function renameDirectory_iterateChildren(
			string $baseDirectory,
			string $newBaseDirectory,
			string $directory
		): void
		{
			$directoryHandle = opendir($directory);
			while (($fileName = readdir($directoryHandle)) !== false) {
				if ($fileName !== "." && $fileName !== "..") {
					$fullPath = sprintf("%s/%s", $directory, $fileName);
					if (is_dir($fullPath)) {
						// Create the directory in the $baseDirectory chain
						$thisDirectoryAsAStub = str_replace($baseDirectory, "", $fullPath);
						$thisDirectoryInTheNewBaseDirectory = sprintf("%s%s", $newBaseDirectory, $thisDirectoryAsAStub);
						mkdir($thisDirectoryInTheNewBaseDirectory, 0777);
						self::renameDirectory_iterateChildren($baseDirectory, $newBaseDirectory, $fullPath);
					} else {
						// It's a file
						// It needs to be moved
						$thisFilePathAsAStub = str_replace($baseDirectory, "", $fullPath);
						$thisFilePathAsAStubInTheNewBaseDirectory = sprintf("%s%s", $newBaseDirectory, $thisFilePathAsAStub);
						rename($fullPath, $thisFilePathAsAStubInTheNewBaseDirectory);
					}
				}
			}
			closedir($directoryHandle);
		}

		/**
		 * Deletes a folder and handles sub-folder and file recursion
		 *
		 * @param string $folderPath
		 */
		public static function recursivelyDeleteFolder(string $folderPath): void
		{
			$dir = opendir($folderPath);
			while (false !== ($file = readdir($dir))) {
				if (($file != '.') && ($file != '..')) {
					$full = $folderPath . '/' . $file;
					if (is_dir($full)) {
						self::recursivelyDeleteFolder($full);
					} else {
						unlink($full);
					}
				}
			}
			closedir($dir);
			rmdir($folderPath);
		}

		/**
		 * Converts a string to a file-safe string
		 *
		 * @param string $name
		 * @return string
		 */
		public static function getFileSafeName(string $name): string
		{
			$fileName = "";
			foreach (str_split($name) as $character) {
				if (preg_match("/[A-Za-z0-9\s]/i", $character) !== 0) {
					if ($character == " ") {
						$fileName .= "-";
					} else {
						$fileName .= strtolower($character);
					}
				}
			}

			// Replace repetitive dashes
			$fileName = preg_replace("/-{2,}/", "-", $fileName);

			return $fileName;
		}

		/**
		 * Checks if a string is a file safe name
		 *
		 * @param string $fileName
		 * @return bool
		 */
		public static function isNameFileSafe(string $fileName): bool
		{
			if ($fileName === "." || $fileName === "..") {
				return false;
			}

			return preg_match("/^[A-Za-z0-9\s()\-_\.\%]+$/i", $fileName) === 1;
		}

		/**
		 * Converts a number of bits to human-readable size
		 *
		 * From: https://stackoverflow.com/questions/5501427/php-filesize-mb-kb-conversion This function is different from what getMaxUploadSize returns in that PHP uses M instead of MB for megabyte and so on - where this returns MB instead of M.
		 *
		 * @param int $size in bits
		 * @return string
		 */
		public static function bitsToHumanReadable(int $size): string
		{
			$units = array('B', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB');
			$power = $size > 0 ? floor(log($size, 1024)) : 0;
			return number_format($size / pow(1024, $power), 0, '.', ',') . $units[$power];
		}

		/**
		 * Used to parse a human-readable file size into bytes for integer comparison
		 * @param string $size The size such as 7M
		 * @return int
		 */
		public static function humanReadableToBytes(string $size): int
		{
			$unit = preg_replace('/[^bkmgtpezy]/i', '', $size); // Remove the non-unit characters from the size.
			$size = preg_replace('/[^0-9\.]/', '', $size); // Remove the non-numeric characters from the size.
			if ($unit) {
				// Find the position of the unit in the ordered string which is the power of magnitude to multiply a kilobyte by.
				return round($size * pow(1024, stripos('bkmgtpezy', $unit[0])));
			} else {
				return round($size);
			}
		}

		/**
		 * Returns a multi-dimensional array of the contents of a folder
		 *
		 * This method is recursive to infinite sub-folders
		 */
		public static function fetchDirectoryAndDescendants(string $directoryStartPath): FSDirectory
		{
			$startPath = realpath($directoryStartPath);
			$maximumFolderDepth = 10;
			$currentFolderDepth = 0;

			$startDirectory = new FSDirectory(
				basename($startPath),
				$startPath,
			);

			self::gatherChildren($startDirectory, $maximumFolderDepth, $currentFolderDepth);
			return $startDirectory;
		}

		private static function gatherChildren(
			FSDirectory $startDirectory,
			$maximumFolderDepth,
			$currentFolderDepth
		): FSDirectory{
			++$currentFolderDepth;

			if ($currentFolderDepth > $maximumFolderDepth) {
				throw new MaxDirectoryDepthExceeded(sprintf("Maximum folder depth exceeded. Last directory read before stopping %s", $startDirectory->fullDirectoryPath));
			}

			$directoryHandle = opendir($startDirectory->fullDirectoryPath);
			do {
				$fileName = readdir($directoryHandle);
				if ($fileName !== false) {
					if (!str_starts_with($fileName, ".")) {
						$fullPathToThisFile = sprintf("%s/%s", $startDirectory->fullDirectoryPath, $fileName);
						if (is_dir($fullPathToThisFile)) {
							$fsDirectory = new FSDirectory(
								$fileName,
								$fullPathToThisFile
							);
							$startDirectory->childDirectories[] = $fsDirectory;
							self::gatherChildren($fsDirectory, $maximumFolderDepth, $currentFolderDepth);
						} else {
							$startDirectory->childFiles[] = new FSFile(
								$fileName,
								$fullPathToThisFile,
								filesize($fullPathToThisFile),
							);
						}
					}
				}
			} while ($fileName !== false);

			closedir($directoryHandle);
			return $startDirectory;
		}

		/**
		 * Returns the directory contents as FSFile objects or FSDirectory objects
		 */
		public static function fetchDirectoryContents(string $directoryStartPath): FSDirectory
		{
			$startPath = realpath($directoryStartPath);

			$topLevelDirectory = new FSDirectory(
				basename($startPath),
				$startPath,
			);

			$directoryHandle = opendir($topLevelDirectory->fullDirectoryPath);
			do {
				$fileName = readdir($directoryHandle);
				if ($fileName !== false) {
					if (!str_starts_with($fileName, ".")) {
						$fullPathToThisFile = sprintf("%s/%s", $topLevelDirectory->fullDirectoryPath, $fileName);
						if (is_dir($fullPathToThisFile)) {
							$fsDirectory = new FSDirectory(
								$fileName,
								realpath($fullPathToThisFile),
							);
							$topLevelDirectory->childDirectories[] = $fsDirectory;
						} else {
							$topLevelDirectory->childFiles[] = new FSFile(
								$fileName,
								realpath($fullPathToThisFile),
								filesize($fullPathToThisFile),
							);
						}
					}
				}
			} while ($fileName !== false);

			closedir($directoryHandle);

			return $topLevelDirectory;
		}

		/**
		 * Gets the next name of a new folder in a directory. This does not return
		 * the new full directory path. Just the name of the new folder/directory itself.
		 * The parentDirectory variable is not validated. You need to validate it prior
		 * to calling this method.
		 * @throws MaximumNewFolderDepthExceeded
		 */
		public static function getNextNewDirectoryName(string $parentDirectory): string
		{
			// Limit the number of possible "New Folder (X)" there can be
			// in case some absolute loony tries to make that many folders
			// in a single directory.
			$maxLimit = 120;
			$tryNewDirectory = sprintf("%s/%s", $parentDirectory, "New Folder");

			if (file_exists($tryNewDirectory)) {
				for ($i = 1; $i <= $maxLimit; $i++) {
					$tryNewDirectory = sprintf("%s/New Folder (%d)", $parentDirectory, $i);
					if (!file_exists($tryNewDirectory)) {
						return sprintf("New Folder (%d)", $i);
					}
				}

				throw new MaximumNewFolderDepthExceeded("Unable to auto-generate a new folder name. The maximum recursion limit of new folder names ({$maxLimit} iterations) was hit.");
			} else {
				return "New Folder";
			}
		}

	}
