<?php

	use Accounts\Account;
	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use FindAndReplace\Dtos\FindAndReplaceFoundDto;
	use FindAndReplace\FindAndReplaceService;
	use MonologWrapper\MonologWrapper;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Roles\PermissionCategories;

	#[Controller]
	#[RouteBase("/uplift/find-and-replace")]
	class FindAndReplaceController extends BaseController{

		#[Route("GET", "/")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::SEARCH_AND_REPLACE)]
		public function mainView(): string
		{
			return Renderer::renderView(
				viewFileName: "find-and-replace/main.php",
			);
		}

		// New API
		#[Route("GET", "/find")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::SEARCH_AND_REPLACE)]
		public function findQueryInPagesAndFiles(Request $request): JSONResult
		{
			$query = $request->getQueryValue("query");

			if (!empty($query)){
				return new JSONSuccess((array)FindAndReplaceService::getFindResultsWithSectionSupport($query));
			}else{
				return new JSONError("Query cannot be empty.");
			}
		}

		// New API
		#[Route("PATCH", "/make-replacements")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::SEARCH_AND_REPLACE)]
		public function makeAllFindResultReplacements(Request $request): JSONResult
		{
			$logger = MonologWrapper::getLogger();
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$findAndReplaceFoundDtoArray = $payload->getArrayPayload("findAndReplaceFoundDto");
				$originalQueryPayload = $payload->getTextPayload("originalQuery");
				$replacementPayload = $payload->getTextPayload("replacement");
			} catch (NoPayloadFound $e) {
				return new JSONError($e->getMessage());
			}

			$findAndReplaceFoundDtoObject = FindAndReplaceFoundDto::fromArray($findAndReplaceFoundDtoArray->contents);
			$resultDto = FindAndReplaceService::makeReplacementsWithSectionSupport($findAndReplaceFoundDtoObject, $replacementPayload->contents);

			$logger->info("$account->id {$account->getFullName()} made replacements via Find & Replace. Content was replaced with: $replacementPayload->contents");

			ActivityLog::log(
				categoryID: ActivityLogCategories::REPLACED_CONTENT_VIA_FIND_REPLACE->value,
				accountID: (Account::getCurrentUser())->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"textMatched"=>$originalQueryPayload->contents,
					"replacementText"=>$replacementPayload->contents,
				]),
			);

			return new JSONSuccess((array)$resultDto);
		}

	}