<?php

	namespace FindAndReplace;

	use Accounts\Account;
	use FileSystemUtilities\FileSystemUtilities;
	use FileSystemUtilities\FSDirectory;
	use FileSystemUtilities\FSFile;
	use FindAndReplace\Dtos\FindAndReplaceFoundDto;
	use FindAndReplace\Dtos\FoundFileDto;
	use FindAndReplace\Dtos\FoundPageDto;
	use FindAndReplace\Dtos\FoundPageSectionMatchDto;
	use FindAndReplace\Dtos\FoundPagesMatchDto;
	use FindAndReplace\Dtos\ReplacementsResultDto;
	use MultiByteString\MultiByteString;
	use Nox\ORM\ColumnQuery;
	use Page\Page;
	use Page\PageContentSection;
	use PageArchives\PageArchivesService;
	use PageEditor\PageEditorService;
	use Roles\PermissionCategories;
	use System\Themes;
	use Uplift\Exceptions\IncompatiblePageType;

	class FindAndReplaceService{

		/**
		 * Find Page objects whose pageBody or pageHead contains the provided query.
		 * @return Page[]
		 */
		public static function getPagesMatchingQuery(string $query): array{
			return Page::query(
				columnQuery: (new ColumnQuery())
					->where("pageHead","LIKE", sprintf("%%%s%%", str_replace("_", "\\_", $query)))
					->or()
					->where("pageBody","LIKE",sprintf("%%%s%%", str_replace("_", "\\_", $query)))
			);
		}

		/**
		 * Find Page objects whose pageBody or pageHead contains the provided query, or whose sections
		 * match the query if that page's layout uses sectioned content.
		 */
		public static function getPagesMatchingQueryWithSectionSupport(string $query): FoundPagesMatchDto{
			// First, get all pages that aren't deleted
			/** @var Page[] $pages */
			$pages = Page::query(
				columnQuery: (new ColumnQuery())
					->where("is_deleted", "=", 0)
			);

			$result = new FoundPagesMatchDto();
			$loweredQuery = mb_strtolower($query);

			// Iterate through each page and determine if it is sectioned or not.
			// This will determine the algorithm for searching the content.
			// Searching the pageHead is done for either choice.
			foreach($pages as $page){
				$didAnythingMatch = false;
				$foundPageDto = new FoundPageDto();
				$foundPageDto->page = $page;
				$pageHasSections = $page->getLayoutDefaultOrFirstSection() !== null;
				if ($pageHasSections){
					$foundPageDto->isSectioned = true;
					$layoutSections = $page->getAllContentSections();

					// Search the content of each layout section for $query
					foreach($layoutSections as $layoutSection){
						if (mb_strpos(mb_strtolower($layoutSection->content), $loweredQuery) !== false){
							$pageSectionMatchDto = new FoundPageSectionMatchDto();
							$pageSectionMatchDto->sectionName = $layoutSection->sectionName;
							$pageSectionMatchDto->sectionContent = $layoutSection->content;
							$foundPageDto->sectionMatches[] = $pageSectionMatchDto;
							$didAnythingMatch = true;
						}
					}
				}else{
					// Search the pageBody
					if (mb_strpos(mb_strtolower($page->pageBody), $loweredQuery) !== false) {
						$foundPageDto->pageBodyMatched = true;
						$didAnythingMatch = true;
					}
				}

				// In either case, also search the pageHead
				if (mb_strpos(mb_strtolower($page->pageHead), $loweredQuery) !== false) {
					$foundPageDto->pageHeadMatched = true;
					$didAnythingMatch = true;
				}

				// Did anything match? If not, don't add it to the FoundPagesMatchDto
				if ($didAnythingMatch){
					$result->foundPages[] = $foundPageDto;
				}
			}

			return $result;
		}

		/**
		 * Iterates through a file system directory and its descendant directories.
		 * Populates the provided array reference $arrayOfFiles with the files found.
		 * @param FSDirectory $directory
		 * @param $arrayOfFiles
		 * @return void
		 */
		private static function iterateFSDirectory(FSDirectory $directory, &$arrayOfFiles): void{
			$arrayOfFiles = array_merge($arrayOfFiles, $directory->childFiles);

			foreach($directory->childDirectories as $childFSDirectory){
				self::iterateFSDirectory($childFSDirectory, $arrayOfFiles);
			}
		}

		/**
		 * Find theme files that contain the search query
		 */
		public static function getThemeFilesMatching(string $query): array{
			/** @var FSFile[] $allFSFiles */
			$allFSFiles = [];

			/** @var FSFile[] $matchingFSFiles */
			$matchingFSFiles = [];

			$themeDirectory = Themes::getCurrentThemeDirectory();

			self::iterateFSDirectory(
				FileSystemUtilities::fetchDirectoryAndDescendants($themeDirectory),
				$allFSFiles,
			);

			// Iterate the files and check their contents
			foreach($allFSFiles as $fsFile){ // allFSFiles is not empty, PHPStorm intellisense is wrong here
				if (
					$fsFile->fileExtension === "php"
					||
					$fsFile->fileExtension === "html"
				) {
					$contents = file_get_contents($fsFile->fullFilePath);
					if (str_contains(haystack: strtolower($contents), needle: strtolower($query))) {
						$matchingFSFiles[] = $fsFile;
					}
				}
			}

			return $matchingFSFiles;
		}

		/**
		 * @param string $query
		 * @param bool $ignorePermissions
		 * @return FindResult[]
		 */
		public static function getFindResults(string $query, bool $ignorePermissions = false): array{
			$results = [];
			$pages = FindAndReplaceService::getPagesMatchingQuery($query);

			$account = Account::getCurrentUser();

			if($ignorePermissions || $account->hasPermission(PermissionCategories::MANAGE_TEMPLATE_FILES)) {
				$fsFiles = FindAndReplaceService::getThemeFilesMatching($query);
			} else {
				$fsFiles = [];
			}


			// Find matching pages and the positions of where the query was found
			foreach($pages as $page){
				$bodyMultiByte = new MultiByteString($page->pageBody);
				$headMultiByte = new MultiByteString($page->pageHead);
				$result = new FindResult();
				$result->pageID = $page->id;
				$result->pageName = $page->pageName;
				$result->type = FindResultType::PAGE;

				$pageBodyFindResults = $bodyMultiByte->findAllOccurrences($query, false);
				$pageHeadFindResults = $headMultiByte->findAllOccurrences($query, false);

				if (!empty($pageBodyFindResults)){
					foreach($pageBodyFindResults as $findResult){
						$multiByteLengthOfMatch = mb_strlen($findResult->match);
						$stubResult = $bodyMultiByte->getSubStringWithPadding($findResult);

						// Set up the sanitized result stub
						$findStub = new FindResultStub();
						$findStub->after = $stubResult->afterStub;
						$findStub->afterSanitized = htmlspecialchars($stubResult->afterStub);
						$findStub->before = $stubResult->beforeStub;
						$findStub->beforeSanitized = htmlspecialchars($stubResult->beforeStub);
						$findStub->result = $stubResult->stub;
						$findStub->resultSanitized = htmlspecialchars($stubResult->stub);

						$resultItem = new FindResultItem();
						$resultItem->start = $findResult->endCharacterPositionOfMatch - $multiByteLengthOfMatch + 1;
						$resultItem->end = $findResult->endCharacterPositionOfMatch;
						$resultItem->stub = $findStub;
						$result->findResults[] = $resultItem;
					}
				}

				if (!empty($pageHeadFindResults)){
					foreach($pageHeadFindResults as $findResult){
						$multiByteLengthOfMatch = mb_strlen($findResult->match);
						$stubResult = $headMultiByte->getSubStringWithPadding($findResult);

						// Set up the sanitized result stub
						$findStub = new FindResultStub();
						$findStub->after = $stubResult->afterStub;
						$findStub->afterSanitized = htmlspecialchars($stubResult->afterStub);
						$findStub->before = $stubResult->beforeStub;
						$findStub->beforeSanitized = htmlspecialchars($stubResult->beforeStub);
						$findStub->result = $stubResult->stub;
						$findStub->resultSanitized = htmlspecialchars($stubResult->stub);

						$resultItem = new FindResultItem();
						$resultItem->isPageHead = true;
						$resultItem->start = $findResult->endCharacterPositionOfMatch - $multiByteLengthOfMatch + 1;
						$resultItem->end = $findResult->endCharacterPositionOfMatch;
						$resultItem->stub = $findStub;
						$result->findResults[] = $resultItem;
					}
				}

				$results[] = $result;
			}

			foreach($fsFiles as $fsFile){
				$fileContents = file_get_contents($fsFile->fullFilePath);
				$fileMultibyte = new MultiByteString($fileContents);
				$result = new FindResult();
				$result->filePath = $fsFile->fullFilePath;
				$result->fileName = $fsFile->fileName;
				$result->type = FindResultType::FILE;

				$fileContentFindResults = $fileMultibyte->findAllOccurrences($query, false);

				if (!empty($fileContentFindResults)){
					foreach($fileContentFindResults as $findResult){
						$multiByteLengthOfMatch = mb_strlen($findResult->match);
						$stubResult = $fileMultibyte->getSubStringWithPadding($findResult);

						// Set up the sanitized result stub
						$findStub = new FindResultStub();
						$findStub->after = $stubResult->afterStub;
						$findStub->afterSanitized = htmlspecialchars($stubResult->afterStub);
						$findStub->before = $stubResult->beforeStub;
						$findStub->beforeSanitized = htmlspecialchars($stubResult->beforeStub);
						$findStub->result = $stubResult->stub;
						$findStub->resultSanitized = htmlspecialchars($stubResult->stub);

						$resultItem = new FindResultItem();
						$resultItem->start = $findResult->endCharacterPositionOfMatch - $multiByteLengthOfMatch + 1;
						$resultItem->end = $findResult->endCharacterPositionOfMatch;
						$resultItem->stub = $findStub;
						$result->findResults[] = $resultItem;
					}
				}
				$results[] = $result;
			}

			return $results;
		}

		public static function makeAllReplacements(FindResult $findResult, string $replacement): void{
			if ($findResult->type === FindResultType::PAGE){
				/** @var Page $page */
				$page = Page::fetch($findResult->pageID);

				foreach($findResult->findResults as $index=>$resultItem){
					$multiByteLengthOfStub = mb_strlen($resultItem->stub->result);

					if ($resultItem->isPageHead){
						$pageHeadMultibyte = new MultiByteString($page->pageHead);
						$pageHeadMultibyte->replaceStub(
							start: $resultItem->start,
							stubMultiByteLength:$multiByteLengthOfStub,
							replacement:$replacement,
						);

						$page->pageHead = $pageHeadMultibyte->currentString;
					}else{
						$pageBodyMultibyte = new MultiByteString($page->pageBody);
						$pageBodyMultibyte->replaceStub(
							start: $resultItem->start,
							stubMultiByteLength:$multiByteLengthOfStub,
							replacement:$replacement,
						);

						$page->pageBody = $pageBodyMultibyte->currentString;
					}

					// Now adjust any existing start and ends that are greater than this one
					// The difference adjustment is how different the replacement is from the original occurrence length
					$multiByteDifferenceOfReplacement = mb_strlen($replacement) - $multiByteLengthOfStub;
					foreach($findResult->findResults as $indexChild=>$resultItemChild){
						if ($indexChild !== $index){
							if ($resultItemChild->isPageHead === $resultItem->isPageHead) {
								if ($resultItemChild->start > $resultItem->start) {
									// This one needs to change
									$resultItemChild->start += $multiByteDifferenceOfReplacement;
									$resultItemChild->end += $multiByteDifferenceOfReplacement;
								}
							}
						}
					}
				}

				$page->save();
			}elseif ($findResult->type === FindResultType::FILE){
				$fileContents = file_get_contents($findResult->filePath);

				foreach($findResult->findResults as $index=>$resultItem){
					$multiByteLengthOfStub = mb_strlen($resultItem->stub->result);

					$fileContentsMultiByte = new MultiByteString($fileContents);
					$fileContentsMultiByte->replaceStub(
						start: $resultItem->start,
						stubMultiByteLength:$multiByteLengthOfStub,
						replacement:$replacement,
					);

					$fileContents = $fileContentsMultiByte->currentString;

					// Now adjust any existing start and ends that are greater than this one
					// The difference adjustment is how different the replacement is from the original occurrence length
					$multiByteDifferenceOfReplacement = mb_strlen($replacement) - $multiByteLengthOfStub;
					foreach($findResult->findResults as $indexChild=>$resultItemChild){
						if ($indexChild !== $index){
							if ($resultItemChild->start > $resultItem->start) {
								// This one needs to change
								$resultItemChild->start += $multiByteDifferenceOfReplacement;
								$resultItemChild->end += $multiByteDifferenceOfReplacement;
							}
						}
					}
				}

				file_put_contents($findResult->filePath, $fileContents);
			}
		}

		/**
		 * Finds all pages and template files (if the current account can manage template files)
		 * that match the given query. This method supports finding matching stubs in page
		 * layout sections.
		 */
		public static function getFindResultsWithSectionSupport(string $query): FindAndReplaceFoundDto{
			$findAndReplaceFoundDto = new FindAndReplaceFoundDto();
			/** @var FSFile[] $fileSystemFiles */
			$fileSystemFiles = [];
			$foundPagesMatchDto = FindAndReplaceService::getPagesMatchingQueryWithSectionSupport($query);
			$findAndReplaceFoundDto->foundPagesMatches = $foundPagesMatchDto;

			if (Account::getCurrentUser()->hasPermission(PermissionCategories::MANAGE_TEMPLATE_FILES)) {
				$fileSystemFiles = FindAndReplaceService::getThemeFilesMatching($query);
			}

			// Find matching pages and the positions of where the query was found
			foreach($foundPagesMatchDto->foundPages as $foundPageDto){
				if ($foundPageDto->isSectioned){
					foreach($foundPageDto->sectionMatches as $foundPageSectionMatch){
						$sectionMultiByte = new MultiByteString($foundPageSectionMatch->sectionContent);
						$sectionOccurrences = $sectionMultiByte->findAllOccurrences($query, false);
						if (!empty($sectionOccurrences)){
							foreach($sectionOccurrences as $sectionFindResult){
								$findStub = new FindResultStub();
								$multiByteLengthOfMatch = mb_strlen($sectionFindResult->match);
								$stubResult = $sectionMultiByte->getSubStringWithPadding($sectionFindResult);
								$findStub->after = $stubResult->afterStub;
								$findStub->afterSanitized = htmlspecialchars($stubResult->afterStub);
								$findStub->before = $stubResult->beforeStub;
								$findStub->beforeSanitized = htmlspecialchars($stubResult->beforeStub);
								$findStub->result = $stubResult->stub;
								$findStub->resultSanitized = htmlspecialchars($stubResult->stub);

								$resultItem = new FindResultItem();
								$resultItem->start = $sectionFindResult->endCharacterPositionOfMatch - $multiByteLengthOfMatch + 1;
								$resultItem->end = $sectionFindResult->endCharacterPositionOfMatch;
								$resultItem->stub = $findStub;
								$foundPageSectionMatch->findResultItems[] = $resultItem;
							}
						}
					}
				}else{
					$bodyMultiByte = new MultiByteString($foundPageDto->page->pageBody);
					$pageBodyFindResults = $bodyMultiByte->findAllOccurrences($query, false);

					if (!empty($pageBodyFindResults)){
						foreach($pageBodyFindResults as $findResult){
							$multiByteLengthOfMatch = mb_strlen($findResult->match);
							$stubResult = $bodyMultiByte->getSubStringWithPadding($findResult);

							// Set up the sanitized result stub
							$findStub = new FindResultStub();
							$findStub->after = $stubResult->afterStub;
							$findStub->afterSanitized = htmlspecialchars($stubResult->afterStub);
							$findStub->before = $stubResult->beforeStub;
							$findStub->beforeSanitized = htmlspecialchars($stubResult->beforeStub);
							$findStub->result = $stubResult->stub;
							$findStub->resultSanitized = htmlspecialchars($stubResult->stub);

							$resultItem = new FindResultItem();
							$resultItem->start = $findResult->endCharacterPositionOfMatch - $multiByteLengthOfMatch + 1;
							$resultItem->end = $findResult->endCharacterPositionOfMatch;
							$resultItem->stub = $findStub;
							$foundPageDto->pageBodyFindResultItems[] = $resultItem;
						}
					}
				}

				$headMultiByte = new MultiByteString($foundPageDto->page->pageHead);
				$pageHeadFindResults = $headMultiByte->findAllOccurrences($query, false);

				if (!empty($pageHeadFindResults)){
					foreach($pageHeadFindResults as $findResult){
						$multiByteLengthOfMatch = mb_strlen($findResult->match);
						$stubResult = $headMultiByte->getSubStringWithPadding($findResult);

						// Set up the sanitized result stub
						$findStub = new FindResultStub();
						$findStub->after = $stubResult->afterStub;
						$findStub->afterSanitized = htmlspecialchars($stubResult->afterStub);
						$findStub->before = $stubResult->beforeStub;
						$findStub->beforeSanitized = htmlspecialchars($stubResult->beforeStub);
						$findStub->result = $stubResult->stub;
						$findStub->resultSanitized = htmlspecialchars($stubResult->stub);

						$resultItem = new FindResultItem();
						$resultItem->start = $findResult->endCharacterPositionOfMatch - $multiByteLengthOfMatch + 1;
						$resultItem->end = $findResult->endCharacterPositionOfMatch;
						$resultItem->stub = $findStub;
						$foundPageDto->pageHeadFindResultItems[] = $resultItem;
					}
				}
			}

			foreach($fileSystemFiles as $fsFile){
				$fileContents = file_get_contents($fsFile->fullFilePath);
				$fileMultibyte = new MultiByteString($fileContents);
				$foundFileDto = new FoundFileDto();
				$foundFileDto->filePath = $fsFile->fullFilePath;
				$foundFileDto->fileName = $fsFile->fileName;

				$fileContentFindResults = $fileMultibyte->findAllOccurrences($query, false);

				if (!empty($fileContentFindResults)){
					foreach($fileContentFindResults as $findResult){
						$multiByteLengthOfMatch = mb_strlen($findResult->match);
						$stubResult = $fileMultibyte->getSubStringWithPadding($findResult);

						// Set up the sanitized result stub
						$findStub = new FindResultStub();
						$findStub->after = $stubResult->afterStub;
						$findStub->afterSanitized = htmlspecialchars($stubResult->afterStub);
						$findStub->before = $stubResult->beforeStub;
						$findStub->beforeSanitized = htmlspecialchars($stubResult->beforeStub);
						$findStub->result = $stubResult->stub;
						$findStub->resultSanitized = htmlspecialchars($stubResult->stub);

						$resultItem = new FindResultItem();
						$resultItem->start = $findResult->endCharacterPositionOfMatch - $multiByteLengthOfMatch + 1;
						$resultItem->end = $findResult->endCharacterPositionOfMatch;
						$resultItem->stub = $findStub;
						$foundFileDto->findResultItems[] = $resultItem;
					}
				}

				$findAndReplaceFoundDto->foundFiles[] = $foundFileDto;
			}

			// Finally, dehydrate the find response of extraneous information that takes up a lot of
			// network bandwidth, but isn't useful to the front-end's functionality
			// such as the page's full pageHead, pageBody, or the section's full content

			// We'll remove pageBody, pageHead, and sectionContent from the response to save network bandwidth
			foreach($findAndReplaceFoundDto->foundPagesMatches->foundPages as $foundPagesMatch){
				$foundPagesMatch->page->pageBody = "";
				$foundPagesMatch->page->pageHead = "";
				if (!empty($foundPagesMatch->sectionMatches)){
					foreach($foundPagesMatch->sectionMatches as $sectionMatch){
						$sectionMatch->sectionContent = "";
					}
				}
			}

			return $findAndReplaceFoundDto;
		}

		/**
		 * Makes replacements to CMS pages or template theme files and supports sectioned page content.
		 *
		 * The general algorithm for replacement is to make the multibyte-supported string replacement,
		 * then iterate over additional replacements on that same content whose start position for replacement
		 * is further out than the replacement just made. Determine the difference in content from the current
		 * replacement (e.g., if it took out 2 characters, or it added 2 characters) and add that difference to
		 * replacements that are ahead of the one we just made. If we do not do this, then the replacements
		 * ahead of the one just made will start on the wrong string index position - since the one just made
		 * probably adjusts the current content length.
		 */
		public static function makeReplacementsWithSectionSupport(
			FindAndReplaceFoundDto $findAndReplaceFoundDto,
			string $replacement
		): ReplacementsResultDto{
			$resultDto = new ReplacementsResultDto();

			// Handle all CMS page replacements
			foreach($findAndReplaceFoundDto->foundPagesMatches?->foundPages as $foundPageMatchDto){
				/** @var ?Page $page */
				$page = Page::fetch($foundPageMatchDto->page->id);

				// Skip page if the page is somehow null
				if ($page === null){
					continue;
				}

				// Increment the total pages affected in the result DTO
				$resultDto->totalNumberOfPagesAffected++;

				// Consider if this will be a sectioned content replacement
				if ($foundPageMatchDto->isSectioned){
					// Replace content found in each section

					// Keep an array of section content to be updated after all section matches are replaced
					/** @var array<string, string> $sectionContentsToBeUpdated The key is the section name. */
					$sectionContentsToBeUpdated = [];
					foreach ($foundPageMatchDto->sectionMatches as $sectionMatchDto){

						// If this section name is not yet stored in $sectionContentsToBeUpdated, then add it
						// as a new key
						if (!array_key_exists($sectionMatchDto->sectionName, $sectionContentsToBeUpdated)){
							// Get the page's section to get the current section content
							$contentSection = $page->getContentSectionByName($sectionMatchDto->sectionName);

							if ($contentSection === null){
								throw new \RuntimeException("No content section named $sectionMatchDto->sectionName found. Cannot continue replacement.");
							}

							$sectionContentsToBeUpdated[$sectionMatchDto->sectionName] = $contentSection->content;
						}

						foreach($sectionMatchDto->findResultItems as $currentResultIndex=>$findResultItem){
							$multiByteLengthOfStub = mb_strlen($findResultItem->stub->result);
							$sectionContentMultibyte = new MultiByteString($sectionContentsToBeUpdated[$sectionMatchDto->sectionName]);
							$sectionContentMultibyte->replaceStub(
								start: $findResultItem->start,
								stubMultiByteLength: $multiByteLengthOfStub,
								replacement: $replacement,
							);

							$sectionContentsToBeUpdated[$sectionMatchDto->sectionName] = $sectionContentMultibyte->currentString;

							// Increment total pageBody replacements
							$resultDto->numberOfPageBodyReplacements++;

							// Now that this section's content has been replaced - we must now go through all the
							// section replacements with the same section name, but where
							// the "start" and "end" of the replacement occurs after this one.
							// Then, we will adjust the start of that replacement by how much content was added
							// or removed by the current replacement.
							$multiByteContentChangeDifference = mb_strlen($replacement) - $multiByteLengthOfStub;
							foreach($sectionMatchDto->findResultItems as $iteratedResultIndex=>$otherFindResultItem){
								// Do not iterate over the one we just replaced
								if ($iteratedResultIndex !== $currentResultIndex){
									if ($otherFindResultItem->start > $findResultItem->start){
										$otherFindResultItem->start += $multiByteContentChangeDifference;
										$otherFindResultItem->end += $multiByteContentChangeDifference;
									}
								}
							}
						}
					}

					// All sections have been updated. Now, replace the section content
					foreach($sectionContentsToBeUpdated as $sectionName=>$sectionContent){
						/** @var ?PageContentSection $contentSectionObject */
						$contentSectionObject = PageContentSection::queryOne(
							columnQuery: (new ColumnQuery())
								->where("page_id", "=", $page->id)
								->and()
								->where("section_name", "=", $sectionName)
						);

						if ($contentSectionObject === null){
							throw new \RuntimeException("No content section with name $sectionName found for page $page->id. Cannot continue replacement.");
						}

						$contentSectionObject->content = $sectionContent;
						$contentSectionObject->save();
					}
				}else{
					// This is a standard pageBody content replacement
					foreach($foundPageMatchDto->pageBodyFindResultItems as $currentResultIndex=>$pageBodyResultItem){
						$multiByteLengthOfStub = mb_strlen($pageBodyResultItem->stub->result);
						$pageContentMultiByte = new MultiByteString($page->pageBody);
						$pageContentMultiByte->replaceStub(
							start: $pageBodyResultItem->start,
							stubMultiByteLength: $multiByteLengthOfStub,
							replacement: $replacement,
						);

						$page->pageBody = $pageContentMultiByte->currentString;

						// Increment total page body replacements
						$resultDto->numberOfPageBodyReplacements++;

						// Similar to the above section replacement algorithm,
						// we find all other pageBody result items that are starting ahead of this one
						// and adjust the "start" and "end" by the content replacement length difference
						$multiByteContentChangeDifference = mb_strlen($replacement) - $multiByteLengthOfStub;
						foreach($foundPageMatchDto->pageBodyFindResultItems as $iteratedResultIndex=>$otherFindResultItem){
							// Do not iterate over the one we just replaced
							if ($iteratedResultIndex !== $currentResultIndex){
								if ($otherFindResultItem->start > $pageBodyResultItem->start){
									$otherFindResultItem->start += $multiByteContentChangeDifference;
									$otherFindResultItem->end += $multiByteContentChangeDifference;
								}
							}
						}
					}

					// Save the pageBody replacements
					$page->save();
				}

				// In either case, process any pageHead replacements
				foreach($foundPageMatchDto->pageHeadFindResultItems as $currentResultIndex=>$pageHeadResultItem){
					$multiByteLengthOfStub = mb_strlen($pageHeadResultItem->stub->result);
					$pageHeadMultiByte = new MultiByteString($page->pageHead);
					$pageHeadMultiByte->replaceStub(
						start: $pageHeadResultItem->start,
						stubMultiByteLength: $multiByteLengthOfStub,
						replacement: $replacement,
					);

					$page->pageHead = $pageHeadMultiByte->currentString;

					// Increment total page head replacements
					$resultDto->numberOfPageHeadReplacements++;

					// Similar to the above section replacement algorithm,
					// we find all other pageHead result items that are starting ahead of this one
					// and adjust the "start" and "end" by the content replacement length difference
					$multiByteContentChangeDifference = mb_strlen($replacement) - $multiByteLengthOfStub;
					foreach($foundPageMatchDto->pageHeadFindResultItems as $iteratedResultIndex=>$otherFindResultItem){
						// Do not iterate over the one we just replaced
						if ($iteratedResultIndex !== $currentResultIndex){
							if ($otherFindResultItem->start > $pageHeadResultItem->start){
								$otherFindResultItem->start += $multiByteContentChangeDifference;
								$otherFindResultItem->end += $multiByteContentChangeDifference;
							}
						}
					}
				}

				// Save the pageHead replacements
				$page->save();

				// We only archive the page here and not when the page body is saved because there
				// is no point in archiving it twice.
				try{
					PageArchivesService::archivePage($page);
				}catch(IncompatiblePageType){
					// Ignore
				}
			}

			// Now, handle any template file replacements
			foreach($findAndReplaceFoundDto->foundFiles as $foundFileDto){
				// Get file contents
				$fileContents = file_get_contents($foundFileDto->filePath);

				// Increment total number of file replacements
				$resultDto->totalNumberOfFilesAffected++;

				foreach($foundFileDto->findResultItems as $index=>$findResultItem){
					$fileContentsMultiByteLength = mb_strlen($findResultItem->stub->result);

					$fileContentsMultiByte = new MultiByteString($fileContents);
					$fileContentsMultiByte->replaceStub(
						start: $findResultItem->start,
						stubMultiByteLength: $fileContentsMultiByteLength,
						replacement: $replacement,
					);

					// Update the $fileContents variable with the new contents after the replacement
					$fileContents = $fileContentsMultiByte->currentString;

					// Increment number of file content replacements
					$resultDto->numberOfFileReplacements++;

					// Now adjust any existing start and ends that are greater than this one
					// The difference adjustment is how different the replacement is from the original occurrence length
					$multiByteDifferenceOfReplacement = mb_strlen($replacement) - $fileContentsMultiByteLength;
					foreach($foundFileDto->findResultItems as $indexChild=>$resultItemChild){
						if ($indexChild !== $index){
							if ($resultItemChild->start > $findResultItem->start) {
								// This one needs to change
								$resultItemChild->start += $multiByteDifferenceOfReplacement;
								$resultItemChild->end += $multiByteDifferenceOfReplacement;
							}
						}
					}
				}

				// Save the updated file contents
				file_put_contents($foundFileDto->filePath, $fileContents);
			}

			return $resultDto;
		}
	}