<?php
	namespace Google;

	use DateInterval;
	use DateTime;
	use GuzzleHttp\Client;
	use GuzzleHttp\RequestOptions;
	use MonologWrapper\MonologWrapper;
	use Settings\Setting;
	use Settings\Settings;
	use System\System;

	class GoogleService{

		const SEARCH_CONSOLE_DIMENSIONS = [
			"DATE",
			"QUERY",
			"PAGE",
			"COUNTRY",
		];

		public static function getSearchConsoleAnalyticsInDateRange(
			string $site,
			DateTime $startDate,
			DateTime $endDate,
		): array{
			$activeSearchConsoleEmail = Setting::getSettingValue(Settings::ACTIVE_SEARCH_CONSOLE_GOOGLE_ACCOUNT_EMAIL->value);
			$uuid = Setting::getSettingValue(Settings::BUILD_UUID->value);
			$apiKey = Setting::getSettingValue(Settings::UPLIFT_CONTROL_PANEL_API_KEY->value);
			$controlPanelHost = System::getUpliftControlPanelHost();
			$client = new Client();
			$response = $client->request(
				method:"GET",
				uri:$controlPanelHost . "/uplift/build/search-console/site-analytics",
				options:[
					RequestOptions::HEADERS => [
						"Uplift-API-Key"=>$apiKey,
						"Uplift-UUID"=>$uuid,
					],
					RequestOptions::QUERY => [
						"email"=>$activeSearchConsoleEmail,
						"site"=>$site,
						"startDate"=>$startDate->format("Y-m-d"),
						"endDate"=>$endDate->format("Y-m-d"),
						"base64DimensionsJSON"=>base64_encode(json_encode(self::SEARCH_CONSOLE_DIMENSIONS)),
					],
				],
			);

			$bodyContents = $response->getBody()->getContents();

			/** @var array{status: int, rows: array} $data */
			$data = json_decode($bodyContents, true);

			return $data['rows'];
		}

		/**
		 * Using the currently active search console site, will pull the search console data for the last week.
		 * Then, it will store it in the database.
		 * @throws \GuzzleHttp\Exception\GuzzleException
		 * @throws \Uplift\Exceptions\MalformedValue
		 */
		public static function pullSearchConsoleDataForWeekForActiveSearchConsoleSite(): void{
			$logger = MonologWrapper::getLogger();
			$logger->info("Pulling Search Console data for this week.");
			$activeSearchConsoleSite = Setting::getSettingValue(Settings::ACTIVE_SEARCH_CONSOLE_SITE->value);

			if (empty($activeSearchConsoleSite)){
				$logger->warning("Now pulling Search Console data. No active console site selected.");
			}else{
				$logger->info("Pulling Search Console data for $activeSearchConsoleSite");
			}

			$oneDayAgo = new DateInterval("P1D");
			$oneDayAndOneWeekAgo = new DateInterval("P1W1D");
			$startDate = new DateTime();
			$startDate->sub($oneDayAndOneWeekAgo);
			$endDate = new DateTime();
			$endDate->sub($oneDayAgo);

			$rows = self::getSearchConsoleAnalyticsInDateRange(
				site: $activeSearchConsoleSite,
				startDate:$startDate,
				endDate:$endDate,
			);

			$logger->info(sprintf("Got %d Search Console rows to process today.", count($rows)));

			/** @var array{clicks: int, ctr:int|float, position:int|float, impressions:int, keys:array}  $row */
			foreach($rows as $row){
				$keys = $row['keys'];
				$date = $keys[0];
				$query = $keys[1];
				$page = $keys[2];
				$country = $keys[3];
				// Try and store them
				// Create the custom key to index them by hashing the date, query, page, and country
				// This helps avoid duplicates in the database
				$keyUnhashed = sprintf("%s-%s-%s-%s", $date, $query, $page, $country);
				$key = hash("sha512", $keyUnhashed);
				$dateAsDateTime = DateTime::createFromFormat("Y-m-d", $date);
				$dataRow = new GoogleAccountSearchConsoleDataRow();
				$dataRow->key = $key;
				$dataRow->query = $query;
				$dataRow->country = $country;
				$dataRow->date = $date;
				$dataRow->page = $page;
				$dataRow->position = $row['position'];
				$dataRow->clicks = $row['clicks'];
				$dataRow->impressions = $row['impressions'];
				$dataRow->ctr = $row['ctr'];
				$dataRow->dateAsUnixTimestamp = $dateAsDateTime->getTimestamp();
				$dataRow->save();
			}

			$logger->info("Finished Search Console pull and processing.");
		}
	}