<?php

	namespace ImageGallery;

	use Nox\ORM\ColumnQuery;
	use Nox\ORM\ResultOrder;
	use Uplift\ImageManager\ImageMissingThumb;

	class ImageGalleryService{

		/**
		 * @return ImageGallery[]
		 */
		public static function getAllGalleries(): array{
			/** @var ImageGallery[] $allGalleries */
			$allGalleries = ImageGallery::query(
				resultOrder: (new ResultOrder())
					->by("name","ASC"),
			);

			// Populate the number of images in each gallery
			foreach($allGalleries as $gallery){
				$gallery->fetchNumImagesInGallery();
			}

			return $allGalleries;
		}

		/**
		 * @param string $galleryName
		 * @return ImageGallery
		 * @throws ImageGalleryNameEmpty
		 * @throws ImageGalleryWithSameNameExists
		 */
		public static function createNewGallery(string $galleryName): ImageGallery{

			$galleryName = trim($galleryName);

			if (empty($galleryName)){
				throw new ImageGalleryNameEmpty("Gallery names cannot be empty.");
			}

			// Find an existing one with the same name
			/** @var ImageGallery $existingGallery */
			$existingGallery = ImageGallery::queryOne(
				columnQuery: (new \Nox\ORM\ColumnQuery())
					->where("name","=",$galleryName),
			);

			if ($existingGallery !== null){
				throw new ImageGalleryWithSameNameExists("A gallery with the provided name already exists.");
			}

			$gallery = new ImageGallery();
			$gallery->name = $galleryName;
			$gallery->save();

			return $gallery;
		}

		/**
		 * @param int $galleryID
		 * @return ImageGalleryMember[]
		 */
		public static function getGalleryMembers(int $galleryID): array{
			return ImageGalleryMember::query(
				columnQuery: (new ColumnQuery())
					->where("gallery_id","=",$galleryID),
			);
		}

		/**
		 * @param int $galleryID
		 * @param array[] $images Array of arrays. Inner arrays have keys "position", "imageURI", "thumbURI", and "altText"
		 * @return ImageGalleryMember[]
		 * @throws ImageMissingThumb
		 */
		public static function addMembersToGallery(int $galleryID, array $images): array{
			$members = [];
			/** @var array{position: int, imageURI: string, thumbURI: string, altText: string} $image */
			foreach($images as $image){
				try{

					if ($image['thumbURI'] === null){
						throw new ImageMissingThumb(sprintf("%s is missing a thumb. Please regenerate a thumb for this image.", $image['imageURI']));
					}

					$members[] = self::addMemberToGallery(
						galleryID:$galleryID,
						imageURI:$image['imageURI'],
						thumbURI:$image['thumbURI'],
						altText:$image['altText'],
						position:$image['position'],
					);
					// Ignore exception
				}catch(ImageGalleryAlreadyContainsImage){}
			}

			return $members;
		}

		/**
		 * @param int $galleryID
		 * @param array[] $members Array of arrays. Inner arrays have keys "id", and "position"
		 * @return void
		 */
		public static function updateMembersPositionInGallery(int $galleryID, array $members): void{
			$galleryMembersToSave = [];
			/** @var array{id: int, position: int} $member */
			foreach($members as $member){
				/** @var ImageGalleryMember $galleryMember */
				$galleryMember = ImageGalleryMember::fetch($member['id']);
				if ($galleryMember !== null){
					$galleryMember->position = $member['position'];
					$galleryMembersToSave[] = $galleryMember;
				}
			}

			ImageGalleryMember::saveAll($galleryMembersToSave);
		}

		/**
		 * @param ImageGallery $gallery
		 * @param string $newName
		 * @return void
		 * @throws ImageGalleryWithSameNameExists
		 */
		public static function updateGalleryName(ImageGallery $gallery, string $newName): void{

			// See if another gallery with the same name already exists
			$existingGalleryWithSameName = ImageGallery::queryOne(
				columnQuery: (new ColumnQuery())
					->where("name","=",$newName)
					->and()
					->where("id","!=",$gallery->id),
			);

			if ($existingGalleryWithSameName !== null){
				throw new ImageGalleryWithSameNameExists();
			}

			$gallery->name = $newName;
			$gallery->save();
		}

		/**
		 * @param ImageGalleryMember $member
		 * @param string $altText
		 * @return void
		 */
		public static function updateMemberAltText(ImageGalleryMember $member, string $altText): void{
			$member->altAttribute = $altText;
			$member->save();
		}

		/**
		 * @param ImageGallery $gallery
		 * @return void
		 * @throws \Nox\ORM\Exceptions\NoPrimaryKey
		 */
		public static function deleteGallery(ImageGallery $gallery): void{

			// Remove all members that have that gallery ID
			/** @var ImageGalleryMember[] $imageGalleryMembers */
			$imageGalleryMembers = ImageGalleryMember::query(
				columnQuery: (new ColumnQuery())
					->where("gallery_id","=",$gallery->id)
			);

			foreach($imageGalleryMembers as $imageGalleryMember){
				$imageGalleryMember->delete();
			}

			$gallery->delete();
		}

		public static function removeGalleryMember(
			int $galleryMemberID,
		): void
		{
			/** @var ?ImageGalleryMember $member */
			$member = ImageGalleryMember::fetch($galleryMemberID);

			if ($member){
				$member->delete();
			}
		}

		/**
		 * @param int $galleryID
		 * @param string $imageURI
		 * @param string $thumbURI
		 * @param string $altText
		 * @param int $position
		 * @return ImageGalleryMember
		 * @throws ImageGalleryAlreadyContainsImage
		 */
		private static function addMemberToGallery(
			int $galleryID,
			string $imageURI,
			string $thumbURI,
			string $altText,
			int $position
		): ImageGalleryMember
		{
			// Check if an image already exists with this imageURI in this gallery
			$existingImage = ImageGalleryMember::queryOne(
				columnQuery: (new ColumnQuery())
					->where("gallery_id","=",$galleryID)
					->and()
					->where("image_uri","=",$imageURI),
			);

			if ($existingImage !== null){
				throw new ImageGalleryAlreadyContainsImage();
			}

			$member = new ImageGalleryMember();
			$member->galleryID = $galleryID;
			$member->altAttribute = $altText;
			$member->imageURI = $imageURI;
			$member->thumbURI = $thumbURI;
			$member->position = $position;
			$member->save();
			return $member;
		}
	}