<?php
	namespace Uplift\ImageManager;

	use FileSystemUtilities\FSImageFile;
	use Settings\Setting;
	use Settings\Settings;
	use Uplift\ImageProcessing\Exceptions\ImageProcessingException;
	use Uplift\ImageProcessing\ImageProcessing;

	require_once __DIR__ . "/Exceptions/ImageFileAlreadyExists.php";
	require_once __DIR__ . "/Exceptions/ImageThumbAlreadyExists.php";
	require_once __DIR__ . "/Exceptions/ImageIsAThumb.php";

	class ImageManager{
		const PUBLIC_IMAGES_URI_BASE = "/uplift-data/images";
		const IMAGES_DIRECTORY = __DIR__ . "/../../../uplift-data/images";

		// These can/will be overridden by
		// individual CMS properties in the system settings database
		const DEFAULT_THUMB_WIDTH = 225;
		const DEFAULT_THUMB_HEIGHT = 225;

		public static function getImageURIFromFilePath(string $fullFilePath): string{
			$fullFilePath = realpath($fullFilePath);
			$imagesDirectory = realpath(self::IMAGES_DIRECTORY);

			$uri = str_replace(
				search: $imagesDirectory,
				replace: self::PUBLIC_IMAGES_URI_BASE,
				subject: $fullFilePath,
			);

			// For Windows OSes, fix backslashes in file directory
			return str_replace(
				search: "\\",
				replace: "/",
				subject:$uri,
			);
		}

		/**
		 * Determines if a directory is a valid location to store user images in the system
		 * @param string $directoryPath
		 * @return bool
		 */
		public static function isValidImageDirectory(string $directoryPath): bool{
			// Resolve symlinks and directory changes (../.)
			$resolvedPath = realpath($directoryPath);

			// Resolve the system's image directory
			$resolvedImagesDirectory = realpath(self::IMAGES_DIRECTORY);

			if ($resolvedPath === false){
				return false;
			}

			// Check if the resolved directory path begins with the resolved system image's directory
			if (!str_starts_with($resolvedPath, $resolvedImagesDirectory)){
				return false;
			}

			// Seems good enough, les go
			return true;
		}

		/**
		 * Determines if an image path is the path to a thumb image.
		 *
		 * This process is performed by moving up one directory, checking if the upper directory is still
		 * a valid image path, then appending the basename of the image path (the image's name) to the parent directory.
		 *
		 * If an image exists in the parent directory with the same exact name, and the passed image file path is in
		 * a path named "thumbs" then the provided image path is a thumb.
		 *
		 * @param string $imageFilePath
		 * @return bool
		 */
		public static function isImagePathAThumbImage(string $imageFilePath): bool{
			// Resolve symlinks and directory changes (../.)
			$resolvedPath = realpath($imageFilePath);

			if (basename(dirname($resolvedPath)) === "thumbs"){
				// Continue, the provided image is in a /thumbs directory
				$parentDirectory = dirname(dirname($resolvedPath));
				$imageFileName = basename($resolvedPath);
				$possibleImage = sprintf("%s/%s", $parentDirectory, $imageFileName);

				// If the image exists with the same name in the parent directory of this one,
				// then it's probable this image path is a thumb file of an existing image
				if (file_exists($possibleImage)){
					return true;
				}
			}

			// No, probably not a thumb file
			return false;
		}

		/**
		 * Renames a valid image file path and its thumb
		 * @param string $imageFilePath
		 * @param string $newFileNameWithoutExtension
		 * @return string
		 * @throws ImageThumbAlreadyExists
		 * @throws \GDHelper\exceptions\AnimatedWebPNotSupported
		 * @throws \GDHelper\exceptions\FileNotFound
		 * @throws \GDHelper\exceptions\InvalidImage
		 * @throws ImageFileAlreadyExists
		 */
		public static function renameImageFile(
			string $imageFilePath,
			string $newFileNameWithoutExtension
		): string
		{
			$resolvedImageFilePath = realpath($imageFilePath);

			// Create an FSImageFile object to detect existence of the thumb
			try {
				$fsImage = new \FileSystemUtilities\FSImageFile(
					fileName: basename($resolvedImageFilePath),
					fullFilePath: $resolvedImageFilePath,
					uri: null,
					acceptSVG: true,
				);
			}catch(\GDHelper\exceptions\FileNotFound $e){
				// Rethrow
				throw $e;
			}

			$newFilePath = sprintf("%s/%s.%s", dirname($resolvedImageFilePath), $newFileNameWithoutExtension, $fsImage->fileExtension);

			// Perform the thumb rename first
			if ($fsImage->thumbFilePath !== null){
				$newThumbPath = sprintf("%s/%s.%s", dirname($fsImage->thumbFilePath), $newFileNameWithoutExtension, $fsImage->fileExtension);
				if (file_exists($newFilePath)){
					throw new ImageThumbAlreadyExists;
				}

				@rename($fsImage->thumbFilePath, $newThumbPath);
			}

			// Rename the primary image
			@rename($imageFilePath, $newFilePath);

			return $newFilePath;
		}

		/**
		 * Strategically generates a thumb file for an FSImage.
		 * Will overwrite an existing one.
		 * @param FSImageFile $fsImage
		 * @throws ImageIsAThumb|ImageProcessingException
		 */
		public static function generateThumbForFSImage(FSImageFile $fsImage): void
		{
			// Possibly generate a new thumb image
			if (self::isImagePathAThumbImage($fsImage->fullFilePath)){
				throw new ImageIsAThumb("The provided image path is already a thumb image. No thumb image can be generated from a thumb.");
			}

			$setThumbWidth = ImageManager::DEFAULT_THUMB_WIDTH;
			$setThumbHeight = ImageManager::DEFAULT_THUMB_HEIGHT;

			// Read system thumb dimensions and use them if they're non-null and above 0 in value
			$systemSettingThumbWidth = Setting::getSettingValueIdentifiedByEnum(Settings::THUMBNAIL_UPLOAD_SIZE_X);
			$systemSettingThumbHeight = Setting::getSettingValueIdentifiedByEnum(Settings::THUMBNAIL_UPLOAD_SIZE_Y);
			if ($systemSettingThumbWidth !== null){
				if ((int)$systemSettingThumbWidth > 0){
					$setThumbWidth = (int)$systemSettingThumbWidth;
				}
			}

			if ($systemSettingThumbHeight !== null){
				if ((int)$systemSettingThumbHeight > 0){
					$setThumbHeight = (int)$systemSettingThumbHeight;
				}
			}

			// Fetch the thumb file path
			if ($fsImage->thumbFilePath !== null){
				$thumbFilePath = $fsImage->thumbFilePath;
			}else{
				// FSImage object doesn't have one. We need to set it, then check if it exists
				$thumbFilePath = sprintf(
					"%s/thumbs/%s",
					dirname($fsImage->fullFilePath),
					$fsImage->fileName,
				);

				$thumbsDirectory = dirname($thumbFilePath);

				if (!file_exists($thumbsDirectory)){
					// The directory doesn't exist
					// Make the thumbs directory
					@mkdir($thumbsDirectory, 0777, true);
				}

				$fsImage->thumbFilePath = $thumbFilePath;
			}

			// In order to retain as much of the original image's visible "scope" (what you can see)
			// in the thumbnail, we must algorithmically resize and crop (or one or the other)
			// the image in such a way that much of the original view is retained.

			// We need the image's dimensions from the binary to make some conditions.

			$imageProcessing = new ImageProcessing();
			$originalImageBinary = file_get_contents($fsImage->fullFilePath);
			$imageInfo = $imageProcessing->getImageInfo($fsImage->fileName, $originalImageBinary);

			// Now, with the image's information (dimensions) we can approach the problem

			if ($imageInfo->width > $setThumbWidth && $imageInfo->height > $setThumbHeight){
				// The image's width and height are both larger than the desired thumbnail dimensions.
				// If the thumb's desired dimensions were inside (imagine a rectangle inside a rectangle) the
				// main image which would hit the edge first
				// (if the inner rectangle was growing) - the width or the height?
				$stepsForInnerRectangleToHitMaxWidth = $imageInfo->width / $setThumbWidth;
				$stepsForInnerRectangleToHitMaxHeight = $imageInfo->height / $setThumbHeight;

				if ($stepsForInnerRectangleToHitMaxWidth < $stepsForInnerRectangleToHitMaxHeight){
					// A growing rectangle that has an aspect ratio for the thumb inside the main image
					// Would hit the width restriction first. In this case, find out what the height
					// Of the growing rectangle would be - as the width will just be the original
					// image width
					$grownRectangleHeight = ($setThumbHeight / $setThumbWidth) * $imageInfo->width;

					// Now, we crop the original image to the OriginalWidth x $grownRectangleHeight
					// from the center
					$croppedImage = $imageProcessing->cropFromCenter(
						$fsImage->fileName,
						$originalImageBinary,
						$imageInfo->width,
						(int)floor($grownRectangleHeight),
					);
				}else{
					// A growing rectangle that has an aspect ratio for the thumb inside the main image
					// Would hit the height restriction first. In this case, find out what the width
					// of the growing rectangle would be - as the height will just be the original
					// image height
					$grownRectangleWidth = ($setThumbWidth / $setThumbHeight) * $imageInfo->height;

					// Now, we crop the original image to the $grownRectangleWidth x OriginalHeight
					// from the center
					$croppedImage = $imageProcessing->cropFromCenter(
						$fsImage->fileName,
						$originalImageBinary,
						(int)floor($grownRectangleWidth),
						$imageInfo->height,
					);
				}

				// We can now safely resize the entire image down to $setThumbWidth x $setThumbHeight
				// with no distortions, because the $croppedImage will already be in the correct
				// aspect ratio
				$resizedImage = $imageProcessing->resize(
					$fsImage->fileName,
					$croppedImage,
					$setThumbWidth,
					$setThumbHeight,
				);

				// Save to the thumbnail path
				file_put_contents($fsImage->thumbFilePath, $resizedImage);
			} elseif ($imageInfo->width > $setThumbWidth && $imageInfo->height < $setThumbHeight){
				// The width of the image is longer than the thumb's desired with
				// But the height is smaller than the desired height of the thumbnail.

				// Use the image's height as the dimensions of the thumbnail.
				// We must now use an aspect ratio calculation to calculate the width of a resulting thumbnail
				$resultingWidth = ($setThumbWidth / $setThumbHeight) * $imageInfo->height;

				// Crop the image, from the center, to be $resultingWidth x OriginalHeight
				$croppedImage = $imageProcessing->cropFromCenter(
					$fsImage->fileName,
					$originalImageBinary,
					(int)floor($resultingWidth),
					$imageInfo->height,
				);

				// Save to the thumbnail path
				file_put_contents($fsImage->thumbFilePath, $croppedImage);
			} elseif ($imageInfo->width < $setThumbWidth && $imageInfo->height > $setThumbHeight){
				// The height of the image is longer than the thumb's desired height
				// But the width is smaller than the desired width of the thumbnail.

				// Use the image's width as the dimensions of the thumbnail.
				// We must now use an aspect ratio calculation to calculate the height of a resulting thumbnail
				$resultingHeight = ($setThumbHeight / $setThumbWidth) * $imageInfo->width;

				// Crop the image, from the center, to be OriginalWidth x $resultingHeight
				$croppedImage = $imageProcessing->cropFromCenter(
					$fsImage->fileName,
					$originalImageBinary,
					$imageInfo->width,
					(int)floor($resultingHeight),
				);

				// Save to the thumbnail path
				file_put_contents($fsImage->thumbFilePath, $croppedImage);
			}else{
				// Both of the image's dimensions are smaller than the desired thumbnail dimensions
				// In this strange case, use the original image as its own thumbnail
				file_put_contents($fsImage->thumbFilePath, $originalImageBinary);
			}

			// Set the FSImage object's thumb URI. The caller will have this populated
			// due to how object references work.
			$fsImage->thumbURI = ImageManager::getImageURIFromFilePath($thumbFilePath);
		}
	}