<?php

	namespace ImageManager;

	use Accounts\Account;
	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use FileSystemUtilities\Exceptions\InvalidFileName;
	use FileSystemUtilities\exceptions\InvalidFolderName;
	use FileSystemUtilities\exceptions\MaximumNewFolderDepthExceeded;
	use FileSystemUtilities\exceptions\NewDirectoryWithSameNameExists;
	use FileSystemUtilities\exceptions\PathDoesntExist;
	use FileSystemUtilities\FileSystemUtilities;
	use GDHelper\exceptions\AnimatedWebPNotSupported;
	use GDHelper\exceptions\FileNotFound;
	use GDHelper\exceptions\ImageTypesAreTheSame;
	use GDHelper\exceptions\InvalidImage;
	use GDHelper\exceptions\UnrecognizedImageExtension;
	use GuzzleHttp\Exception\GuzzleException;
	use ImageManager\Exception\InvalidDirectory;
	use ImageManager\Exception\InvalidFilePath;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\Request;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Roles\PermissionCategories;
	use System\HttpHelper;
	use Uplift\ImageManager\Exceptions\InvalidImageDimension;
	use Uplift\ImageManager\ImageFileAlreadyExists;
	use Uplift\ImageManager\ImageIsAThumb;
	use Uplift\ImageManager\ImageManager;
	use Uplift\ImageManager\ImageThumbAlreadyExists;
	use Uplift\ImageProcessing\Exceptions\ImageProcessingException;

	#[Controller]
	#[RouteBase("/uplift/image-manager")]
	class ImageManagerController extends BaseController{

		#[Route("GET", "/fetch-directories-recursively")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function fetchAllImageDirectories(): JSONResult
		{
			return new JSONSuccess([
				"imagesDirectory"=>FileSystemUtilities::fetchDirectoryAndDescendants(ImageManager::IMAGES_DIRECTORY),
			]);
		}

		#[Route("GET", "/fetch-directory-images")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function fetchImagesInDirectory(Request $request): JSONResult
		{
			$directory = $request->getQueryValue("directory");

			try {
				$imageFiles = ImageManagerService::fetchDirectoryImages(
					directory: $directory,
				);
			} catch (AnimatedWebPNotSupported|FileNotFound|Exception\InvalidDirectory $e) {
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess([
				"imageFiles"=>$imageFiles,
			]);
		}

		#[Route("GET", "/get-image-dimensions")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function getImageFileDimensions(Request $request): JSONResult
		{
			$imagePath = $request->getQueryValue("image-path");

			$imageDimensions = ImageManagerService::getImageDimensions(
				imageFilePath: $imagePath,
			);

			return new JSONSuccess([
				"width"=>$imageDimensions['width'],
				"height"=>$imageDimensions['height'],
			]);
		}

		#[Route("PATCH", "/move-image")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function moveImage(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$imagePath = $payload->getTextPayload("imagePath");
				$newDirectoryPath = $payload->getTextPayload("newDirectoryPath");
				$allowOverwrites = $payload->getTextPayload("allowOverwrite");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				ImageManagerService::moveImage(
					imagePath: $imagePath->contents,
					newDirectoryPath: $newDirectoryPath->contents,
					allowOverwrites: (int)$allowOverwrites->contents,
				);
			} catch (InvalidImage|AnimatedWebPNotSupported|FileNotFound|Exception\InvalidDirectory|Exception\InvalidFilePath|Exception\MissingParameter $e) {
				return new JSONError($e->getMessage());
			} catch (ImageFileAlreadyExists $e) {
				return new JSONError(
					$e->getMessage(),
					[
						"nameCollision"=>true,
					],
				);
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::MOVE_IMAGE->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"oldPath"=>$imagePath,
					"newPath"=>$newDirectoryPath,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("PATCH", "/rename-image")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function renameImageFile(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$originalFilePath = $payload->getTextPayload('file-path');
				$newNameWithoutExtension = $payload->getTextPayload('new-file-name-without-extension');
			} catch (NoPayloadFound $e) {
				return new JSONError($e->getMessage());
			}

			try {
				$newImageFilePath = ImageManagerService::renameImage(
					filePath: $originalFilePath->contents,
					newFileNameWithoutExtension: $newNameWithoutExtension->contents,
				);
			} catch (InvalidFileName|InvalidImage|ImageIsAThumb|InvalidFilePath|InvalidDirectory $e) {
				return new JSONError($e->getMessage());
			} catch (AnimatedWebPNotSupported $e) {
				return new JSONError("Animated WebP images are not supported.");
			} catch (FileNotFound $e) {
				return new JSONError("No image file found at the provided path.");
			} catch (ImageFileAlreadyExists $e) {
				return new JSONError("An image already exists with that same name.");
			} catch (ImageThumbAlreadyExists $e) {
				return new JSONError("An image thumb already exists with that same name.");
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::MOVE_IMAGE->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"oldPath"=>$originalFilePath,
					"newPath"=>$newImageFilePath,
				]),
			);

			return new JSONSuccess([
				"newImageFilePath"=>$newImageFilePath,
				"newFileNameWithoutExtension"=>$newNameWithoutExtension->contents,
				"newImageURI"=>ImageManagerService::getImageURIFromFilePath($newImageFilePath),
				"newImageThumbURI"=>ImageManagerService::getImageURIFromFilePath(ImageManagerService::getThumbPathFromFilePath($newImageFilePath)),
			]);
		}

		#[Route("PATCH", "/resize-image")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function resizeImageFile(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			// Don't check the proportions of the image if
			// the constrained flag is enabled. This isn't vital to be sanitized
			// and just assume the user didn't mess up
			try {
				$newWidth = $payload->getTextPayload("width");
				$newHeight = $payload->getTextPayload("height");
				$imageFilePath = $payload->getTextPayload("image-file-path");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				ImageManagerService::resizeImage(
					filePath: $imageFilePath->contents,
					newWidth: (int) floor($newWidth->contents),
					newHeight: (int) floor($newHeight->contents),
				);
			} catch (AnimatedWebPNotSupported|InvalidImage|FileNotFound|InvalidDirectory|InvalidFilePath|InvalidImageDimension|ImageIsAThumb|ImageProcessingException $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::RESIZE_IMAGE->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"filePath"=>$imageFilePath,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("patch", "/crop-image")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function cropImageFile(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try{
				$topX = $payload->getTextPayload('top-x');
				$bottomX = $payload->getTextPayload('bottom-x');
				$topY = $payload->getTextPayload('top-y');
				$bottomY = $payload->getTextPayload('bottom-y');
				$imageFilePath = $payload->getTextPayload('image-file-path');
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				ImageManagerService::cropImage(
					filePath: $imageFilePath->contents,
					topX: (int)$topX->contents,
					bottomX: (int)$bottomX->contents,
					topY: (int)$topY->contents,
					bottomY: (int)$bottomY->contents,
				);
			} catch (AnimatedWebPNotSupported|InvalidImage|FileNotFound|InvalidFilePath|InvalidImageDimension|ImageProcessingException $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::CROP_IMAGE->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"filePath"=>$imageFilePath,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("post", "/upload-image-file")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function uploadImageFiles(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$uploadDirectory = $payload->getTextPayload("to-directory");
				$imageFile = $payload->getFileUploadPayload("file-binary");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				ImageManagerService::uploadImage(
					filePath: sprintf("%s/%s", $uploadDirectory->contents, $imageFile->fileName),
					contents: $imageFile->contents,
					overrideExistingFile: false,
				);
			} catch (InvalidFileName|AnimatedWebPNotSupported|InvalidImage|FileNotFound|InvalidDirectory|ImageFileAlreadyExists|ImageIsAThumb $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::UPLOAD_IMAGE->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"fileName"=>$imageFile->fileName,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("post", "/upload-image-by-url-source")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function uploadImageByURLSource(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$uploadDirectory = $payload->getTextPayload("to-directory");
				$imageSourceURL = $payload->getTextPayload("image-source");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				$newImageFilePath = ImageManagerService::downloadImageByURL(
					saveToDirectoryPath: $uploadDirectory->contents,
					url: $imageSourceURL->contents,
				);
			} catch (InvalidFileName|AnimatedWebPNotSupported|InvalidImage|FileNotFound|GuzzleException|ImageFileAlreadyExists|InvalidDirectory|ImageIsAThumb $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::UPLOAD_IMAGE->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"fromURL"=>$imageSourceURL,
					"fileName"=>basename($newImageFilePath),
				]),
			);

			return new JSONSuccess();
		}

		#[Route("post", "/clone-image")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function cloneImage(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$imagePath = $payload->getTextPayload("image-path");
				$cloneNameWithoutExtension = $payload->getTextPayload("new-image-name");
				$imageExtension = $payload->getTextPayload("image-type");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				$clonedImageFilePath = ImageManagerService::cloneImage(
					filePath: $imagePath->contents,
					newFileNameWithoutExtension: $cloneNameWithoutExtension->contents,
					desiredImageExtension: $imageExtension->contents,
				);
			} catch (InvalidFileName|AnimatedWebPNotSupported|InvalidImage|FileNotFound|InvalidDirectory|InvalidFilePath|Exception\InvalidImageExtension|Exception\NoOperation|ImageFileAlreadyExists|ImageIsAThumb|ImageProcessingException $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::CLONE_IMAGE->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"originalImage"=>$imagePath->contents,
					"newFilePath"=>$clonedImageFilePath,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("patch", "/regenerate-thumb")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function regenerateImageThumb(Request $request): JSONResult{
			$payload = $request->getPayload();

			try {
				$imagePath = $payload->getTextPayload("image-path");
			} catch (NoPayloadFound $e) {
				return new JSONError($e->getMessage());
			}

			try {
				ImageManagerService::regenerateThumb(
					filePath: $imagePath->contents,
				);
			} catch (AnimatedWebPNotSupported|InvalidImage|FileNotFound|InvalidFilePath|Exception\MissingParameter|ImageIsAThumb|ImageProcessingException $e) {
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess();
		}

		#[Route("delete", "/delete-image")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function deleteImage(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$imagePath = $payload->getTextPayload('image-path');
			} catch (NoPayloadFound $e) {
				return new JSONError($e->getMessage());
			}

			try {
				ImageManagerService::deleteImage(
					filePath: $imagePath->contents,
				);
			} catch (InvalidFilePath|Exception\MissingParameter|AnimatedWebPNotSupported|InvalidImage|FileNotFound $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::DELETE_IMAGE->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"filePath"=>$imagePath->contents,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("post", "/new-directory")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function createNewDirectory(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$parentDirectory = $payload->getTextPayload("parent-directory");
			} catch (NoPayloadFound $e) {
				return new JSONError($e->getMessage());
			}

			try {
				$newFolderPath = ImageManagerService::newDirectory(
					parentDirectory: $parentDirectory->contents,
				);
			} catch (MaximumNewFolderDepthExceeded|InvalidDirectory $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::CREATE_IMAGE_DIRECTORY->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"imageDirectory"=>$newFolderPath,
				]),
			);

			return new JSONSuccess([
				"newFolderName"=>basename($newFolderPath),
				"newFolderPath"=>$newFolderPath,
			]);
		}

		#[Route("patch", "/rename-directory")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function renameEntireDirectory(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$currentFullPath = $payload->getTextPayload("current-full-path");
				$newFolderName = $payload->getTextPayload("directory-name");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				$newFolderPath = ImageManagerService::renameDirectory(
					directoryPath: $currentFullPath->contents,
					newFolderName: $newFolderName->contents
				);
			} catch (InvalidFolderName|NewDirectoryWithSameNameExists|PathDoesntExist|InvalidDirectory|Exception\MissingParameter $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::RENAME_IMAGE_DIRECTORY->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"oldFolderPath"=>$currentFullPath->contents,
					"newFolderPath"=>$newFolderPath,
				]),
			);

			return new JSONSuccess([
				"newDirectoryPath"=>$newFolderPath,
				"childContents"=>FileSystemUtilities::fetchDirectoryAndDescendants($newFolderPath),
			]);
		}

		#[Route("delete", "/delete-directory")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function removeDirectory(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$directoryToRemove = $payload->getTextPayload("directory-path");
			} catch (NoPayloadFound $e) {
				return new JSONError($e->getMessage());
			}

			try {
				ImageManagerService::deleteDirectory(
					directoryPath: $directoryToRemove->contents,
				);
			} catch (InvalidDirectory $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::DELETE_IMAGE_DIRECTORY->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"imageDirectory"=>$directoryToRemove->contents,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("patch", "/move-directory")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGES)]
		public function moveDirectory(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$currentFullPath = $payload->getTextPayload("current-full-path");
				$newParentDirectory = $payload->getTextPayload("new-parent-directory");
			} catch (NoPayloadFound $e) {
				return new JSONError($e->getMessage());
			}

			try {
				$newDirectoryPath = ImageManagerService::moveDirectory(
					currentPath: $currentFullPath->contents,
					newPath: $newParentDirectory->contents,
				);
			} catch (InvalidFolderName|NewDirectoryWithSameNameExists|PathDoesntExist|InvalidDirectory $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::MOVE_IMAGE_DIRECTORY->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"oldPath"=>$currentFullPath->contents,
					"imageDirectory"=>basename($newDirectoryPath),
				]),
			);

			return new JSONSuccess([
				"destinationNewChildContents"=>FileSystemUtilities::fetchDirectoryAndDescendants($newParentDirectory->contents),
			]);
		}
	}