<?php
	namespace ImageManager;

	use FileSystemUtilities\Exceptions\InvalidFileName;
	use FileSystemUtilities\Exceptions\InvalidFolderName;
	use FileSystemUtilities\exceptions\MaximumNewFolderDepthExceeded;
	use FileSystemUtilities\Exceptions\NewDirectoryWithSameNameExists;
	use FileSystemUtilities\Exceptions\PathDoesntExist;
	use FileSystemUtilities\FileSystemUtilities;
	use FileSystemUtilities\FSImageFile;
	use GDHelper\Exceptions\AnimatedWebPNotSupported;
	use GDHelper\exceptions\FileNotFound;
	use GDHelper\exceptions\ImageTypesAreTheSame;
	use GDHelper\Exceptions\InvalidImage;
	use GDHelper\exceptions\UnrecognizedImageExtension;
	use GDHelper\GDHelper;
	use GuzzleHttp\Client;
	use GuzzleHttp\Exception\GuzzleException;
	use ImageManager\Exception\InvalidDirectory;
	use ImageManager\Exception\InvalidFilePath;
	use ImageManager\Exception\InvalidImageExtension;
	use ImageManager\Exception\MissingParameter;
	use ImageManager\Exception\NoOperation;
	use Uplift\ImageManager\Exceptions\InvalidImageDimension;
	use Uplift\ImageManager\ImageFileAlreadyExists;
	use Uplift\ImageManager\ImageIsAThumb;
	use Uplift\ImageManager\ImageManager;
	use Uplift\ImageManager\ImageThumbAlreadyExists;
	use Uplift\ImageManager\NoFallbackFileFound;
	use Uplift\ImageProcessing\Exceptions\ImageProcessingException;
	use Uplift\ImageProcessing\ImageProcessing;
	use ValueError;

	class ImageManagerService{

		public static function getImageURIFromFilePath(string $fullFilePath): string{
			$slashNormalizedDirectoryPath = str_replace(
				"\\",
				"/",
				str_replace(
					realpath(ImageManager::IMAGES_DIRECTORY),
					"",
					$fullFilePath
				)
			);

			// Attach the public URI base to the image path
			return sprintf("%s%s", ImageManager::PUBLIC_IMAGES_URI_BASE, $slashNormalizedDirectoryPath);
		}

		/**
		 * @throws InvalidImage
		 * @throws ImageFileAlreadyExists
		 * @throws FileNotFound
		 * @throws AnimatedWebPNotSupported
		 * @throws ValueError
		 * @throws ImageIsAThumb
		 * @throws InvalidDirectory
		 * @throws InvalidFileName
		 * @throws ImageProcessingException
		 */
		public static function uploadImage(
			string $filePath,
			string $contents,
			bool $overrideExistingFile,
		): FSImageFile{
			// Resolve the file path
			$directoryToUploadTo = dirname($filePath);
			$directoryPathNormalized = realpath($directoryToUploadTo);
			$fileName = basename($filePath);

			if ($directoryPathNormalized === false){
				throw new ValueError("The directory to upload the image to does not exist. Directory provided: {$directoryToUploadTo}");
			}

			// Is the normalized path within the image manager uploads directory?
			$normalizedImageManagerUploadsDirectory = realpath(ImageManager::IMAGES_DIRECTORY);
			if (!str_starts_with(haystack: $directoryPathNormalized, needle: $normalizedImageManagerUploadsDirectory)){
				// Not allowed
				throw new ValueError("The normalized path to upload the image to is not within the allowed image uploads directory. Resolved directory path: {$directoryPathNormalized}");
			}

			$fullUploadPath = sprintf("%s/%s", $directoryPathNormalized, $fileName);
			$fileNameWithoutExtension = pathinfo($fullUploadPath, PATHINFO_FILENAME);

			if (!FileSystemUtilities::isNameFileSafe($fileNameWithoutExtension)){
				throw new InvalidFileName(
					sprintf(
						"The file name %s is not an acceptable file name to store in the file system.",
						$fileNameWithoutExtension,
					),
				);
			}

			if (!ImageManager::isValidImageDirectory(dirname($fullUploadPath))){
				throw new InvalidDirectory(
					sprintf(
						"The directory %s is not a valid image upload directory.",
						dirname($fullUploadPath),
					),
				);
			}

			// Good to go - but does the file already exist?
			if ($overrideExistingFile === false){
				if (file_exists($fullUploadPath)){
					throw new ImageFileAlreadyExists(
						sprintf(
							"There is already a file at %s",
							$fullUploadPath,
						),
					);
				}
			}

			// Good to upload
			file_put_contents($fullUploadPath, $contents);

			try {
				$fsImage = new FSImageFile(
					fileName: $fileName,
					fullFilePath: $fullUploadPath,
					uri: ImageManager::getImageURIFromFilePath($fullUploadPath),
					acceptSVG: true,
				);

				ImageManager::generateThumbForFSImage($fsImage);

				return $fsImage;
			}catch(InvalidImage $e){
				// Unlink the image if it is invalid
				unlink($fullUploadPath);

				// Rethrow
				throw $e;
			}
		}

		/**
		 * @throws InvalidImage
		 * @throws AnimatedWebPNotSupported
		 * @throws ImageIsAThumb
		 * @throws ImageFileAlreadyExists
		 * @throws FileNotFound
		 * @throws GuzzleException
		 * @throws InvalidDirectory
		 * @throws InvalidFileName|ImageProcessingException
		 */
		public static function downloadImageByURL(
			string $saveToDirectoryPath,
			string $url,
		): string{

			// Get a file name from the URL
			$parsedURL = parse_url($url);
			$urlPath = $parsedURL['path'];
			$fileName = basename($urlPath);
			$fullPathToNewLocalImage = sprintf("%s/%s", $saveToDirectoryPath, $fileName);

			$httpClient = new Client();

			$response = $httpClient->request(
				method:"GET",
				uri:$url,
			);

			$bodyStream = $response->getBody();
			$fileContents = $bodyStream->getContents();
			ImageManagerService::uploadImage(
				filePath: $fullPathToNewLocalImage,
				contents:$fileContents,
				overrideExistingFile:false,
			);

			return $fullPathToNewLocalImage;
		}

		/**
		 * @throws FileNotFound
		 * @throws InvalidDirectory
		 * @throws \GDHelper\exceptions\AnimatedWebPNotSupported
		 * @return FSImageFile[]
		 */
		public static function fetchDirectoryImages(string $directory): array{
			$requiredBasedPath = realpath(ImageManager::IMAGES_DIRECTORY);

			// Check for an empty directory parameter
			if (empty($directory)){
				throw new InvalidDirectory("Empty directory parameter.");
			}

			// Check for change-directory paths
			if (str_contains($directory, "/./") || str_contains($directory, "/..")){
				throw new InvalidDirectory("Invalid dot access directories in directory parameter.");
			}

			// The directory will be a full file system path.
			if (!str_starts_with($directory, $requiredBasedPath)){
				throw new InvalidDirectory(
					sprintf(
						"The provided path (%s) does not match the required beginning path of (%s).",
						$directory,
						$requiredBasedPath
					),
				);
			}

			// Check if the directory exists
			if (!file_exists($directory)){
				throw new InvalidDirectory("The provided directory does not exist..");
			}

			// Fetch the files
			$fileNames = array_diff(scandir($directory), ['.','..']);
			$imageFiles = [];
			foreach($fileNames as $fileName){
				$fullPath = sprintf("%s%s%s", $directory, DIRECTORY_SEPARATOR, $fileName);
				if (!is_dir($fullPath)) {
					try {
						$newFSImage = new FSImageFile(
							fileName: $fileName,
							fullFilePath: $fullPath,
							uri: self::getImageURIFromFilePath($fullPath),
							acceptSVG: true,
							createGDHelper: false,
						);

						// If it's a WebP, get the fallback URI
						if (strtolower($newFSImage->fileExtension) === "webp"){
							try {
								$fallbackImage = self::getFallbackImage($newFSImage->fullFilePath);
								$newFSImage->fallbackImage = $fallbackImage;
							} catch (AnimatedWebPNotSupported|InvalidImage|FileNotFound|InvalidFilePath|NoFallbackFileFound) {}
						}

						$imageFiles[] = $newFSImage;
					} catch (InvalidImage $e) {

					}
				}
			}

			// Some operating systems will sort case-sensitive. Sort them here insensitively
			usort($imageFiles, function (FSImageFile $a, FSImageFile $b){
				return strtolower($a->fileName) > strtolower($b->fileName) ? 1 : 0;
			});

			return $imageFiles;
		}

		/**
		 * @throws MissingParameter
		 * @throws InvalidImage
		 * @throws InvalidFilePath
		 * @throws \GDHelper\exceptions\AnimatedWebPNotSupported
		 * @throws ImageFileAlreadyExists
		 * @throws FileNotFound
		 * @throws InvalidDirectory
		 */
		public static function moveImage(
			string $imagePath,
			string $newDirectoryPath,
			int $allowOverwrites,
		): void{
			if (empty($imagePath)){
				throw new MissingParameter("Missing parameters for the imagePath");
			}

			if (empty($newDirectoryPath)){
				throw new MissingParameter("Missing parameters for the newDirectoryPath");
			}

			// Verify both the imagePath and the newDirectoryPath start with the image directory
			$osSpecificImagePath = realpath(ImageManager::IMAGES_DIRECTORY);

			if (!str_starts_with($imagePath, $osSpecificImagePath)){
				throw new InvalidFilePath("The imagePath must begin with the system image's directory");
			}

			if (!str_starts_with($newDirectoryPath, $osSpecificImagePath)){
				throw new InvalidDirectory("The newDirectoryPath must begin with the system image's directory");
			}

			// Verify the new location exists
			if (!file_exists($newDirectoryPath)){
				throw new InvalidDirectory("The newDirectoryPath does not exist on the file system.");
			}

			// Create an FSImageFile object so it can handle the thumb check for us
			$fsImage = new FSImageFile(
				fileName: basename($imagePath),
				fullFilePath: $imagePath,
				uri: null,
				acceptSVG: true,
			);

			// Get the new file paths and check if files exist in the new location with the same name
			$newImageLocation = sprintf("%s/%s", $newDirectoryPath, $fsImage->fileName);
			$newThumbLocation = null;
			if ($fsImage->thumbFilePath !== null) {
				// Is the new path a thumbs folder?
				if (basename($newDirectoryPath) === "thumbs"){
					// Do not check for a thumbs directory
				}else{
					// Is the current image a thumb?
					if (basename(dirname($imagePath)) === "thumbs"){
						// Do not check for a thumbs folder or image.
						// The image being moved is a thumb for some reason
					}else {
						// Does a thumbs folder exist at the new path?
						$newPathThumbDirectory = sprintf("%s/thumbs", $newDirectoryPath);
						if (!file_exists($newPathThumbDirectory)){
							@mkdir($newPathThumbDirectory);
						}

						$newThumbLocation = sprintf("%s/thumbs/%s", $newDirectoryPath, $fsImage->fileName);
					}
				}
			}

			if (file_exists($newImageLocation) && $allowOverwrites === 0){
				// TODO allow overwrite prompts or something
				throw new ImageFileAlreadyExists("An image with the same name already exists in the new directory.");
			}

			if ($newThumbLocation !== null) {
				if (file_exists($newThumbLocation) && $allowOverwrites === 0) {
					// TODO allow overwrite prompts or something
					throw new ImageFileAlreadyExists("A thumb image file with the same name already exists in the new directory's thumbs folder.");
				}
			}

			// Error checks done, just move the files
			@rename($imagePath, $newImageLocation);
			if ($newThumbLocation !== null){
				@rename($fsImage->thumbFilePath, $newThumbLocation);
			}
		}

		/**
		 * @throws InvalidImage
		 * @throws ImageThumbAlreadyExists
		 * @throws InvalidFilePath
		 * @throws ImageIsAThumb
		 * @throws AnimatedWebPNotSupported
		 * @throws ImageFileAlreadyExists
		 * @throws FileNotFound
		 * @throws InvalidDirectory
		 * @throws InvalidFileName
		 */
		public static function renameImage(
			string $filePath,
			string $newFileNameWithoutExtension,
		): string{
			if (empty($filePath)){
				throw new InvalidFilePath("Empty original file path parameter sent.");
			}

			if (empty($newFileNameWithoutExtension)){
				throw new InvalidFileName("Empty new file name sent.");
			}

			if (!FileSystemUtilities::isNameFileSafe($newFileNameWithoutExtension)){
				throw new InvalidFileName("New file name is invalid.");
			}

			if (!ImageManager::isValidImageDirectory($filePath)){
				throw new InvalidDirectory("Modified file path provided. Invalid directory change in file path.");
			}

			if (ImageManager::isImagePathAThumbImage($filePath)){
				throw new ImageIsAThumb("Renaming image thumbs not allowed. Please rename the parent image instead.");
			}

			// Handles renaming the thumb file as well
			return realpath(ImageManager::renameImageFile(
				imageFilePath: $filePath,
				newFileNameWithoutExtension: $newFileNameWithoutExtension,
			));
		}

		/**
		 * @throws InvalidImageDimension
		 * @throws InvalidImage
		 * @throws InvalidFilePath
		 * @throws AnimatedWebPNotSupported
		 * @throws ImageIsAThumb
		 * @throws FileNotFound
		 * @throws InvalidDirectory
		 * @throws ImageProcessingException
		 */
		public static function resizeImage(
			string $filePath,
			int $newWidth,
			int $newHeight,
		): void{

			if (empty($filePath)){
				throw new InvalidFilePath("Image path is empty.");
			}

			// Resolve sym links and stuff
			$filePath = realpath($filePath);
			if ($filePath === false){
				throw new InvalidFilePath("Invalid file path for image. The image or directory does not exist.");
			}

			if ($newWidth <= 0){
				throw new InvalidImageDimension("Image dimensions must be greater than 0. Width is too small.");
			}

			if ($newHeight <= 0){
				throw new InvalidImageDimension("Image dimensions must be greater than 0. Height is too small.");
			}

			if (!ImageManager::isValidImageDirectory(dirname($filePath))){
				throw new InvalidDirectory(
					sprintf(
						"The provided image path (%s) is not a valid directory for system images.",
						dirname($filePath),
					),
				);
			}

			if (ImageManager::isImagePathAThumbImage($filePath)){
				throw new ImageIsAThumb("Cannot resize a thumb image.");
			}

			$fsImage = new FSImageFile(
				fileName: basename($filePath),
				fullFilePath: $filePath,
				uri:null,
				acceptSVG: false,
				createGDHelper: false,
			);

			$imageProcessing = new ImageProcessing();
			$newImage = $imageProcessing->resize($fsImage->fileName, file_get_contents($fsImage->fullFilePath), $newWidth, $newHeight);
			// Save the newGDImage binary data
			file_put_contents($filePath, $newImage);

			try {
				ImageManager::generateThumbForFSImage($fsImage);
			} catch (ImageIsAThumb $e) {
				// Ignore. Can't happen. This is checked in an if statement above
			}
		}

		/**
		 * Crops an image given rectangle locations (top left and bottom right corners). Modifies the
		 * filePath provided on disk when the operation is successful.
		 * @throws InvalidImageDimension
		 * @throws InvalidImage
		 * @throws InvalidFilePath
		 * @throws AnimatedWebPNotSupported
		 * @throws FileNotFound|ImageProcessingException
		 */
		public static function cropImage(
			string $filePath,
			int $topX,
			int $bottomX,
			int $topY,
			int $bottomY,
		): void{
			if (empty($filePath)){
				throw new InvalidFilePath("Image path is empty.");
			}

			// Resolve sym links and stuff
			$filePath = realpath($filePath);
			if ($filePath === false){
				throw new InvalidFilePath("Invalid file path for image. The image or directory does not exist.");
			}

			if ($topX < 0){
				throw new InvalidImageDimension(
					sprintf(
						"The top X location is invalid. Provided value: %s",
						$topX,
					),
				);
			}

			if ($bottomX <= 0){
				throw new InvalidImageDimension(
					sprintf("The bottom X location is invalid. Provided value: %s", $bottomX)
				);
			}

			if ($topY < 0){
				throw new InvalidImageDimension(
					sprintf("The top Y location is invalid. Provided value: %s", $topY)
				);
			}

			if ($bottomY <= 0){
				throw new InvalidImageDimension(
					sprintf("The bottom Y location is invalid. Provided value: %s", $bottomY)
				);
			}

			if (!ImageManager::isValidImageDirectory(dirname($filePath))){
				throw new InvalidFilePath(
					sprintf(
						"The provided image path (%s) is not a valid directory for system images.",
						dirname($filePath)
					)
				);
			}

			$fsImage = new FSImageFile(
				fileName: basename($filePath),
				fullFilePath: $filePath,
				uri:null,
				acceptSVG: false,
				createGDHelper: false,
			);

			$imageProcessing = new ImageProcessing();
			$croppedImage = $imageProcessing->crop(
				$fsImage->fileName,
				file_get_contents($fsImage->fullFilePath),
				$topX,
				$topY,
				$bottomX,
				$bottomY
			);

			// Save the cropped image
			file_put_contents($filePath, $croppedImage);

			try {
				ImageManager::generateThumbForFSImage($fsImage);
			}catch(ImageIsAThumb $e){
				// The user cropped a thumb image.
				// Facepalm?
			}
		}

		/**
		 * Clones an image with a new file name and a new file extension (the file extension doesn't have to
		 * be different from the original if no type conversion is desired).
		 * @throws InvalidImage
		 * @throws InvalidFilePath
		 * @throws ImageIsAThumb
		 * @throws AnimatedWebPNotSupported
		 * @throws InvalidImageExtension
		 * @throws InvalidFileName
		 * @throws NoOperation
		 * @throws ImageFileAlreadyExists
		 * @throws FileNotFound
		 * @throws InvalidDirectory|ImageProcessingException
		 */
		public static function cloneImage(
			string $filePath,
			string $newFileNameWithoutExtension,
			string $desiredImageExtension,
		): string{
			$filePathNormalized = realpath($filePath);

			if ($filePathNormalized === false){
				throw new InvalidFilePath(
					sprintf(
						"There is no image file at the provided path %s",
						$filePath,
					),
				);
			}

			// -1 is the value of no extension selected in the HTML <option> list
			if ($desiredImageExtension === "-1"){
				throw new InvalidImageExtension("Please select an image extension.");
			}

			if (empty($newFileNameWithoutExtension)){
				throw new InvalidFileName("The cloned image must have a name.");
			}

			// Check if the image name is the same and the image extension is the same
			// If so, ignore
			$currentImageNameWithoutExtension = pathinfo($filePathNormalized, PATHINFO_FILENAME);
			$currentImageExtension = strtolower(pathinfo($filePathNormalized, PATHINFO_EXTENSION));
			if ($currentImageNameWithoutExtension === $newFileNameWithoutExtension && strtolower($desiredImageExtension) === strtolower($currentImageExtension)){
				throw new NoOperation("Nothing to clone. Resulting image would be the same name and image type.");
			}

			$directory = dirname($filePathNormalized);

			if (!ImageManager::isValidImageDirectory($directory)){
				throw new InvalidDirectory("The upload directory is not a valid image directory.");
			}

			if (!FileSystemUtilities::isNameFileSafe($newFileNameWithoutExtension)){
				throw new InvalidFileName(
					sprintf("The new file name (%s) contains invalid characters.", $newFileNameWithoutExtension)
				);
			}

			if ($currentImageExtension === "jpeg"){
				// Correct it to JPG
				$currentImageExtension = "jpg";
			}

			// Is the changed extension valid?
			$acceptedExtensions = ["jpg", "png", "gif", "webp"];
			if (!in_array($desiredImageExtension, $acceptedExtensions)){
				throw new InvalidImageExtension("The extension you're trying to convert to is not a valid image extension.");
			}

			// Check if an image exists in the new path already
			$newImageLocation = sprintf("%s/%s.%s", $directory, $newFileNameWithoutExtension, $desiredImageExtension);

			if (file_exists($newImageLocation)){
				throw new ImageFileAlreadyExists(sprintf("An image already exists at the path %s", $newImageLocation));
			}

			$fsImage = new FSImageFile(
				fileName: basename($filePathNormalized),
				fullFilePath: $filePathNormalized,
				uri: null,
				acceptSVG: true,
				createGDHelper: false,
			);

			if ($currentImageExtension === $desiredImageExtension){
				// Just clone the image
				@copy($fsImage->fullFilePath, $newImageLocation);

				// New FSImage for it
				$newFSImage = new FSImageFile(
					fileName: basename($newImageLocation),
					fullFilePath: $newImageLocation,
					uri: null,
					acceptSVG: true,
					createGDHelper: false,
				);

				// Generate the thumb for the new image, if necessary
				if (!ImageManager::isImagePathAThumbImage($newImageLocation)) {
					ImageManager::generateThumbForFSImage($newFSImage);
				}
			}else{
				// Image's type has been changed.

				$imageProcessing = new ImageProcessing();
				$newImageMimeType = ImageProcessing::IMAGE_EXTENSION_TO_MIME_MAP[$desiredImageExtension];
				$convertedImage = $imageProcessing->convertImageType(
					basename($newImageLocation),
					file_get_contents($fsImage->fullFilePath), // Use the original image as the file contents
					$newImageMimeType
				);

				// Save the converted image
				file_put_contents($newImageLocation, $convertedImage);

				// Create a new FSImage of it
				$newFSImage = new FSImageFile(
					fileName: basename($newImageLocation),
					fullFilePath: $newImageLocation,
					uri: null,
					acceptSVG: true,
					createGDHelper: false,
				);

				// Generate a thumb image if necessary
				if (!ImageManager::isImagePathAThumbImage($newFSImage->fullFilePath)) {
					ImageManager::generateThumbForFSImage($newFSImage);
				}
			}

			return $newImageLocation;
		}

		/**
		 * @throws MissingParameter
		 * @throws InvalidImage
		 * @throws FileNotFound
		 * @throws InvalidFilePath
		 * @throws AnimatedWebPNotSupported
		 */
		public static function deleteImage(
			string $filePath,
		): void{
			if (empty($filePath)){
				throw new MissingParameter("Parameter image-path cannot be empty.");
			}

			$imagePathNormalized = realpath($filePath);

			if ($imagePathNormalized === false){
				throw new InvalidFilePath(sprintf("The image to be cloned no longer exists at location %s.", $filePath));
			}

			$directory = dirname($imagePathNormalized);

			if (!ImageManager::isValidImageDirectory($directory)){
				throw new InvalidFilePath("The provided image is not in a valid system image directory.");
			}

			$fsImage = new FSImageFile(
				fileName: basename($imagePathNormalized),
				fullFilePath: $imagePathNormalized,
				uri: null,
				acceptSVG: true,
			);

			if ($fsImage->thumbFilePath !== null){
				// Has a thumb. Remove it
				unlink($fsImage->thumbFilePath);
			}

			// Remove the image
			unlink($imagePathNormalized);
		}

		/**
		 * @throws InvalidImage
		 * @throws MissingParameter
		 * @throws FileNotFound
		 * @throws InvalidFilePath
		 * @throws ImageIsAThumb
		 * @throws AnimatedWebPNotSupported|ImageProcessingException
		 */
		public static function regenerateThumb(
			string $filePath,
		): void{
			if (empty($filePath)){
				throw new MissingParameter("Parameter image-path cannot be empty.");
			}

			$imagePathNormalized = realpath($filePath);

			if ($imagePathNormalized === false){
				throw new InvalidFilePath(sprintf("The image to be cloned no longer exists at location %s.", $filePath));
			}

			$directory = dirname($imagePathNormalized);

			if (!ImageManager::isValidImageDirectory($directory)){
				throw new InvalidFilePath("The provided image is not in a valid system image directory.");
			}

			if (ImageManager::isImagePathAThumbImage($imagePathNormalized)) {
				throw new ImageIsAThumb("Will not regenerate a thumb file for an image that is already a thumb.");
			}

			$fsImage = new FSImageFile(
				fileName: basename($imagePathNormalized),
				fullFilePath: $imagePathNormalized,
				uri: null,
				acceptSVG: true,
			);

			ImageManager::generateThumbForFSImage($fsImage);
		}

		/**
		 * Generates a new directory inside the provided parentDirectory. This method will auto-generate the new
		 * directory name.
		 * @throws InvalidDirectory
		 * @throws MaximumNewFolderDepthExceeded
		 */
		public static function newDirectory(
			string $parentDirectory,
		): string{

			$parentDirectoryNormalized = realpath($parentDirectory);

			if ($parentDirectoryNormalized === false){
				throw new InvalidDirectory(sprintf("The parent directory (%s) does not exist.", $parentDirectory));
			}

			if (!ImageManager::isValidImageDirectory($parentDirectoryNormalized)){
				throw new InvalidDirectory("The parent directory is not a valid location for images.");
			}

			$newFolderName = FileSystemUtilities::getNextNewDirectoryName($parentDirectoryNormalized);
			$fullNewPath = sprintf("%s/%s", $parentDirectoryNormalized, $newFolderName);

			// Ask the magical wizard to make the folder poor into existence
			@mkdir($fullNewPath);

			return $fullNewPath;
		}

		/**
		 * Renames a directory. It does this by utilizing a FileSystemUtilities method that first
		 * recurses into all descendants and moves them to the new location first - as you cannot rename
		 * a directory that has children.
		 * @throws InvalidDirectory
		 * @throws InvalidFolderName
		 * @throws MissingParameter
		 * @throws NewDirectoryWithSameNameExists
		 * @throws PathDoesntExist
		 */
		public static function renameDirectory(
			string $directoryPath,
			string $newFolderName,
		): string{

			$directoryPathNormalized = realpath($directoryPath);

			if ($directoryPathNormalized === false){
				throw new InvalidDirectory(sprintf("The current directory provided (%s) does not exist.", $directoryPath));
			}

			if (empty($newFolderName)){
				throw new MissingParameter("The new folder name cannot be empty.");
			}

			if (!FileSystemUtilities::isNameFileSafe($newFolderName)){
				throw new InvalidDirectory("The new directory name - {$newFolderName} - is not valid or allowed.");
			}

			if (!ImageManager::isValidImageDirectory(dirname($directoryPathNormalized))){
				throw new InvalidDirectory(sprintf("The new folder path (%s) is not a valid image directory.", dirname($directoryPathNormalized)));
			}

			// Perform the operation
			$newFolderPath = sprintf("%s/%s", dirname($directoryPathNormalized), $newFolderName);

			FileSystemUtilities::moveDirectoryToNewLocation($directoryPathNormalized, $newFolderPath);

			return $newFolderPath;
		}

		/**
		 * @throws InvalidDirectory
		 */
		public static function deleteDirectory(
			string $directoryPath,
		): void{

			$directoryPathNormalized = realpath($directoryPath);

			if ($directoryPathNormalized === false){
				throw new InvalidDirectory(sprintf("The directory provided (%s) does not exist.", $directoryPath));
			}

			if ($directoryPathNormalized === realpath(ImageManager::IMAGES_DIRECTORY)){
				throw new InvalidDirectory("You cannot remove the top level image directory root.");
			}

			if (!ImageManager::isValidImageDirectory($directoryPathNormalized)){
				throw new InvalidDirectory(sprintf("The provided directory (%s) cannot be removed by the image manager API. It is not a valid image directory.", $directoryPathNormalized));
			}

			// Remove the directory and all its contents
			FileSystemUtilities::recursivelyDeleteFolder($directoryPathNormalized);
		}

		/**
		 * @throws NewDirectoryWithSameNameExists
		 * @throws PathDoesntExist
		 * @throws InvalidFolderName
		 * @throws InvalidDirectory
		 */
		public static function moveDirectory(
			string $currentPath,
			string $newPath,
		): string{
			$currentPathNormalized = realpath($currentPath);
			$newPathNormalized = realpath($newPath);

			if ($currentPathNormalized === false){
				throw new InvalidDirectory(sprintf("The current directory provided (%s) does not exist.", $currentPath));
			}

			if ($newPathNormalized === false){
				throw new InvalidDirectory(sprintf("The new parent directory provided (%s) does not exist.", $newPath));
			}

			if ($currentPathNormalized === realpath(ImageManager::IMAGES_DIRECTORY)){
				throw new InvalidDirectory("You cannot move the top level image directory root.");
			}

			// Check if they are the same location.
			// This would cause an infinite loop of folder creation
			if ($currentPathNormalized === $newPathNormalized){
				throw new InvalidDirectory("You cannot move a folder to be within itself. That would be very strange recursion.");
			}

			// Get the new location
			$newDirectoryLocation = sprintf("%s/%s", $newPathNormalized, basename($currentPathNormalized));

			if (file_exists($newDirectoryLocation)){
				throw new InvalidDirectory("A folder in the new directory with the same name already exists.");
			}

			if (!ImageManager::isValidImageDirectory($newPathNormalized)){
				throw new InvalidDirectory(sprintf("The new folder path (%s) is not a valid image directory.", $newDirectoryLocation));
			}

			FileSystemUtilities::moveDirectoryToNewLocation($currentPathNormalized, $newDirectoryLocation);

			return $newDirectoryLocation;
		}

		/**
		 * @throws InvalidImage
		 * @throws FileNotFound
		 * @throws InvalidFilePath
		 * @throws AnimatedWebPNotSupported
		 * @throws NoFallbackFileFound
		 */
		public static function getFallbackImage(string $imageFilePath): FSImageFile{
			$normalizedFilePath = realpath($imageFilePath);

			if ($normalizedFilePath === false){
				throw new InvalidFilePath("$imageFilePath is not an existing image or file.");
			}

			$imageDirectory = dirname($imageFilePath);
			$currentImage = new FSImageFile(
				fileName: basename($imageFilePath),
				fullFilePath:$imageFilePath,
				uri:self::getImageURIFromFilePath($imageFilePath),
				acceptSVG:false,
				createGDHelper: false,
			);

			// Iterate the directory's files to see if there is a file with the same file name but different extension
			/** @var FSImageFile[] $fallbackFiles */
			$fallbackFiles = [];
			$directoryFiles = array_diff(scandir($imageDirectory), ['.','..']);

			foreach($directoryFiles as $fileName){
				$fullFilePath = sprintf("%s%s%s", $imageDirectory, DIRECTORY_SEPARATOR, $fileName);
				if (!is_dir($fullFilePath)) {
					$fileURI = self::getImageURIFromFilePath($fullFilePath);

					try {
						$fsImage = new FSImageFile(
							fileName: $fileName,
							fullFilePath: $fullFilePath,
							uri: $fileURI,
							acceptSVG: false,
							createGDHelper: false,
						);
					} catch (AnimatedWebPNotSupported|FileNotFound|InvalidImage $e) {
						// Ignore invalid image
						continue;
					}

					if (strtolower($fsImage->fileNameWithoutExtension) === strtolower($currentImage->fileNameWithoutExtension)) {
						if (strtolower($fsImage->fileExtension) !== strtolower($currentImage->fileExtension)) {
							// Same file name, different extension
							$fallbackFiles[] = $fsImage;
						}
					}
				}
			}

			if (!empty($fallbackFiles)){
				// Try and find a JPEG/JPG file first - JPEG takes priority
				foreach($fallbackFiles as $fallbackFile){
					$extension = strtolower($fallbackFile->fileNameWithoutExtension);
					if ($extension === "jpeg" || $extension === "jpg"){
						return $fallbackFile;
					}
				}

				// If we get here, then just pick the first file in the array - no JPEG found
				return $fallbackFiles[0];
			}else{
				throw new NoFallbackFileFound("No fallback file found for {$currentImage->fileName}.");
			}
		}

		public static function getThumbPathFromFilePath(string $imageFilePath): string{
			$fileName = basename($imageFilePath);

			return sprintf(
				"%s/thumbs/%s",
				dirname($imageFilePath),
				$fileName,
			);
		}

		/**
		 * @param string $imageFilePath
		 * @return array{width: int, height: int}
		 * @throws AnimatedWebPNotSupported
		 * @throws FileNotFound
		 * @throws InvalidImage
		 */
		public static function getImageDimensions(string $imageFilePath): array{
			$fsImage = new FSImageFile(
				fileName: basename($imageFilePath),
				fullFilePath: $imageFilePath,
				uri: null,
				acceptSVG:false,
			);

			return [
				"width"=>$fsImage->gdHelper->width,
				"height"=>$fsImage->gdHelper->height,
			];
		}
	}