<?php
	namespace Installation;

	use Accounts\Account;
	use GuzzleHttp\Client;
	use GuzzleHttp\Exception\GuzzleException;
	use GuzzleHttp\RequestOptions;
	use MonologWrapper\MonologWrapper;
	use Nox\ClassLoader\ClassLoader;
	use Nox\Http\FileUploadPayload;
	use Nox\ORM\Abyss;
	use Nox\RenderEngine\Exceptions\ParseError;
	use Nox\RenderEngine\Parser;
	use Page\Page;
	use Page\PageBreadcrumb;
	use Page\PageData;
	use Page\PageType;
	use Page\PublicationStatus;
	use PageArchives\PageArchivesService;
	use Roles\PermissionAlreadyExists;
	use Roles\PermissionCategories;
	use Roles\PermissionsService;
	use Roles\Role;
	use Settings\Settings;
	use Settings\SettingService;
	use System\HttpHelper;
	use System\System;
	use System\SystemInstallationState;
	use System\Themes;
	use Uplift\Exceptions\EmptyValue;
	use Uplift\Exceptions\IncompatiblePageType;
	use Uplift\Exceptions\MalformedValue;
	use Uplift\ImageManager\ImageFileAlreadyExists;

	class InstallationService{

		/**
		 * @return ParsedInitialPage[]
		 * @throws ParseError
		 */
		public static function getInitialPages(): array{
			$initialPages = [];

			$initialPagesDirectory = __DIR__ . "/../../resources/initial-pages";
			$directoryHandle = opendir($initialPagesDirectory);
			do{
				$fileName = readdir($directoryHandle);
				if ($fileName !== false){
					if ($fileName !== "." && $fileName !== "..") {
						$fullFilePath = sprintf("%s/%s", $initialPagesDirectory, $fileName);
						$initialPage = new ParsedInitialPage();
						$noxParser = new Parser($fullFilePath, null);
						$noxParser->parse();

						if (array_key_exists(key:"@Breadcrumbs", array:$noxParser->directives)) {
							$initialPage->breadcrumbs = json_decode(trim($noxParser->directives['@Breadcrumbs']), true);
						}else{
							$initialPage->breadcrumbs = [];
						}

						// Handle page data
						$initialPage->pageData = [];
						if (array_key_exists(key:"@PageData", array:$noxParser->directives)) {
							$pageDataObjectsJSON = $noxParser->directives['@PageData'];
							$pageDataObjects = json_decode($pageDataObjectsJSON, true);

							/** @var array{name: string, value: string} $pageDataObject */
							foreach($pageDataObjects as $pageDataObject){
								$initialPage->pageData[] = $pageDataObject;
							}
						}

						$initialPage->pageName = $noxParser->directives['@PageName'];
						$initialPage->layoutName = $noxParser->directives['@Layout'];
						$initialPage->pageType = PageType::fromName($noxParser->directives['@PageType']);
						$initialPage->route = $noxParser->directives['@Route'];
						$initialPage->excludedFromSitemap = (int) $noxParser->directives['@ExcludeFromSitemap'];
						$initialPage->body = $noxParser->directives['@Body'];
						$initialPage->head = $noxParser->directives['@Head'];
						$initialPage->fullFilePath = $fullFilePath;
						$initialPages[] = $initialPage;
					}
				}
			}while($fileName !== false);

			closedir($directoryHandle);

			return $initialPages;
		}

		private static function createMasterAccount(
			string $masterAccountUsername,
			string $masterAccountFirstName,
			string $masterAccountLastName,
			string $masterAccountPassword,
			int $adminRoleID,
		): Account{
			$account = new Account();
			$account->firstName = $masterAccountFirstName;
			$account->lastName = $masterAccountLastName;
			$account->password = password_hash($masterAccountPassword, PASSWORD_DEFAULT);
			$account->username = $masterAccountUsername;
			$account->roleID = $adminRoleID;
			$account->save();

			return $account;
		}

		private static function createClientAccount(
			string $clientAccountUsername,
			string $clientAccountFirstName,
			string $clientAccountLastName,
			string $clientAccountPassword,
			int $clientRoleID,
		): Account{
			$account = new Account();
			$account->firstName = $clientAccountFirstName;
			$account->lastName = $clientAccountLastName;
			$account->password = password_hash($clientAccountPassword, PASSWORD_DEFAULT);
			$account->username = $clientAccountUsername;
			$account->roleID = $clientRoleID;
			$account->save();

			return $account;
		}

		private static function createAdminRole(): Role{
			$adminRole = new Role();
			$adminRole->name = "Administrator";
			$adminRole->canBeDeleted = 0;
			$adminRole->hasAllPermissions = 1;
			$adminRole->save();

			// Create all permissions for this role
			// Create permissions for the role
			foreach(PermissionCategories::cases() as $category) {
				try {
					PermissionsService::createPermission(
						role: $adminRole,
						category: $category,
						isEnabled: true,
					);
				} catch (PermissionAlreadyExists) {
				}
			}

			return $adminRole;
		}

		/**
		 * Creates the default client role - which will only have access to the IPP form
		 */
		private static function createClientRole(): Role{
			$clientRole = new Role();
			$clientRole->name = "Client";
			$clientRole->canBeDeleted = 1;
			$clientRole->save();

			// Create a curated set of permissions for this role
			try {
				PermissionsService::createPermission(
					role: $clientRole,
					category: PermissionCategories::ACCESS_CLIENT_IPP_FORM,
					isEnabled: true,
				);
				PermissionsService::createPermission(
					role: $clientRole,
					category: PermissionCategories::MANAGE_PROJECT_PAGES,
					isEnabled: true,
				);
				PermissionsService::createPermission(
					role: $clientRole,
					category: PermissionCategories::ACCESS_EASY_EDITOR,
					isEnabled: true,
				);
			}catch(PermissionAlreadyExists $ignored){}

			return $clientRole;
		}

		/**
		 * @throws ParseError
		 * @throws IncompatiblePageType
		 */
		public static function installInitialPages(
			array $initialPageFileNames,
		): void{
			$logger = MonologWrapper::getLogger();
			$logger->info("Installing initial pages.");
			$initialPages = self::getInitialPages();
			foreach($initialPages as $initialPage){
				if (in_array(needle: basename($initialPage->fullFilePath), haystack: $initialPageFileNames)){
					$logger->info("Installing " . $initialPage->pageName);
					// Create the page
					$page = new Page();
					$page->pageName = $initialPage->pageName;
					$page->pageType = $initialPage->pageType->name;
					$page->pageRoute = $initialPage->route;
					$page->excludeFromSitemap = $initialPage->excludedFromSitemap;
					$page->pageHead = trim($initialPage->head);
					$page->pageBody = trim($initialPage->body);
					$page->pageLayout = $initialPage->layoutName;
					$page->publicationStatus = PublicationStatus::Published->value;
					$page->save();

					// Create the breadcrumbs
					/** @var array{uri: string, label:string} $crumb */
					foreach($initialPage->breadcrumbs as $index=>$crumb){
						$breadcrumb = new PageBreadcrumb();
						$breadcrumb->pageID = $page->id;
						$breadcrumb->position = $index;
						$breadcrumb->uri = $crumb['uri'];
						$breadcrumb->label = $crumb['label'];
						$breadcrumb->save();
					}

					// Create the page data
					/** @var array{name: string, value: string} $pageDataObject */
					foreach($initialPage->pageData as $pageDataObject){
						$pageData = new PageData();
						$pageData->name = $pageDataObject['name'];
						$pageData->value = $pageDataObject['value'];
						$pageData->pageID = $page->id;
						$pageData->save();
					}

					// Archive the page
					match($page->pageType){
						PageType::General->name => PageArchivesService::archiveGeneralPage($page),
						PageType::Service->name => PageArchivesService::archiveServicePage($page),
						PageType::Blog->name => PageArchivesService::archiveBlogPage($page),
						PageType::City->name => PageArchivesService::archiveCityPage($page),
						PageType::Project->name => PageArchivesService::archiveProjectPage($page),
					};

				}else{
					$logger->info("Ignoring initial page " . $initialPage->pageName . " as it wasn't selected by the user at the install page.");
				}
			}
		}

		public static function synchronizeModels(): void{
			$modelReflections = ClassLoader::$modelClassReflections;
			$abyss = new Abyss();
			$abyss->syncModels($modelReflections);
		}

		/**
		 * Registers this Uplift build with the master Uplift control panel. Will receive
		 * an API key and a Uuid for this build.
		 * @throws GuzzleException
		 * @throws MalformedValue
		 */
		public static function registerWithControlPanel(
			int $acceloCompanyID,
			string $companyName,
			bool $isTestBuild,
			bool $indexingDisabled,
			bool $isSuspended,
		): RegistrationResult{
			$logger = MonologWrapper::getLogger();
			$logger->info("Registering build with Uplift Control Panel.");
			$client = new Client();
			$registerEndpoint = sprintf("%s/uplift/build/register", System::getUpliftControlPanelHost());
			$request = $client->request(
				method:"PUT",
				uri:$registerEndpoint,
				options:[
					RequestOptions::MULTIPART => [
						[
							"name"=>"acceloCompanyID",
							"contents"=>$acceloCompanyID,
						],
						[
							"name"=>"companyName",
							"contents"=>$companyName,
						],
						[
							"name"=>"serverIP",
							"contents"=>HttpHelper::getIPFromRequest(),
						],
						[
							"name"=>"website",
							"contents"=>HttpHelper::getWebsiteBaseURL(),
						],
						[
							"name"=>"indexingDisabled",
							"contents"=>$indexingDisabled ? 1 : 0,
						],
						[
							"name"=>"version",
							"contents"=>System::VERSION,
						],
						[
							"name"=>"isTestBuild",
							"contents"=>$isTestBuild ? 1 : 0,
						],
						[
							"name"=>"isSuspended",
							"contents"=>$isSuspended ? 1 : 0,
						],
					],
				]
			);

			$logger->info("Request made successfully.");

			$body = $request->getBody();
			$returnContents = $body->getContents();
			/** @var array{uuid: string, apiKey: string} | null $data */
			$data = json_decode($returnContents, true);
			if ($data === null){
				$logger->error("Malformed JSON response. Response from server is: $returnContents");
				throw new MalformedValue(sprintf("The Uplift control panel registration endpoint returned malformed JSON. The response was %s.", $returnContents));
			}else{
				$uuid = $data['uuid'] ?? null;
				$apiKey = $data['apiKey'] ?? null;

				if ($uuid === null){
					$logger->error("Missing UUID from response. Response was: $returnContents");
					throw new MalformedValue("The Uplift control panel registration did not return a valid build UUID.");
				}

				if ($apiKey === null){
					$logger->error("Missing API key from response. Response was: $returnContents");
					throw new MalformedValue("The Uplift control panel registration did not return a valid API key for this build to utilize.");
				}

				$logger->info("Successfully registered build.");

				return new RegistrationResult($uuid, $apiKey);
			}
		}

		/**
		 * @throws EmptyValue
		 * @throws ImageFileAlreadyExists
		 * @throws InvalidInstallationState
		 * @throws MalformedValue
		 * @throws ParseError
		 * @throws GuzzleException
		 */
		public static function install(
			string $schemaType,
			int $acceloCompanyID,
			string $companyName,
			string $companyStreet,
			string $companyCity,
			string $companyState,
			string $companyPostal,
			string $companyPhoneNumber,
			string $companyFaxNumber,
			?FileUploadPayload $logo,
			?FileUploadPayload $favicon,
			string $masterAccountUsername,
			string $masterAccountFirstName,
			string $masterAccountLastName,
			string $masterAccountPassword,
			string $clientAccountUsername,
			string $clientAccountFirstName,
			string $clientAccountLastName,
			string $clientAccountPassword,
			?string $initialPageFileNamesToCreateJSON,
			bool $isTestBuild,
		): void{

			$companyName = trim($companyName);
			$masterAccountUsername = trim($masterAccountUsername);
			$masterAccountFirstName = trim($masterAccountFirstName);
			$masterAccountLastName = trim($masterAccountLastName);
			if ($initialPageFileNamesToCreateJSON !== null) {
				$initialPageFileNamesToCreate = json_decode($initialPageFileNamesToCreateJSON);
				if ($initialPageFileNamesToCreate === null){
					throw new MalformedValue(
						sprintf(
							"Invalid JSON value for the initial page file names payload. JSON error was: %s",
							json_last_error_msg(),
						),
					);
				}
			}

			if (empty($acceloCompanyID)){
				throw new EmptyValue("Please provide the numerical ID of the company profile in Accelo.");
			}

			if (empty($companyName)){
				throw new EmptyValue("The company name cannot be empty.");
			}

			if (empty($masterAccountUsername)){
				throw new EmptyValue("The master account username cannot be empty.");
			}

			if (empty($masterAccountFirstName)){
				throw new EmptyValue("The master account first name cannot be empty.");
			}

			if (empty($masterAccountLastName)){
				throw new EmptyValue("The master account last name cannot be empty.");
			}

			if (empty($masterAccountPassword)){
				throw new EmptyValue("The master account password cannot be empty.");
			}

			if (empty($clientAccountUsername)){
				throw new EmptyValue("The client account username cannot be empty.");
			}

			if (empty($clientAccountFirstName)){
				throw new EmptyValue("The client account first name cannot be empty.");
			}

			if (empty($clientAccountFirstName)){
				throw new EmptyValue("The client account last name cannot be empty.");
			}

			if (empty($clientAccountPassword)){
				throw new EmptyValue("The client account password cannot be empty.");
			}

			$installationState = System::getInstallationState();

			// NO_DATABASE_TABLES means it isn't installed
			if ($installationState === SystemInstallationState::NO_DATABASE_TABLES){

				$registrationResult = self::registerWithControlPanel(
					acceloCompanyID: $acceloCompanyID,
					companyName: $companyName,
					isTestBuild:$isTestBuild,
					indexingDisabled: true, // On new builds, indexing is always disabled
					isSuspended: false // On new builds, nobody is by-default suspended due to billing
				);

				self::synchronizeModels();

				SettingService::saveSetting(
					name: Settings::BUILD_UUID->value,
					value:$registrationResult->uuid,
				);

				SettingService::saveSetting(
					name: Settings::FORCE_HTTPS->value,
					value:1,
				);

				SettingService::saveSetting(
					name: Settings::ENTIRE_SITE_NO_INDEX->value,
					value:1,
				);

				SettingService::saveSetting(
					name: Settings::UPLIFT_CONTROL_PANEL_API_KEY->value,
					value:$registrationResult->apiKey,
				);

				SettingService::saveSetting(
					name: Settings::IS_TEST_BUILD->value,
					value:$isTestBuild,
				);

				// By default, opt out of Lightbox injections on new builds
				// This injection was created on 11/17/2023 to fix an issue where
				// v3->v4 upgrades didn't have Lightbox (v3 auto-injected Lightbox, while v4 did not and relied
				// on the layout having lightbox)
				SettingService::saveSetting(
					name: Settings::OPT_OUT_LIGHTBOX_INJECTION->value,
					value:"1",
				);

				$adminRole = self::createAdminRole();
				$clientRole = self::createClientRole();

				self::createMasterAccount(
					masterAccountUsername: $masterAccountUsername,
					masterAccountFirstName: $masterAccountFirstName,
					masterAccountLastName: $masterAccountLastName,
					masterAccountPassword: $masterAccountPassword,
					adminRoleID: $adminRole->id,
				);

				self::createClientAccount(
					clientAccountUsername: $clientAccountUsername,
					clientAccountFirstName: $clientAccountFirstName,
					clientAccountLastName: $clientAccountLastName,
					clientAccountPassword: $clientAccountPassword,
					clientRoleID: $clientRole->id,
				);

				SettingService::saveCompanySettings(
					schemaType: $schemaType,
					acceloCompanyID: $acceloCompanyID,
					companyName:$companyName,
					companyStreet:$companyStreet,
					companyCity:$companyCity,
					companyState:$companyState,
					companyPostal:$companyPostal,
					phoneNumbersJSON:json_encode([$companyPhoneNumber]),
					faxNumbersJSON:json_encode([$companyFaxNumber]),
					additionalAddressesJSON:json_encode([]),
					companyLogo:$logo,
					companyFavicon:$favicon,
				);

				if ($initialPageFileNamesToCreateJSON !== null) {
					self::installInitialPages($initialPageFileNamesToCreate);
				}

				// Set the theme to the default theme
				SettingService::saveSetting(Settings::THEME_NAME->value, Themes::DEFAULT_THEME_DIRECTORY_NAME);
			}else{
				$invalidInstallationState = new InvalidInstallationState(sprintf("The system is already partially installed. The current installation state is %s, but for an install to be successfully performed the installation state must be %s.", $installationState->name, SystemInstallationState::NO_DATABASE_TABLES->name));
				$invalidInstallationState->installationState = $installationState;
				throw $invalidInstallationState;
			}
		}
	}