<?php

	namespace Page\PageAttribute;

	use Nox\ORM\Abyss;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Exceptions\NoPrimaryKey;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;
	use Page\PageType;
	use Uplift\Exceptions\EmptyValue;
	use Uplift\Exceptions\MalformedValue;
	use Uplift\Exceptions\NoObjectFound;
	use Uplift\Exceptions\ObjectAlreadyExists;

	class PageAttributeService
	{
		const MAX_ATTRIBUTE_RESULTS_PER_PAGE = 25;

		/**
		 * @return array{attributes: PageAttributeQueryResponse[], totalPages: int}
		 */
		public static function query(
			string $query,
			int $pageNumber,
		): array{
			$query = trim($query);
			$columnQuery = new ColumnQuery();

			if (!empty($query)) {
				$columnQuery->where("name", "LIKE", "%$query%");
			}

			$resultOrder = new ResultOrder();
			$resultOrder->by("name", "ASC");
			$pager = new Pager(
				pageNumber: $pageNumber,
				limit: self::MAX_ATTRIBUTE_RESULTS_PER_PAGE
			);

			$attributes = PageAttribute::query(
				columnQuery: $columnQuery,
				resultOrder: $resultOrder,
				pager: $pager
			);

			$totalAttributesNoLimit = PageAttribute::count($columnQuery);
			$pageAttributeQueryResponses = [];

			foreach($attributes as $attribute){
				$pageAttributeQueryResponses[] = PageAttributeQueryResponse::fromPageAttribute($attribute);
			}

			return [
				"attributes"=>$pageAttributeQueryResponses,
				"totalPages"=>ceil($totalAttributesNoLimit / self::MAX_ATTRIBUTE_RESULTS_PER_PAGE)
			];
		}

		/**
		 * @throws MalformedValue
		 * @throws EmptyValue
		 * @throws ObjectAlreadyExists
		 */
		public static function createAttribute(
			string $name,
			array $pageTypes,
		): PageAttribute{
			$name = trim($name);

			if (empty($name)){
				throw new EmptyValue("Page attribute names cannot be empty.");
			}

			// Check that this attribute has some assigned page types
			if (empty($pageTypes)){
				throw new EmptyValue("Page attributes must have a page type, one or more, assigned to them.");
			}

			// Validate all page types
			foreach($pageTypes as $pageTypeName){
				if (PageType::fromName($pageTypeName) === null){
					throw new MalformedValue("Page type of '$pageTypeName' does not exist.");
				}
			}

			// Check for an existing attribute
			$existingAttribute = PageAttribute::queryOne(
				columnQuery: (new ColumnQuery())
					->where("name","=",$name)
			);

			if ($existingAttribute !== null){
				throw new ObjectAlreadyExists("A page attribute with that name already exists.");
			}

			$pageAttribute = new PageAttribute();
			$pageAttribute->name = $name;
			$pageAttribute->save();

			$pageAttribute->setAvailablePageTypes($pageTypes);


			return $pageAttribute;
		}

		/**
		 * @throws EmptyValue
		 * @throws NoObjectFound
		 * @throws ObjectAlreadyExists
		 * @throws MalformedValue
		 */
		public static function saveAttribute(
			int $attributeID,
			string $name,
			array $pageTypes,
		): PageAttribute{
			$name = trim($name);

			if (empty($name)){
				throw new EmptyValue("Page attribute names cannot be empty.");
			}

			// Check that this attribute has some assigned page types
			if (empty($pageTypes)){
				throw new EmptyValue("Page attributes must have a page type, one or more, assigned to them.");
			}

			// Validate all page types
			foreach($pageTypes as $pageTypeName){
				if (PageType::fromName($pageTypeName) === null){
					throw new MalformedValue("Page type of '$pageTypeName' does not exist.");
				}
			}

			// Fetch the attribute
			/** @var ?PageAttribute $pageAttribute */
			$pageAttribute = PageAttribute::fetch($attributeID);

			if ($pageAttribute === null){
				throw new NoObjectFound("No page attribute with ID $attributeID");
			}

			// Check for an existing attribute
			$existingAttribute = PageAttribute::queryOne(
				columnQuery: (new ColumnQuery())
					->where("name","=",$name)
					->and()
					->where("id", "!=", $attributeID)
			);

			if ($existingAttribute !== null){
				throw new ObjectAlreadyExists("A page attribute with that name already exists.");
			}

			$pageAttribute->name = $name;
			$pageAttribute->save();
			$pageAttribute->setAvailablePageTypes($pageTypes);

			return $pageAttribute;
		}

		/**
		 * @throws NoObjectFound
		 */
		public static function deleteAttribute(
			int $attributeID,
		): void{
			/** @var ?PageAttribute $pageAttribute */
			$pageAttribute = PageAttribute::fetch($attributeID);

			if ($pageAttribute === null){
				throw new NoObjectFound("No page attribute with ID $attributeID");
			}

			$pageAttribute->delete();

			// Delete any attribute->page connection values
			/** @var PageAttributePageValue[] $pageAttributePageValues */
			$pageAttributePageValues = PageAttributePageValue::query(
				columnQuery: (new ColumnQuery())
					->where("page_attribute_id","=", $attributeID)
			);

			foreach($pageAttributePageValues as $pageAttributePageValue){
				$pageAttributePageValue->delete();
			}
		}

		/**
		 * @return array|PageAttribute[]
		 */
		public static function getAvailableAttributesForPageType(PageType $pageType): array{
			// Fetch the IDs of PageAttribute instances belonging to that page type
			$abyss = new Abyss();
			$mysqli = $abyss->getConnectionToDatabase(\NoxEnv::MYSQL_DB_NAME);
			$result = $mysqli->query(sprintf("
				SELECT * 
				FROM `fbm_page_attributes` 
				WHERE `id` IN (
					SELECT `page_attribute_id` 
					FROM `fbm_page_attribute_page_type` 
					WHERE `page_type` = \"%s\"
				);
			", $pageType->name));
			$rows = $result->fetch_all(MYSQLI_ASSOC);
			$pageAttributeIDs = [];

			/** @var array{id: int, name: string, creation_time: int} $row */
			foreach($rows as $row){
				$pageAttributeIDs[] = $row['id'];
			}

			if (!empty($pageAttributeIDs)) {
				// Fetch the attributes
				/** @var PageAttribute[] $pageAttributes */
				$pageAttributes = PageAttribute::query(
					columnQuery: (new ColumnQuery())
						->where("id", "in", sprintf("(%s)", implode(",", $pageAttributeIDs)))
				);

				return $pageAttributes;
			}else{
				return [];
			}
		}

		/**
		 * @throws ObjectAlreadyExists
		 */
		public static function addAttributeToPage(int $attributeID, int $pageID): PageAttributePageValue{
			// Check if it's already added
			/** @var ?PageAttributePageValue $existingAttributeValue */
			$existingAttributeValue = PageAttributePageValue::queryOne(
				columnQuery: (new ColumnQuery())
					->where("page_attribute_id", "=", $attributeID)
					->and()
					->where("page_id", "=", $pageID)
			);

			if ($existingAttributeValue !== null){
				throw new ObjectAlreadyExists("This page already has that attribute assigned to it. Try refreshing?");
			}

			// Create the connection
			$newPageAttributeValue = new PageAttributePageValue();
			$newPageAttributeValue->pageAttributeID = $attributeID;
			$newPageAttributeValue->pageID = $pageID;
			$newPageAttributeValue->value = "";
			$newPageAttributeValue->save();

			return $newPageAttributeValue;
		}

		/**
		 * @throws NoPrimaryKey
		 * @throws NoObjectFound
		 */
		public static function deletePageAttributeValue(int $pageAttributeValueID): void{
			/** @var ?PageAttributePageValue $pageAttributeValue */
			$pageAttributeValue = PageAttributePageValue::fetch($pageAttributeValueID);

			if ($pageAttributeValue !== null){
				$pageAttributeValue->delete();
			}else{
				throw new NoObjectFound("There is no page attribute page value object with ID $pageAttributeValueID.");
			}
		}

	}