<?php
	namespace Page;

	use Nox\ORM\ColumnQuery;
	use Uplift\Exceptions\MalformedValue;

	class PageService{
		/**
		 * Searches page names
		 * @return Page[]
		 * @throws MalformedValue
		 */
		public static function queryPagesByPageName(string $query): array{
			$query = trim($query);

			if (empty($query)){
				throw new MalformedValue("Search query cannot be empty.");
			}

			$columnQuery = new ColumnQuery();
			$columnQuery->where("pageName", "LIKE", "%$query%");

			// Split the words up
			$words = explode(" ", $query);
			if (count($words) > 1){
				$columnQuery->or();
				$columnQuery->startConditionGroup();
				foreach($words as $index=>$word){
					if ($index > 0){
						$columnQuery->and();
					}

					$columnQuery->where("pageName", "LIKE", "%$word%");
				}

				$columnQuery->endConditionGroup();
			}

			/** @var Page[] $pages */
			$pages = Page::query($columnQuery);

			// A second query, for less relevant results, where the page could contain any of the words
			if (count($words) > 1){

				// Build an array of page IDs to ignore from the below query
				$pageIDsToIgnore = [];

				foreach($pages as $page){
					$pageIDsToIgnore[] = $page->id;
				}

				$lessRelevantColumnQuery = new ColumnQuery();

				if (!empty($pageIDsToIgnore)) {
					$lessRelevantColumnQuery->where(
						"`id`",
						"NOT IN",
						sprintf("(%s)", implode(",", $pageIDsToIgnore)),
						true
					);
					$lessRelevantColumnQuery->and();
					$lessRelevantColumnQuery->startConditionGroup();
				}

				foreach($words as $index=>$word){
					if ($index > 0){
						$lessRelevantColumnQuery->or();
					}

					$lessRelevantColumnQuery->where("pageName", "LIKE", "%$word%");
				}

				if (!empty($pageIDsToIgnore)) {
					$lessRelevantColumnQuery->endConditionGroup();
				}

				$pages = [...$pages, ...Page::query($lessRelevantColumnQuery)];
			}

			uasort($pages, function(Page $a, Page $b) use ($query){
				if (strtolower($a->pageName) === strtolower($query)){
					return -1;
				}

				return strtolower($a->pageName) > strtolower($b->pageName) ? 1 : -1;
			});

			return $pages;
		}

		/**
		 * Attempts to find the index page for projects by searching for a General page with the project reel shortcode.
		 * @return Page|null
		 */
		public static function getProjectsIndexPage(): ?Page{
			/** @var Page[] $generalPages */
			$generalPages = Page::query(
				columnQuery: (new ColumnQuery())
					->where("pageType", "=", PageType::General->name)
			);

			foreach($generalPages as $generalPage){
				// Do we need to check content sections or the pageBody?
				$hasContentSections = $generalPage->getLayoutDefaultOrFirstSection() !== null;
				if ($hasContentSections){
					foreach($generalPage->getAllContentSections() as $contentSection){
						if (str_contains($contentSection->content, "{{ get-ipps")){
							return $generalPage;
						} elseif (str_contains($contentSection->content, '{{ include file="project-reel')){
							return $generalPage;
						}
					}

				}else{
					$pageBodyContent = $generalPage->pageBody;
					if (str_contains($pageBodyContent, "{{ get-ipps")){
						return $generalPage;
					} elseif (str_contains($pageBodyContent, '{{ include file="project-reel')){
						return $generalPage;
					}
				}

			}

			return null;
		}
	}